# -*- coding: utf-8 -*-
from __future__ import division

import logging
import requests
import json

from requests.adapters import HTTPAdapter

from sandbox import common, sdk2
from sandbox.projects.findurl.resources import FindurlComparisonFile, FindurlResponse1, FindurlResponse2

GET_TASK_URL = 'https://findurl.z.yandex-team.ru/api/findurl.Findurl/getTask'
POST_TASK_URL = 'https://findurl.z.yandex-team.ru/api/findurl.Findurl/postTask'

session = requests.Session()
session.mount(GET_TASK_URL, HTTPAdapter(max_retries=10))
session.mount(POST_TASK_URL, HTTPAdapter(max_retries=10))


class CompareRequestsFindurl(sdk2.Task):
    """
    Makes 2 requests to findurl service, then calculates percentige difference between responses
    """
    headers = {}

    class Parameters(sdk2.Parameters):
        max_allowed_diff = sdk2.parameters.Integer(default=5, label='Max allowed difference')
        retry_time = sdk2.parameters.Integer(default=60*3, label='Query retry time')
        test_table_name = sdk2.parameters.String('Test input table name', default='//home/findurl/test_2k')

    class Context(sdk2.Task.Context):
        initialized = False
        task1_id = None
        task2_id = None
        task1_done = False
        task2_done = False

        difference_percent = 0.0

    is_task_broken = False

    def put_task(self):
        return (
            session.post(
                url=POST_TASK_URL,
                json={
                    'domain': 'hamster.yandex.ru',
                    'use_scraper': True,
                    'description': 'Test findurl diff launch, sandbox_task_id={}'.format(self.id),
                    'service_name': 'WEB',
                    'input_table': self.Parameters.test_table_name,
                    'disable_guilty_rearr_search': False,
                    'use_robot_canonization': False,
                },
                headers=self.headers,
            )
            .json()
            .get('id')
        )

    def get_task(self, task_id):
        return (
            session.post(
                url=GET_TASK_URL,
                json={
                    'id': task_id,
                },
                headers=self.headers,
            )
            .json()
        )

    @staticmethod
    def get_pk(qurl):
        return qurl['country'], qurl['region'], qurl['url'], qurl['query'], qurl['device']

    def calc_diff(self, task1, task2, file_path):
        diff_count = 0

        if task1['task']['status'] != task2['task']['status']:
            file_path.write_bytes('Task1 status {} != {}\n'.format(task1['status'], task2['status']))

        qurls_count = min(len(task1['qurls']), len(task2['qurls']))

        if not qurls_count:
            raise common.errors.TaskError("No data")

        qurls1, qurls2 = task1['qurls'], task2['qurls']

        # qurls don't have to be sorted

        qurl2_pk_to_obj = {
            self.get_pk(qurl2): qurl2
            for qurl2 in qurls2
        }

        for qurl1 in qurls1:
            pk = self.get_pk(qurl1)
            qurl2 = qurl2_pk_to_obj.get(pk)
            if qurl2:
                if qurl1['status'] != qurl2['status']:
                    file_path.write_bytes('Pk = {}, {} != {}\n'.format(pk, qurl1['status'], qurl2['status']))
                    diff_count += 1
            else:
                file_path.write_bytes('Can not find pk {} in second response\n'.format(pk))
                diff_count += 1

        difference = round(diff_count * 100 / qurls_count, 3)
        file_path.write_bytes('Found {}% difference\n'.format(difference))
        self.Context.difference_percent = difference

        self.is_task_broken = difference > self.Parameters.max_allowed_diff
        self.Context.save()

    def on_execute(self):
        if not self.Context.initialized:
            logging.info('initializing headers')
            try:
                self.headers = {
                    'authorization': 'oauth {}'.format(sdk2.Vault.data('findurl_token')),
                    'Content-Type': 'application/json',
                }
            except:
                raise common.errors.TaskError('findurl token is not set')

            logging.info('initializing tasks')
            self.Context.task1_id, self.Context.task2_id = self.put_task(), self.put_task()

            self.Context.initialized = True
            logging.info('task1_id = %s, task2_id = %s', self.Context.task1_id, self.Context.task2_id)
            self.Context.save()

            raise sdk2.WaitTime(self.Parameters.retry_time)

        # load resources
        response1 = sdk2.ResourceData(FindurlResponse1(
            self, 'response1 file', 'response1.txt'
        ))

        response2 = sdk2.ResourceData(FindurlResponse2(
            self, 'response2 file', 'response2.txt'
        ))

        while not (self.Context.task1_done and self.Context.task2_done):

            if not self.Context.task1_done:
                task1_res = self.get_task(self.Context.task1_id)
                try:
                    self.Context.task1_done = 'completed' in task1_res.get('task')
                except:
                    logging.error('can not get task1 status in %s', task1_res)
                    raise common.errors.TaskError('can not get task1 status / findurl did not response')

                if self.Context.task1_done:
                    response1.path.write_bytes(json.dumps(task1_res))
                    response1.ready()

            if not self.Context.task2_done:
                task2_res = self.get_task(self.Context.task2_id)
                try:
                    self.Context.task2_done = 'completed' in task2_res.get('task')
                except:
                    logging.error('can not get task2 status in %s', task2_res)
                    raise common.errors.TaskError('can not get task1 status / findurl did not response')

                if self.Context.task2_done:
                    response2.path.write_bytes(json.dumps(task2_res))
                    response2.ready()
            logging.info('task1_done = %s, task2_done = %s', self.Context.task1_done, self.Context.task2_done)

            self.Context.save()

            if not (self.Context.task1_done and self.Context.task2_done):
                raise sdk2.WaitTime(self.Parameters.retry_time)

        resource = sdk2.ResourceData(FindurlComparisonFile(
            self, 'Output file', 'compare_results.txt'
        ))

        logging.info('calculating diff')

        with open(str(response1.path)) as response1_file:
            with open(str(response2.path)) as response2_file:
                self.calc_diff(json.load(response1_file), json.load(response2_file), resource.path)

        resource.ready()

        if self.is_task_broken:
            raise common.errors.TaskError('Difference more than {}%'.format(self.Parameters.max_allowed_diff))
