# coding=utf-8
import os
import subprocess

from sandbox import sdk2
from sandbox.projects.common.arcadia import sdk as arcadia_sdk

PYTHON3_DEFAULT_EXECUTABLE = '/usr/bin/python3.8'


class FinsrvPythonPrecommitCheck(sdk2.Task):
    """Run precommit check on the arcadia project"""

    _mypy_output_file = None

    class Requirements(sdk2.Task.Requirements):
        """
        Используется кастомный LXC контейнер с установленным python3.8.

        Инструкция: https://wiki.yandex-team.ru/sandbox/cookbook/#use-custom-lxc-image
        Ещё одна инструкция: https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/sandbox/sandbox_lxc_image/__init__.py
        Таска: https://sandbox.yandex-team.ru/task/801557372/view

        История ресурсов:
        * 2020-10-20: https://sandbox.yandex-team.ru/resource/1786902223/view
        * 2021-09-28: https://sandbox.yandex-team.ru/resource/2459734777/view
        * 2022-06-10: https://sandbox.yandex-team.ru/resource/3199711288/view


        container_resource  - ID ресурса с контейнером. Если задолбаешься обновлять это число,
        можно релизить контейнер с кастомным типом ресурса, тогда самая новая версия будет подтягиваться
        автоматически.
        """
        container_resource = 1786902223

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 30*60
        owner = "YANDEX-PAY"
        description = "Run precommit check on the arcadia project"
        project_import_path = sdk2.parameters.String("Arcadia project import path e.g. mail.payments", required=True)
        project_path = sdk2.parameters.String("Arcadia project path e.g. /mail/payments", required=True)
        commit_hash = sdk2.parameters.String("Arc commit hash e.g. #9fd3c297477cce3985f0e22694966de27a7b451d without #", required=True)

        python3_executable_path = sdk2.parameters.String("Path to python3 executable")

        requirements = sdk2.parameters.List("List of required packages with hashes e.g. https://paste.yandex-team.ru/3047746", required=True)

        check_mypy = sdk2.parameters.Bool("Run mypy check")
        mypy_options = sdk2.parameters.List("List of mypy cmdline options")
        mypy_path = sdk2.parameters.List('Additional MYPYPATH folders ("/" is always present)')

        check_isort = sdk2.parameters.Bool("Run isort check")
        isort_options = sdk2.parameters.List("List of isort cmdline options")
        isort_check_path = sdk2.parameters.String("Isort check path, defaults to project_path")

        check_flake8 = sdk2.parameters.Bool("Run flake8 check")
        flake8_options = sdk2.parameters.List("List of flake8 cmdline options")
        flake8_check_path = sdk2.parameters.String("Flake8 check path, e.g. /mail/payments/payments, defaults to project_path")

        check_ya_style = sdk2.parameters.Bool("Run ya style check")
        ya_style_options = sdk2.parameters.List("List of ya style cmdline options")
        ya_style_check_path = sdk2.parameters.String("Ya style check path, e.g. /mail/payments/payments, defaults to project_path")

    @sdk2.header(title='Precommit report')
    def show_report(self):
        COLOR_GOOD = '#47C971'
        COLOR_BAD = '#FD0D1B'

        def badge(text, color):
            return '<span style="background-color: %s; color: #ffffff; font-weight: bold; padding: 0px 8px">%s</span>' % (color, text)

        mypy_report = ''
        flake8_report = ''
        isort_report = ''
        ya_style_report = ''
        if self.Parameters.check_flake8:
            if self.Context.flake8_output:
                flake8_report = badge('Flake8 FAILED', COLOR_BAD) + '\nErrors:\n'+ self.Context.flake8_output
            else:
                flake8_report = badge('Flake8 OK', COLOR_GOOD)
        if self.Parameters.check_isort:
            if self.Context.isort_output:
                isort_report = badge('Isort FAILED', COLOR_BAD) + '\nErrors:\n'+ self.Context.isort_output
            else:
                isort_report = badge('Isort OK', COLOR_GOOD)
        if self.Parameters.check_mypy:
            if self.Context.mypy_output:
                mypy_report = badge('Mypy FAILED', COLOR_BAD) + '\nErrors:\n'+ self.Context.mypy_output
            else:
                mypy_report = badge('Mypy OK', COLOR_GOOD)
        if self.Parameters.check_ya_style:
            if self.Context.ya_style_output:
                ya_style_report = badge('Ya style FAILED', COLOR_BAD) + '\nErrors:\n'+ self.Context.ya_style_output
            else:
                ya_style_report = badge('Ya style OK', COLOR_GOOD)
        return '\n\n'.join([mypy_report, flake8_report, isort_report, ya_style_report]).strip().replace('\n', '<br>')

    def make_relative(self, path):
        return path.lstrip('/')

    def get_ya_style_check_path(self):
        return self.make_relative(self.Parameters.ya_style_check_path or self.Parameters.project_path)

    def get_isort_check_path(self):
        return self.make_relative(self.Parameters.isort_check_path or self.Parameters.project_path)

    def get_flake8_check_path(self):
        return self.make_relative(self.Parameters.flake8_check_path or self.Parameters.project_path)

    @property
    def python3_executable_path(self):
        return self.Parameters.python3_executable_path or PYTHON3_DEFAULT_EXECUTABLE

    def install_requirements(self, workdir):
        """
        Устанавливаем все зависимости.
        В requirements должны быть указаны hashes, для секурити.
        """
        requirements_path = os.path.join(workdir, 'requirements.txt')
        with open(requirements_path, 'w') as requirements_file:
            requirements = '\n'.join([param for param in self.Parameters.requirements])
            requirements_file.write(requirements)

        with sdk2.helpers.ProcessLog(self, logger="pip-log") as pl:
            try:
                subprocess.check_call(
                    [
                        self.python3_executable_path,
                        '-m',
                        'pip',
                        'install',
                        '--index-url',
                        'https://pypi.yandex-team.ru/simple',
                        '--require-hashes',
                        '-r',
                        requirements_path,
                    ],
                    stdout=pl.stdout,
                    stderr=pl.stdout,
                    cwd=os.path.join(workdir, self.make_relative(self.Parameters.project_path))
                )
            finally:
                with open(str(pl.stdout.path), 'r') as logs:
                    self.Context.pip_stdout = logs.read()

    def run_mypy_cmd(self, arcadia_src_dir):
        with sdk2.helpers.ProcessLog(self, logger="mypy-log") as pl:
            try:
                env = os.environ.copy()
                env['MYPYPATH'] = ':'.join([arcadia_src_dir] + self.Parameters.mypy_path)
                env['ARCADIA_ROOT'] = arcadia_src_dir
                subprocess.check_call(
                    [
                        self.python3_executable_path,
                        '-m',
                        'mypy',
                        '-p',
                        self.Parameters.project_import_path,
                    ] + self.Parameters.mypy_options,
                    env=env,
                    stdout=pl.stdout,
                    stderr=pl.stdout,
                    cwd=os.path.join(arcadia_src_dir, self.make_relative(self.Parameters.project_path))
                )
            finally:
                with open(str(pl.stdout.path), 'r') as logs:
                    self.Context.mypy_output = logs.read()

    def run_flake8_cmd(self, arcadia_src_dir):
        with sdk2.helpers.ProcessLog(self, logger="flake8-log") as pl:
            try:
                subprocess.check_call(
                    [
                        self.python3_executable_path,
                        '-m',
                        'flake8',
                        os.path.join(arcadia_src_dir, self.get_flake8_check_path()),
                    ] + self.Parameters.flake8_options,
                    stdout=pl.stdout,
                    stderr=pl.stdout,
                    cwd=os.path.join(arcadia_src_dir, self.make_relative(self.Parameters.project_path))
                )
            finally:
                with open(str(pl.stdout.path), 'r') as logs:
                    self.Context.flake8_output = logs.read()

    def run_isort_cmd(self, arcadia_src_dir):
        with sdk2.helpers.ProcessLog(self, logger="isort-log") as pl:
            try:
                subprocess.check_call(
                    [
                        self.python3_executable_path,
                        '-m',
                        'isort',
                        '--check-only',
                        os.path.join(arcadia_src_dir, self.get_isort_check_path()),
                    ] + self.Parameters.isort_options,
                    stdout=pl.stdout,
                    stderr=pl.stdout,
                    cwd=os.path.join(arcadia_src_dir, self.make_relative(self.Parameters.project_path))
                )
            finally:
                with open(str(pl.stdout.path), 'r') as logs:
                    self.Context.isort_output = logs.read()

    def run_ya_style_cmd(self, arcadia_src_dir):
        with sdk2.helpers.ProcessLog(self, logger="ya-style-log") as pl:
            subprocess.check_call(
                [
                    os.path.join(arcadia_src_dir, 'ya'),
                    'style',
                    '--dry-run',
                    os.path.join(arcadia_src_dir, self.get_ya_style_check_path()),
                ] + self.Parameters.ya_style_options,
                stdout=pl.stdout,
                stderr=pl.stderr,
                cwd=os.path.join(arcadia_src_dir, self.make_relative(self.Parameters.project_path))
            )

            # В dry-run режиме ошибки летят в stdout
            with open(str(pl.stdout.path), 'r') as ya_style_errors:
                errors = ya_style_errors.read()
                self.Context.ya_style_output = errors
                if len(errors.strip()) > 0:
                    raise Exception('Ya style failed')

    def on_execute(self):
        with arcadia_sdk.mount_arc_path('arcadia-arc://#' + self.Parameters.commit_hash) as arcadia_src_dir:
            self.install_requirements(arcadia_src_dir)
            if self.Parameters.check_flake8:
                self.run_flake8_cmd(arcadia_src_dir)
            if self.Parameters.check_isort:
                self.run_isort_cmd(arcadia_src_dir)
            if self.Parameters.check_mypy:
                self.run_mypy_cmd(arcadia_src_dir)
            if self.Parameters.check_ya_style:
                self.run_ya_style_cmd(arcadia_src_dir)
