# -*- coding: utf-8 -*-
import logging
import os
import shutil
import sandbox.common.types.client as ctc

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.resource_types import OTHER_RESOURCE as OtherResource


class NodeModulesBackup(sdk2.Task):
    name = 'NODE_MODULES_BACKUP'

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        client_tags = sdk2.parameters.ClientTags(
            "Client tags. May only narrow the default set of client_tags that is specified in the RunCommand.Requirements",
            default=ctc.Tag.GENERIC
        )

        branch = sdk2.parameters.String(
            'Arcadia branch',
            description='Checkout branch',
            default='trunk',
            required=True,
        )
        project_path_in_arcadia = sdk2.parameters.String(
            'Path to project in Arcadia',
            description='Relative path',
            default='',
            required=True,
        )
        command = sdk2.parameters.String(
            'Command to execute',
            required=True
        )
        arc_token = sdk2.parameters.YavSecret(
            'Arc OAuth token',
            description='Use arc to create branch',
            default='',
            required=True
        )

    def _execute(self, command, shell=False):
        logging.debug('Try to call command: %s in path %s' % (str(command), os.getcwd()))
        result, err = sp.Popen(
            command, cwd=os.getcwd(), stdout=sp.PIPE, stderr=sp.PIPE, shell=shell
        ).communicate()

        logging.info('result')
        logging.info(result)
        logging.error('err')
        logging.error(err)

        return result

    def on_save(self):
        self.Requirements.client_tags = self.Parameters.client_tags

    def on_execute(self):
        branch = self.Parameters.branch

        logging.debug('Initializing arc branch %s' % branch)

        with arcadiasdk.mount_arc_path('arcadia-arc:/#trunk', arc_oauth_token=self.Parameters.arc_token.value()) as arcadia_path:
            arcadia_path = sdk2.Path(arcadia_path)
            logging.debug('Arcadia path: %s' % (str(arcadia_path)))
            project_path_in_arcadia = self.Parameters.project_path_in_arcadia
            work_dir = 'work_dir'
            shutil.copytree(str(arcadia_path / project_path_in_arcadia), work_dir)

            with sdk2.helpers.ProcessLog(self, logger="command") as pl:
                command = self.Parameters.command

                sdk2.helpers.subprocess.check_call(
                    command,
                    shell=True,
                    stdout=pl.stdout,
                    stderr=pl.stderr,
                    close_fds=True
                )

                resource_description = 'node_modules for %s' % (project_path_in_arcadia)
                resource_path = os.path.join(work_dir, 'node_modules.tar.gz')
                tags = str(self.Parameters.client_tags)
                resource = OtherResource(
                    self,
                    resource_description,
                    resource_path,
                    ttl=30,
                    project=project_path_in_arcadia,
                    tags=tags,
                )
                data = sdk2.ResourceData(resource)
                data.ready()
