# -*- coding: utf-8 -*-
import logging

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common.yasm import push_api as yasm

import sandbox.projects.sandbox.resources as sb_resources

from checksum import checksum
from s3helper import S3Helper

DEFAULT_S3_SECRET = "sec-01fmm77rvpkprmgftjgdgy3t6x"
DEFAULT_NPM_OAUTH_TOKEN_SECRET = "sec-01fnq17dxe9gvgfdcenb5v7gxv"
DEFAULT_S3_ENDPOINT = "https://s3.mds.yandex.net"
DEFAULT_S3_BUCKET = "nots-lockfiles"
DEFAULT_PREPARE_OPTIONS = "--platform linux --platform darwin --arch x64 --target node@14.17.0 --target node@12.13.0 " \
    + "--npm-username robot-notsprebuilder --npm-email robot-notsprebuilder@yandex-team.ru"


class NotsPrebuilder(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        description = "Remote runner for https://a.yandex-team.ru/arc_vcs/frontend/projects/infratest/packages/nots-prebuilder/"
        kill_timeout = 60 * 60  # in seconds
        container = sdk2.parameters.Container(
            "Environment container resource",
            default_value="2674786124",
            resource_type=sb_resources.LXC_CONTAINER,
            platform="linux_ubuntu_18.04_bionic",
            required=True,
        )
        binary_task_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group('prebuilder parameters') as s3_params:
            nots_prebuilder_version = sdk2.parameters.String(
                "@yandex-int/nots-prebuilder version",
                default="latest",
                required=True
                )
            nots_prebuilder_prepare_options = sdk2.parameters.String(
                "Options for @yandex-int/nots-prebuilder prepare",
                default=DEFAULT_PREPARE_OPTIONS,
                required=True
                )

        with sdk2.parameters.Group('npm parameters') as s3_params:
            npm_secret = sdk2.parameters.YavSecret(
                "Yav secret with npm OAuth token",
                default=DEFAULT_NPM_OAUTH_TOKEN_SECRET,
                required=True
                )

        with sdk2.parameters.Group('S3 parameters') as s3_params:
            s3_secret = sdk2.parameters.YavSecret(
                "Yav secret with S3 tokens",
                default=DEFAULT_S3_SECRET,
                required=True
                )
            s3_endpoint = sdk2.parameters.String(
                "S3 endpoint url",
                default=DEFAULT_S3_ENDPOINT,
                required=True
                )
            s3_bucket = sdk2.parameters.String(
                "S3 bucket name",
                default=DEFAULT_S3_BUCKET,
                required=True
                )
            original_lockfile_object_id = sdk2.parameters.String(
                "Original lockfile object id",
                required=True
                )
            original_lockfile_name = sdk2.parameters.String(
                "Original lockfile name",
                required=True
                )

        with sdk2.parameters.Output(reset_on_restart=True):
            s3_endpoint = sdk2.parameters.String("S3 endpoint url")
            s3_bucket = sdk2.parameters.String("S3 bucket name")
            processed_lockfile_name = sdk2.parameters.String("Processed lockfile name")
            processed_lockfile_object_id = sdk2.parameters.String("Processed lockfile object id")

    def _s3_client(self):
        secret = self.Parameters.s3_secret.data()
        s3 = S3Helper(
            access_key_id=secret["AccessKeyId"],
            secret_access_key=secret["AccessSecretKey"],
            bucket=self.Parameters.s3_bucket,
            endpoint=self.Parameters.s3_endpoint
        )

        return s3

    def _npm_login(self):
        logging.info("Login to npm.yandex-team.ru as nots-prebuilder@…")
        self._exec([
            "npm", "config", "set",
            "registry", "https://npm.yandex-team.ru",
            ],
            log="npm-config-set-registry"
        )
        self._exec([
            "npm", "config", "set",
            "email", "robot-notsprebuilder@yandex-team.ru",
            ],
            log="npm-config-set-email"
        )
        secret = self.Parameters.npm_secret.data()
        self._exec([
            "npm", "config", "set",
            "//npm.yandex-team.ru/:_authToken", secret["npm-oauth-token"],
            ],
            log="npm-config-set-oauthtoken"
        )

    def _exec(self, *args, **kwargs):
        log_name = kwargs.pop("log", None)

        with sdk2.helpers.ProcessLog(self, logger=log_name) as pl:
            # https://docs.python.org/3/library/subprocess.html#subprocess.check_call
            sdk2.helpers.subprocess.check_call(
                *args,
                stdout=pl.stdout,
                stderr=pl.stderr,
                **kwargs
            )

    def _download_lockfile(self, client):
        object_id = self.Parameters.original_lockfile_object_id
        lockfile_name = self.Parameters.original_lockfile_name
        lockfile_path = './{}'.format(lockfile_name)

        logging.info("Downloading original lockfile…")
        client.download_file(object_id, lockfile_path)

        return lockfile_path

    def _install_deps(self):
        logging.info("Installing dependencies…")
        self._exec(
            [
                "pnpm",
                "install",
                "--ignore-scripts",
                # TODO: подумать про debug режим
                # "--loglevel", "silly",
            ],
            log="pnpm-install",
        )

    def _run_prepare(self):
        logging.info("Running nots-prebuilder prepare…")
        version = self.Parameters.nots_prebuilder_version
        opts = self.Parameters.nots_prebuilder_prepare_options.split()
        self._exec(
            [
                "npx",
                "@yandex-int/nots-prebuilder@{}".format(version),
                "prepare",
            ] + opts,
            log="nots-prebuilder-prepare",
        )

    def _run_apply(self):
        logging.info("Running nots-prebuilder apply-lockfile…")
        version = self.Parameters.nots_prebuilder_version
        self._exec(
            [
                "npx",
                "@yandex-int/nots-prebuilder@{}".format(version),
                "apply-lockfile",
            ],
            log="nots-prebuilder-apply-lockfile",
        )

    def _upload_lockfile(self, client, lockfile_path):
        logging.info("Uploading processed lockfile…")
        processed_object_id = checksum(lockfile_path)
        client.upload_file(lockfile_path, processed_object_id)

        return processed_object_id

    def on_execute(self):
        client = self._s3_client()
        lockfile_path = self._download_lockfile(client)
        self._npm_login()
        self._install_deps()
        self._run_prepare()
        self._run_apply()
        processed_object_id = self._upload_lockfile(client, lockfile_path)

        # outputs:
        self.Parameters.s3_endpoint = self.Parameters.s3_endpoint
        self.Parameters.s3_bucket = self.Parameters.s3_bucket
        self.Parameters.lockfile_name = self.Parameters.original_lockfile_name
        self.Parameters.original_lockfile_object_id = self.Parameters.original_lockfile_object_id
        self.Parameters.processed_lockfile_object_id = processed_object_id

    def on_timeout(self, prev_status, status):
        super(NotsPrebuilder, self).on_break(prev_status, status)
        self._push_signals_to_yasm(status)

    def on_break(self, prev_status, status):
        super(NotsPrebuilder, self).on_break(prev_status, status)
        self._push_signals_to_yasm(status)

    def on_finish(self, prev_status, status):
        super(NotsPrebuilder, self).on_break(prev_status, status)
        self._push_signals_to_yasm(status)

    def _push_signals_to_yasm(self, status):
        if getattr(self.Context, 'copy_of', False):
            return

        try:
            yasm.push_signals(
                signals={
                    'status_{}_mmmm'.format(status.lower()): 1,
                },
                tags={
                    'itype': 'nots_prebuilder_remote',
                    # TODO: какой проект?
                    'prj': '',
                    'ctype': 'prod',
                },
            )
        except Exception as e:
            logging.exception('Exception while pushing signals to yasm: {}'.format(e))
