# -*- coding: utf-8 -*-
import logging


class S3Helper(object):
    def __init__(self, access_key_id, secret_access_key, bucket, endpoint='https://s3.mds.yandex.net'):
        import boto3
        from botocore.config import Config

        logging.getLogger('botocore').setLevel(logging.INFO)
        logging.getLogger('s3transfer').setLevel(logging.INFO)

        self.bucket = bucket
        session = boto3.session.Session(
            aws_access_key_id=access_key_id,
            aws_secret_access_key=secret_access_key,
        )
        self.client = session.client(
            service_name='s3',
            endpoint_url=endpoint,
            config=Config(
                parameter_validation=False,
            )
        )
        logging.info('S3 connected to endpoint {}'.format(endpoint))

    def has_object(self, object_id):
        from botocore.errorfactory import ClientError

        ret = False
        try:
            res = self.client.head_object(Bucket=self.bucket, Key=object_id)
            ret = res['ResponseMetadata']['HTTPStatusCode'] == 200
        except ClientError:
            pass

        return ret

    def download_file(self, object_id, file_name):
        logging.info('Downloading object: {} from bucket: {} to file: {}'.format(
            object_id,
            self.bucket,
            file_name,
        ))
        return self.client.download_file(self.bucket, object_id, file_name)

    def upload_file(self, path, object_id):
        # TODO: проверить что content-type правильно выставляется при загрузке
        logging.info('Uploading path: {}, bucket: {}, object id: {}'.format(
            path,
            self.bucket,
            object_id,
        ))

        self.client.upload_file(
            path,
            self.bucket,
            object_id,
        )
