# coding: utf-8

import hashlib
import logging
import os
import random
import itertools

import sandbox.common.types.resource as ctr
from sandbox import sdk2
from sandbox.common import fs

import sandbox.projects.fuzzing.resources as fuzz_res
from sandbox.projects import resource_types
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import time_utils as tu
from sandbox.projects.common.search import bugbanner2
from sandbox.projects.fuzzing import fuzzers
from sandbox.projects.websearch.middlesearch import resources as msres
from sandbox.projects.websearch.upper import resources as upper_resources

IN_RES = {
    "base": resource_types.PLAIN_TEXT_QUERIES,
    "video_base": resource_types.PLAIN_TEXT_QUERIES,
    "middle": msres.WebMiddlesearchApphostRequests,
    "blender": upper_resources.BlenderGrpcClientPlan,
}
OUT_RES = {
    "base": fuzz_res.PlainTextQueriesFuzzy,
    "video_base": fuzz_res.VideoBasesearchQueriesFuzzy,
    "middle": fuzz_res.WebMiddlesearchQueriesFuzzy,
    "blender": fuzz_res.WebBlenderQueriesFuzzy,
}


class PrepareFuzzyQueries(bugbanner2.BugBannerTask):
    """
        SEARCH-2680
    """
    class Requirements(sdk2.Task.Requirements):
        ram = 2 * 1024
        disk_space = 20 * 1024  # 20 Gb
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.String("Queries for", required=True) as queries_for:
            queries_for.values.base = queries_for.Value("base")
            queries_for.values.video_base = queries_for.Value("video_base")
            queries_for.values.middle = queries_for.Value("middle")
            queries_for.values.blender = queries_for.Value("blender")
        prepare_random_prod_queries = sdk2.parameters.Bool("Prepare random prod queries")
        query_limit = sdk2.parameters.Integer("Max queries to prepare (0 - prepare all)", default=0)
        with prepare_random_prod_queries.value[False]:
            queries_res = sdk2.parameters.Resource(
                "Initial queries",
                resource_type=list(set(IN_RES.values())),
                required=True,
            )

    class Context(sdk2.Context):
        out_resource_id = 0

    # @property
    # def footer(self):
    #     pass

    def on_enqueue(self):
        sdk2.Task.on_enqueue(self)
        queries_for = self.Parameters.queries_for
        res = OUT_RES[queries_for](
            self, "Queries for {} fuzzer".format(queries_for),
            "additional_{}_fuzzer_queries.tar.gz".format(queries_for)
        )
        self.Context.out_resource_id = res.id

    def on_execute(self):
        self.add_bugbanner(bugbanner2.Banners.WebBaseSearch)
        queries_for = self.Parameters.queries_for
        logging.info("Prepare queries for {} fuzzer".format(queries_for))
        fuzz_queries_folder = "additional_{}_fuzzer_queries_{}".format(queries_for, tu.date_ymd(sep="_"))
        fs.make_folder(fuzz_queries_folder)
        with self.get_path_to_queries(queries_for) as p:
            with p.open() as f:
                queries = itertools.islice(f, self.Parameters.query_limit) if self.Parameters.query_limit else f
                for query in queries:
                    q = query.strip()
                    filename = hashlib.md5(q).hexdigest()  # https://st.yandex-team.ru/SEARCH-2680#1478173063000
                    fu.write_file(os.path.join(fuzz_queries_folder, filename), q)

        fuzzers.pack_queries(
            self,
            fuzz_queries_folder,
            str(sdk2.ResourceData(sdk2.Resource[self.Context.out_resource_id]).path)
        )

    def get_path_to_queries(self, queries_for):
        """
            :return: object of pathlib2.Path
        """
        res_type = IN_RES[queries_for]
        if self.Parameters.prepare_random_prod_queries:
            q_type = random.choice(["search", "factors", "snippets"])
            q_tier = "PlatinumTier0"
            attr_name = "TE_web_base_prod_queries_{q_type}_{q_tier}".format(q_type=q_type, q_tier=q_tier)
            res = list(sdk2.Resource.find(
                resource_type=res_type,
                state=ctr.State.READY,
                attr_name=attr_name
            ).limit(1))[0]
            logging.info("Random prod queries selected: %s", res.id)
        else:
            res = self.Parameters.queries_res
        return sdk2.ResourceData(res).path
