# coding: utf-8

import logging
import os
from hashlib import md5

import sandbox.common.types.resource as ctr
import sandbox.projects.fuzzing.resources
from sandbox import sdk2
from sandbox.projects.websearch.begemot.resources import WizardQueriesJson
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import time_utils as tu
from sandbox.projects.fuzzing import fuzzers
from sandbox.common import fs


class PrepareWizardFuzzyQueries(sdk2.Task):
    """
        SEARCH-3143
    """
    class Requirements(sdk2.Task.Requirements):
        ram = 2 * 1024
        disk_space = 10 * 1024  # 10 Gb
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        prepare_last_prod_queries = sdk2.parameters.Bool("Prepare last prod queries")
        with prepare_last_prod_queries.value[False]:
            queries_res = sdk2.parameters.Resource(
                "Queries for wizard",
                resource_type=WizardQueriesJson,
                required=True,
            )

    class Context(sdk2.Context):
        out_resource_id = 0

    def on_enqueue(self):
        sdk2.Task.on_enqueue(self)
        res = sandbox.projects.fuzzing.resources.WizardQueriesFuzzy(self, "Queries for fuzzer", "additional_fuzzer_queries.tar.gz")
        self.Context.out_resource_id = res.id

    def on_execute(self):
        logging.info("Prepare queries for fuzzer")
        fuzz_queries_folder = "additional_fuzzer_queries_{}".format(tu.date_ymd(sep="_"))
        fs.make_folder(fuzz_queries_folder)
        with self.get_path_to_queries() as p:
            for query in p.read_bytes().split("\n--------\n"):
                filename = md5(query).hexdigest()
                fu.write_file(os.path.join(fuzz_queries_folder, filename), query)

        fuzzers.pack_queries(
            self,
            fuzz_queries_folder,
            str(sdk2.ResourceData(sdk2.Resource[self.Context.out_resource_id]).path)
        )

    def get_path_to_queries(self):
        """
            :return: object of pathlib2.Path
        """
        if self.Parameters.prepare_last_prod_queries:
            res = list(sdk2.Resource.find(
                resource_type=WizardQueriesJson,
                state=ctr.State.READY,
                attr_name="TE_web_production_wizard_reqs"
            ).limit(1))[0]
            logging.info("Prod queries selected: %s", res.id)
        else:
            res = self.Parameters.queries_res
        return sdk2.ResourceData(res).path
