# coding: utf-8

import logging
import os

import sandbox.common.types.resource as ctr
import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt
import sandbox.projects.common.task_env as task_env
import sandbox.projects.fuzzing.resources as fuzz_res
import sandbox.projects.release_machine.core.task_env as rm_task_env

from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.fuzzing import fuzzers
from sandbox.projects.websearch.basesearch.GetWebBasesearchResponses import GetWebBasesearchResponses


class RunBasesearchFuzzy(sdk2.Task):
    """
        SEARCH-2681
    """
    class Requirements(task_env.TinyRequirements):
        ram = 40 * 1024
        disk_space = 40 * 1024  # 40 Gb
        client_tags = (ctc.Tag.MULTISLOT | ctc.Tag.GENERIC) & rm_task_env.TaskTags.startrek_client
        environments = [rm_task_env.TaskRequirements.startrek_client]

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 23 * 60 * 60  # 23 hours
        count_coverage = sdk2.parameters.Bool("Count coverage")
        basesearch_executable_no_fuzzy = sdk2.parameters.Resource(
            "Basesearch binary",
            resource_type=resource_types.BASESEARCH_EXECUTABLE,
            required=False,
        )
        basesearch_executable_resource = sdk2.parameters.Resource(
            "Basesearch fuzzy binary",
            resource_type=resource_types.BASESEARCH_EXECUTABLE_FUZZY,
            required=True,
        )
        basesearch_config_resource = sdk2.parameters.Resource(
            "Basesearch fuzzy config",
            resource_type=resource_types.SEARCH_CONFIG,
            required=True,
        )
        basesearch_models_archive_resource = sdk2.parameters.Resource(
            "Basesearch models archive",
            resource_type=resource_types.DYNAMIC_MODELS_ARCHIVE_BASE,
            required=True,
        )
        basesearch_database_resource = sdk2.parameters.Resource(
            "Basesearch database",
            resource_type=resource_types.SEARCH_DATABASE,
            required=True,
        )
        runs = sdk2.parameters.Integer("Runs number (0 = no fuzzing)", default=-1)
        max_total_time = sdk2.parameters.Integer("Fuzzy time (sec)", default=72000)
        last_fuzzed_queries = sdk2.parameters.Bool("Use last fuzzed queries")
        with last_fuzzed_queries.value[False]:
            basesearch_queries_resource = sdk2.parameters.Resource(
                "Queries for basesearch",
                resource_type=fuzz_res.PlainTextQueriesFuzzy,
                required=True,
            )
        # two different additional queries for factors and snippets
        add_queries_1 = sdk2.parameters.Resource(
            "Additional queries 1 for basesearch",
            resource_type=fuzz_res.PlainTextQueriesFuzzy,
            required=False,
        )
        add_queries_2 = sdk2.parameters.Resource(
            "Additional queries 2 for basesearch",
            resource_type=fuzz_res.PlainTextQueriesFuzzy,
            required=False,
        )

    def on_execute(self):
        if self.Context.shoot_task_id:
            return
        if self.Parameters.last_fuzzed_queries:
            queries = list(sdk2.Resource.find(
                resource_type=fuzz_res.PlainTextQueriesFuzzy,
                state=ctr.State.READY,
                attr_name="last_fuzzed_queries"
            ).limit(1))[0]
            logging.info("Last fuzzed queries resource id: %s", queries.id)
        else:
            queries = self.Parameters.basesearch_queries_resource

        basesearch_fuzzer = fuzzers.WebBasesearchFuzzer(
            self.Parameters.basesearch_executable_resource,
            self.Parameters.basesearch_config_resource,
            self.Parameters.basesearch_models_archive_resource,
            self.Parameters.basesearch_database_resource,
            queries,
            add_queries=(self.Parameters.add_queries_1, self.Parameters.add_queries_2),
            cmd_args={
                "rss_limit_mb": 40960,
                "max_total_time": self.Parameters.max_total_time,
                "runs": self.Parameters.runs,
            },
            count_coverage=self.Parameters.count_coverage,
            task=self,
        )

        with basesearch_fuzzer:
            basesearch_fuzzer.use_component()
        if self.Parameters.count_coverage:
            missing_info = basesearch_fuzzer.output_results[fuzzers.CoverageCounting.NAME]["missing_info"]
            self.Context.coverage_info = missing_info
            self.Context.coverage_per_cent = 100.0 * missing_info["pcs_covered"] / (missing_info["pcs_instrumented"] or 1)
            self.set_info("Coverage: {:.3f} %".format(self.Context.coverage_per_cent))
            if missing_info["pcs_not_found_exist"]:
                self.set_info("WARNING: probably coverage info is not correct for this binary!")
        else:
            problem_queries = list(self.get_problem_queries())
            if self.Parameters.basesearch_executable_no_fuzzy:
                if problem_queries:
                    queries_res = self.save_bad_queries_as_plain_text(problem_queries)
                    self.check_on_real_basesearch(queries_res)
                else:
                    self.set_info("No problems found, no queries to ask basesearch")

    def get_problem_queries(self):
        for stage_info in self.Context.output_results.itervalues():
            for problem_queries, problem_path in zip(
                stage_info["problems"].values(), stage_info["problems_paths"].values()
            ):
                for p in problem_queries:
                    logging.info("Save %s query to plain text resource", p)
                    yield os.path.join(problem_path, p)

    def save_bad_queries_as_plain_text(self, problem_queries):
        ptq = resource_types.PLAIN_TEXT_QUERIES(self, "Queries with fuzzy problems", "fuzzy_problem_queries.txt")
        with open(str(ptq.path)) as out_f:
            for problem_query in problem_queries:
                with open(problem_query) as in_f:
                    out_f.write(in_f.read().strip() + "\n")
        sdk2.ResourceData(ptq).ready()
        return ptq

    def check_on_real_basesearch(self, queries_res):
        """SEARCH-3723"""
        shoot_task_id = GetWebBasesearchResponses(
            self,
            executable=self.Parameters.basesearch_executable_no_fuzzy,
            config=self.Parameters.basesearch_config_resource,
            models_archive=self.Parameters.basesearch_models_archive_resource,
            database=self.Parameters.basesearch_database_resource,
            requests=queries_res.id,
            n_workers=1,
            stabilize=False,
            need_dbgrlv=False,
        )
        self.Context.shoot_task_id = shoot_task_id
        raise sdk2.WaitTask(shoot_task_id, ctt.Status.Group.FINISH)
