# coding: utf-8

import logging

import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.common.types.resource as ctr
import sandbox.projects.fuzzing.resources as fuzz_res
import sandbox.projects.websearch.upper.resources as upper_resources
from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.fuzzing import fuzzers


class RunBlenderFuzzy(sdk2.Task):
    """
        SEARCH-4751
    """
    class Requirements(sdk2.Task.Requirements):
        cores = 6
        ram = 40 * 1024
        disk_space = 50 * 1024  # 50 Gb
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 18 * 60 * 60  # 18 hours
        count_coverage = sdk2.parameters.Bool("Count coverage")
        blender_executable_resource = sdk2.parameters.Resource(
            "Blender fuzzy binary",
            resource_type=upper_resources.WebBlenderExecutableFuzzy,
            required=True,
        )
        blender_config_resource = sdk2.parameters.Resource(
            "Blender fuzzy config",
            resource_type=resource_types.NOAPACHEUPPER_CONFIG,
            required=True,
        )
        blender_data_resource = sdk2.parameters.Resource(
            "Blender data",
            resource_type=resource_types.NOAPACHEUPPER_DATA,
            required=True,
        )
        runs = sdk2.parameters.Integer("Runs number (0 = no fuzzing)", default=-1)
        max_total_time = sdk2.parameters.Integer("Fuzzy time (sec)", default=9000)
        last_fuzzed_queries = sdk2.parameters.Bool("Use last fuzzed queries")

        with last_fuzzed_queries.value[False]:
            blender_queries_resource = sdk2.parameters.Resource(
                "Queries for blender",
                resource_type=fuzz_res.WebBlenderQueriesFuzzy,
                required=True,
            )

        add_queries = sdk2.parameters.Resource(
            "Additional queries for blender",
            resource_type=fuzz_res.WebBlenderQueriesFuzzy,
            required=False,
        )

    def on_execute(self):
        if self.Parameters.last_fuzzed_queries:
            queries = list(sdk2.Resource.find(
                resource_type=fuzz_res.WebBlenderQueriesFuzzy,
                state=ctr.State.READY,
                attr_name="last_fuzzed_queries"
            ).limit(1))[0]
            logging.info("Last fuzzed queries resource id: %s", queries.id)
        else:
            queries = self.Parameters.blender_queries_resource

        blender_fuzzer = fuzzers.BlenderFuzzer(
            self.Parameters.blender_executable_resource,
            self.Parameters.blender_config_resource,
            self.Parameters.blender_data_resource,
            queries,
            add_queries=(self.Parameters.add_queries,),
            cmd_args={
                "rss_limit_mb": 25 * 1024,
                "max_total_time": self.Parameters.max_total_time,
                "runs": self.Parameters.runs,
            },
            count_coverage=self.Parameters.count_coverage,
            task=self,
        )
        with blender_fuzzer:
            blender_fuzzer.use_component()
        if self.Parameters.count_coverage:
            missing_info = blender_fuzzer.output_results[fuzzers.CoverageCounting.NAME]["missing_info"]
            self.Context.coverage_info = missing_info
            self.Context.coverage_per_cent = 100.0 * missing_info["pcs_covered"] / (missing_info["pcs_instrumented"] or 1)
            self.set_info("Coverage: {:.3f} %".format(self.Context.coverage_per_cent))
            if missing_info["pcs_not_found_exist"]:
                self.set_info("WARNING: probably coverage info is not correct for this binary!")
