# coding: utf-8

import logging

import sandbox.common.types.resource as ctr
import sandbox.projects.fuzzing.resources as fuzz_res
import sandbox.projects.release_machine.core.task_env as task_env
from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.fuzzing import fuzzers
from sandbox.projects.websearch.basesearch import components as bs_components
from sandbox.projects.websearch.basesearch import params as bs_params
from sandbox.projects.websearch.middlesearch import resources as wmres


class RunMiddlesearchFuzzy(sdk2.Task):
    """
        SEARCH-3714
    """
    class Requirements(sdk2.Task.Requirements):
        ram = 80 * 1024
        disk_space = 80 * 1024  # 80 Gb
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 16 * 60 * 60  # 16 hours
        count_coverage = sdk2.parameters.Bool("Count coverage")
        middlesearch_executable_resource = sdk2.parameters.Resource(
            "Middlesearch fuzzy binary",
            resource_type=wmres.WebMiddlesearchExecutableFuzzy,
            required=True,
        )
        middlesearch_config_resource = sdk2.parameters.Resource(
            "Middlesearch fuzzy config",
            resource_type=resource_types.MIDDLESEARCH_CONFIG,
            required=True,
        )
        middlesearch_data_resource = sdk2.parameters.Resource(
            "Middlesearch data",
            resource_type=resource_types.MIDDLESEARCH_DATA,
            required=True,
        )
        middlesearch_models_resource = sdk2.parameters.Resource(
            "Middlesearch models",
            resource_type=resource_types.DYNAMIC_MODELS_ARCHIVE,
            required=True,
        )
        basesearch = bs_params.SingleBasesearchParameters
        runs = sdk2.parameters.Integer("Runs number (0 = no fuzzing)", default=-1)
        max_total_time = sdk2.parameters.Integer("Fuzzy time (sec)", default=9000)
        last_fuzzed_queries = sdk2.parameters.Bool("Use last fuzzed queries")

        with last_fuzzed_queries.value[False]:
            middlesearch_queries_resource = sdk2.parameters.Resource(
                "Queries for middlesearch",
                resource_type=fuzz_res.WebMiddlesearchQueriesFuzzy,
                required=True,
            )

        add_queries = sdk2.parameters.Resource(
            "Additional queries for middlesearch",
            resource_type=fuzz_res.WebMiddlesearchQueriesFuzzy,
            required=False,
        )
        with sdk2.parameters.Output:
            coverage = sdk2.parameters.Float("Coverage percent (0 if not counted)")

    def on_execute(self):
        if self.Parameters.last_fuzzed_queries:
            queries = list(sdk2.Resource.find(
                resource_type=fuzz_res.WebMiddlesearchQueriesFuzzy,
                state=ctr.State.READY,
                attr_name="last_fuzzed_queries"
            ).limit(1))[0]
            logging.info("Last fuzzed queries resource id: %s", queries.id)
        else:
            queries = self.Parameters.middlesearch_queries_resource

        wbs = bs_components.WebBasesearch(
            self,
            binary=self.Parameters.basesearch.executable,
            config=self.Parameters.basesearch.config,
            models=self.Parameters.basesearch.models_archive,
            database=self.Parameters.basesearch.database,
            polite_mode=self.Parameters.basesearch.polite_mode,
            # patch_request_threads=self.Parameters.basesearch.patch_request_threads,
        )

        middlesearch_fuzzer = fuzzers.MiddlesearchFuzzer(
            self.Parameters.middlesearch_executable_resource,
            self.Parameters.middlesearch_config_resource,
            self.Parameters.middlesearch_data_resource,
            self.Parameters.middlesearch_models_resource,
            queries,
            basesearches=[{
                'basesearch_type': "PLATINUM",
                'searchers': [('localhost', wbs.port)] * 3,
            }],
            add_queries=(self.Parameters.add_queries,),
            cmd_args={
                "rss_limit_mb": 65535,
                "max_total_time": self.Parameters.max_total_time,
                "runs": self.Parameters.runs,
            },
            count_coverage=self.Parameters.count_coverage,
            task=self,
        )
        with middlesearch_fuzzer:
            middlesearch_fuzzer.use_component()
        if self.Parameters.count_coverage:
            missing_info = middlesearch_fuzzer.output_results[fuzzers.CoverageCounting.NAME]["missing_info"]
            missing_info["coverage_per_cent"] = 100.0 * missing_info["pcs_covered"] / (missing_info["pcs_instrumented"] or 1)
            self.Context.coverage_info = missing_info
            self.set_info("Coverage: {:.3f} %".format(missing_info["coverage_per_cent"]))
            self.Parameters.coverage = round(missing_info["coverage_per_cent"], 3)
            if missing_info["pcs_not_found_exist"]:
                self.set_info("WARNING: probably coverage info is not correct for this binary!")
