# coding: utf-8

import logging

import sandbox.common.types.resource as ctr
from sandbox import sdk2

import sandbox.projects.fuzzing.resources as fuzz_res
import sandbox.projects.VideoSearch.video_resource_types as vid_base_resources
from sandbox.projects import resource_types
from sandbox.projects.fuzzing import fuzzers


class RunVideoBasesearchFuzzy(sdk2.Task):
    """
        SEARCH-5194
    """
    class Requirements(sdk2.Task.Requirements):
        ram = 40 * 1024
        disk_space = 40 * 1024  # 40 Gb

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 22 * 60 * 60  # 22 hours
        count_coverage = sdk2.parameters.Bool("Count coverage")
        video_base_executable_resource = sdk2.parameters.Resource(
            "Video basesearch fuzzy binary",
            resource_type=vid_base_resources.VIDEO_BASESEARCH_EXECUTABLE_FUZZY,
            required=True,
        )
        video_base_config_resource = sdk2.parameters.Resource(
            "Video basesearch fuzzy config",
            resource_type=resource_types.VIDEO_SEARCH_CONFIG,
            required=True,
        )
        video_base_models_archive_resource = sdk2.parameters.Resource(
            "Video basesearch models archive",
            resource_type=resource_types.VIDEO_DYNAMIC_MODELS_ARCHIVE,
            required=True,
        )
        video_base_database_resource = sdk2.parameters.Resource(
            "Video basesearch database",
            resource_type=resource_types.VIDEO_SEARCH_DATABASE,
            required=True,
        )
        runs = sdk2.parameters.Integer("Runs number (0 = no fuzzing)", default=-1)
        max_total_time = sdk2.parameters.Integer("Fuzzy time (sec)", default=72000)
        last_fuzzed_queries = sdk2.parameters.Bool("Use last fuzzed queries")
        with last_fuzzed_queries.value[False]:
            video_base_queries_resource = sdk2.parameters.Resource(
                "Queries for video basesearch",
                resource_type=fuzz_res.VideoBasesearchQueriesFuzzy,
                required=True,
            )
        # two different additional queries for factors and snippets
        add_queries_1 = sdk2.parameters.Resource(
            "Additional queries 1 for video basesearch",
            resource_type=fuzz_res.VideoBasesearchQueriesFuzzy,
            required=False,
        )
        add_queries_2 = sdk2.parameters.Resource(
            "Additional queries 2 for video basesearch",
            resource_type=fuzz_res.VideoBasesearchQueriesFuzzy,
            required=False,
        )

    def on_execute(self):
        if self.Parameters.last_fuzzed_queries:
            queries = list(sdk2.Resource.find(
                resource_type=fuzz_res.VideoBasesearchQueriesFuzzy,
                state=ctr.State.READY,
                attr_name="last_fuzzed_queries"
            ).limit(1))[0]
            logging.info("Last fuzzed queries resource id: %s", queries.id)
        else:
            queries = self.Parameters.video_base_queries_resource

        basesearch_fuzzer = fuzzers.VideoBasesearchFuzzer(
            self.Parameters.video_base_executable_resource,
            self.Parameters.video_base_config_resource,
            self.Parameters.video_base_models_archive_resource,
            self.Parameters.video_base_database_resource,
            queries,
            add_queries=(self.Parameters.add_queries_1, self.Parameters.add_queries_2),
            cmd_args={
                "rss_limit_mb": 40960,
                "max_total_time": self.Parameters.max_total_time,
                "runs": self.Parameters.runs,
            },
            count_coverage=self.Parameters.count_coverage,
            task=self,
        )

        with basesearch_fuzzer:
            basesearch_fuzzer.use_component()
        if self.Parameters.count_coverage:
            missing_info = basesearch_fuzzer.output_results[fuzzers.CoverageCounting.NAME]["missing_info"]
            self.Context.coverage_info = missing_info
            self.Context.coverage_per_cent = 100.0 * missing_info["pcs_covered"] / (missing_info["pcs_instrumented"] or 1)
            self.set_info("Coverage: {:.3f} %".format(self.Context.coverage_per_cent))
            if missing_info["pcs_not_found_exist"]:
                self.set_info("WARNING: probably coverage info is not correct for this binary!")
