# coding: utf-8

import logging

import sandbox.common.types.resource as ctr
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.fuzzing.resources

from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.fuzzing import fuzzers


class RunWizardFuzzy(sdk2.Task):
    """
        SEARCH-3038
    """
    class Requirements(sdk2.Task.Requirements):
        ram = 40 * 1024
        disk_space = 70 * 1024  # 70 Gb
        client_tags = task_env.TaskTags.startrek_client
        environments = [task_env.TaskRequirements.startrek_client]

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10 * 60 * 60  # 10 hours
        count_coverage = sdk2.parameters.Bool("Count coverage")
        wizard_executable_resource = sdk2.parameters.Resource(
            "Wizard fuzzy binary",
            resource_type=resource_types.WIZARD_EXECUTABLE_FUZZY,
            required=True,
        )
        wizard_config_resource = sdk2.parameters.Resource(
            "Wizard fuzzy config",
            resource_type=resource_types.WIZARD_CONFIG,
            required=True,
        )
        wizard_data_resource = sdk2.parameters.Resource(
            "Wizard data",
            resource_type=resource_types.WIZARD_SHARD,
            required=True,
        )
        runs = sdk2.parameters.Integer("Runs number (0 = no fuzzing)", default=-1)
        max_total_time = sdk2.parameters.Integer("Fuzzy time (sec)", default=9000)
        last_fuzzed_queries = sdk2.parameters.Bool("Use last fuzzed queries")

        with last_fuzzed_queries.value[False]:
            wizard_queries_resource = sdk2.parameters.Resource(
                "Queries for wizard",
                resource_type=sandbox.projects.fuzzing.resources.WizardQueriesFuzzy,
                required=True,
            )

        add_queries = sdk2.parameters.Resource(
            "Additional queries for wizard",
            resource_type=sandbox.projects.fuzzing.resources.WizardQueriesFuzzy,
            required=False,
        )

    def on_execute(self):
        if self.Parameters.last_fuzzed_queries:
            queries = list(sdk2.Resource.find(
                resource_type=sandbox.projects.fuzzing.resources.WizardQueriesFuzzy,
                state=ctr.State.READY,
                attr_name="last_fuzzed_queries"
            ).limit(1))[0]
            logging.info("Last fuzzed queries resource id: %s", queries.id)
        else:
            queries = self.Parameters.wizard_queries_resource

        wizard_fuzzer = fuzzers.WizardFuzzer(
            self.Parameters.wizard_executable_resource,
            self.Parameters.wizard_config_resource,
            self.Parameters.wizard_data_resource,
            queries,
            add_queries=(self.Parameters.add_queries,),
            cmd_args={
                "rss_limit_mb": 40960,
                "max_total_time": self.Parameters.max_total_time,
                "runs": self.Parameters.runs,
            },
            count_coverage=self.Parameters.count_coverage,
            task=self,
        )
        with wizard_fuzzer:
            wizard_fuzzer.use_component()
        if self.Parameters.count_coverage:
            missing_info = wizard_fuzzer.output_results[fuzzers.CoverageCounting.NAME]["missing_info"]
            self.Context.coverage_info = missing_info
            self.Context.coverage_per_cent = 100.0 * missing_info["pcs_covered"] / (missing_info["pcs_instrumented"] or 1)
            self.set_info("Coverage: {:.3f} %".format(self.Context.coverage_per_cent))
            if missing_info["pcs_not_found_exist"]:
                self.set_info("WARNING: probably coverage info is not correct for this binary!")
