"""
    Components for fuzzers
"""
import glob
import logging
import os
import six

from sandbox.common.errors import TaskFailure
from sandbox.common.types import resource as ctr
from sandbox.common.fs import make_folder
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

import sandbox.projects.fuzzing.resources as fuzz_res
from sandbox.projects import resource_types
from sandbox.projects.common import config_patcher_tool as cfg_patcher
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import sanitizer
from sandbox.projects.common import time_utils as tu
from sandbox.projects.common.startrek_utils import ST
from sandbox.projects.common.search import config as sconf
from sandbox.projects.release_machine.core import const as rm_const


LOGGER = logging.getLogger(__name__)
PROBLEM_UNITS = ("crash", "slow-unit", "timeout", "leak", "mismatch", "oom")
SIGNIFICANT_PROBLEMS = frozenset(["crash", "leak", "mismatch", "oom"])


class FuzzingStage(object):
    NAME = "UNKNOWN_STAGE"
    ERR = "ERROR: "
    SYMBOLIZER_PATH = None

    def __init__(self, component, notify_on):
        self.component = component
        self.notify_on = notify_on
        self._st_client = None
        self.error_path = None
        self.result = {
            "return_code": None,
            "err_explanation": None,
            "err_dump": "",
            "problems": {},
            "problems_paths": {},
            "new_problems": {},
        }

    @classmethod
    def symbolizer_path(cls):
        if cls.SYMBOLIZER_PATH is None:
            cls.SYMBOLIZER_PATH = sanitizer.get_symbolizer()
        return cls.SYMBOLIZER_PATH

    def __enter__(self):
        LOGGER.info("Fuzzing stage: %s", self.NAME)
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.process_fails()
        self.collect_problems()
        self.custom_process()

    def __str__(self):
        return "{} for {}".format(self.NAME, self.component)

    def _command(self):
        raise NotImplementedError("Run command wasn't implemented for %s", type(self).__name__)

    def run(self):
        LOGGER.info("Run fuzzing")
        cmd = self._command()
        if cmd:
            with sdk2.helpers.ProcessLog(self.component.task, logger=self.NAME) as pl:
                self.result["return_code"] = sp.call(" ".join(cmd), shell=True, stdout=pl.stdout, stderr=pl.stderr)
                self.error_path = pl.stderr.path
        else:
            LOGGER.info("Command is empty, nothing to do")

    def process_fails(self):
        if self.result["return_code"]:
            LOGGER.info("Stage %s ends with return code %s", self.NAME, self.result["return_code"])
            self.find_err_explanation()
            self.link_to_error_log()
        else:
            LOGGER.info("Stage %s success!", self.NAME)

    def custom_process(self):
        """Define in children"""
        pass

    def link_to_error_log(self):
        task_log = resource_types.TASK_LOGS.find(task=self.component.task).first()
        if task_log:
            log_link = os.path.join(task_log.http_proxy, self.error_path.name)

            self.component.task.set_info(
                "Exception at {}: {}".format(self.NAME, lb.HREF_TO_ITEM.format(
                    link=log_link, name="log"
                )),
                do_escape=False,
            )
        else:
            LOGGER.error("Unable to get task log to parse error from it")

    def find_err_explanation(self):
        with open(str(self.error_path)) as err_file:
            in_err = 0
            for line in err_file:
                if self.ERR in line:
                    in_err = 1
                    self.result["err_explanation"] = line.split(self.ERR)[1]
                if in_err:  # dump first 30 lines of error trace
                    if in_err <= 30:
                        in_err += 1
                        self.result["err_dump"] += line
                    else:
                        return

    @property
    def st_client(self):
        if self._st_client is None:
            import startrek_client
            token = sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME)
            self._st_client = startrek_client.Startrek(token=token, useragent=rm_const.ROBOT_RELEASER_USER_NAME)
        return self._st_client

    def refresh_st_ticket(self, problem_name, files):
        """Finds first startrek ticket with tag=<problem-name>-md5-name and ping it"""
        try:
            issues = set()
            for file_name in files:
                issues |= set(self.st_client.issues.find(filter={'tags': file_name}))
            self.component.task.set_info("Try to refresh issues:\n{}".format([i.key for i in issues]))
            for i in issues:
                if not i.summary.startswith(type(self.component).__name__):
                    LOGGER.info("Skip issue for different fuzzy component")
                    continue
                is_closed = ST.is_issue_closed(i)
                if is_closed:
                    LOGGER.info("Ticket is closed, try to reopen it")
                    i.transitions['reopen'].execute()
                if ST.is_issue_forsaken(i, silence_time=7) or is_closed:
                    text = "Fuzzer still has {} problems, new fail: {}".format(
                        problem_name, lb.task_wiki_link(self.component.task.id)
                    )
                    i.comments.create(text=text)
        except Exception:
            LOGGER.error("Unable to ping ticket on existing problem: %s", eh.shifted_traceback())

    def new_st_ticket(self, problem_name, files):
        """SEARCH-4783"""
        try:
            list_files = list(files)
            issue = self.st_client.issues.create(
                queue=u"SEARCH",
                assignee=six.text_type(self.component.st_assignee),
                summary=u"{} has {} problem: {}".format(
                    type(self.component).__name__, problem_name, self.result["err_explanation"]
                ),
                description=u"Task: {}\nMD5 for new problems:\n{}\n%%{}%%".format(
                    lb.task_link(self.component.task.id, plain=True),
                    "\n".join(list_files),
                    self.result["err_dump"]

                ),
                tags=list_files
            )
            self.component.task.set_info("Issue created: {}".format(issue.key))
        except Exception:
            LOGGER.error("Unable to create new ticket on problem %s: %s", problem_name, eh.shifted_traceback())

    def problem_folder(self, problem_name):
        return "{}__{}__{}".format(self.NAME, problem_name, tu.date_ymd(sep="_"))

    def collect_problems(self):
        for problem_name in PROBLEM_UNITS:
            LOGGER.info("Try to collect '%s' queries for %s", problem_name)
            found_files = set(glob.glob("{}*".format(problem_name)))
            LOGGER.info("Found '%s' files: %s", problem_name, found_files)
            prev_problems_res, _, prev_problems_list = self.component.prev_problem_queries(problem_name)
            prev_problems_names = {i.name for i in prev_problems_list}
            LOGGER.info("Already reported '%s' files: %s", problem_name, prev_problems_names)
            new_problems = list(found_files - prev_problems_names)
            LOGGER.info("There are new problem queries: %s", new_problems)
            total_problems = len(found_files | prev_problems_names)
            setattr(self.component.task.Context, "total_{}_queries".format(problem_name), total_problems)
            res = None
            if len(found_files):
                if not new_problems:
                    LOGGER.info("All '%s' queries are already in last queries resource", problem_name)
                    res = prev_problems_res
                    if problem_name in self.notify_on:
                        self.refresh_st_ticket(problem_name, found_files)
                else:
                    res = self.component.problem_resources[problem_name](
                        self.component.task, "{} queries for task: {}".format(problem_name, self.component.task.id),
                        self.problem_folder(problem_name),
                        last_problem_queries=problem_name
                    )
                    res.path.mkdir(mode=0o777, exist_ok=True)
                    for i in new_problems:
                        transfer(self.component.task, i, str(res.path))
                    for i in prev_problems_list:
                        transfer(self.component.task, str(i), str(res.path), cmd="cp", cmd_descr="copy")
                    sdk2.ResourceData(res).ready()
                    if problem_name in self.notify_on:
                        self.new_st_ticket(problem_name, found_files)

                self.component.task.set_info(
                    "[{}] {} queries are in {}".format(self.NAME, problem_name, lb.resource_link(res.id)),
                    do_escape=False
                )
                for problem_file in found_files:
                    self.component.task.set_info(
                        "[{}] {} query is in {}".format(
                            self.NAME, problem_name,
                            lb.HREF_TO_ITEM.format(
                                link=os.path.join(res.http_proxy, problem_file),
                                name=problem_file
                            )
                        ),
                        do_escape=False
                    )
            self.result["problems"][problem_name] = list(found_files)
            self.result["problems_paths"][problem_name] = str(res.path) if res else None
            self.result["new_problems"][problem_name] = new_problems


class PreliminaryCheckCrash(FuzzingStage):
    NAME = "PRELIMINARY_CHECK_CRASH"

    def _command(self):
        _, crash_queries_path, _ = self.component.prev_problem_queries("crash")
        if crash_queries_path:
            return [
                "LSAN_OPTIONS=verbosity=1:log_threads=1",
                "ASAN_SYMBOLIZER_PATH={}".format(self.symbolizer_path()),
                str(self.component.binary_data.path),
                str(crash_queries_path),
                "-rss_limit_mb={}".format(self.component.cmd_args['rss_limit_mb']),
                "-runs=0",
            ]
        LOGGER.info("Cannot find previous problem for preliminary check")


class PreliminaryCheckOom(FuzzingStage):
    NAME = "PRELIMINARY_CHECK_OOM"

    def _command(self):
        _, oom_queries_path, _ = self.component.prev_problem_queries("oom")
        if oom_queries_path:
            return [
                "LSAN_OPTIONS=verbosity=1:log_threads=1",
                "ASAN_SYMBOLIZER_PATH={}".format(self.symbolizer_path()),
                str(self.component.binary_data.path),
                str(oom_queries_path),
                "-rss_limit_mb={}".format(self.component.cmd_args['rss_limit_mb']),
                "-runs=0",
            ]
        LOGGER.info("Cannot find previous problem for preliminary check")


class MainFuzzing(FuzzingStage):
    NAME = "MAIN_FUZZING"

    def _command(self):
        cmd = [
            "LSAN_OPTIONS=verbosity=1:log_threads=1",
            "ASAN_SYMBOLIZER_PATH={}".format(self.symbolizer_path()),
            str(self.component.binary_data.path),
            os.path.abspath(self.component.FUZZ_QUERIES_FOLDER),
        ]
        cmd.extend(self.component.add_queries_paths)
        for arg in self.component.cmd_args.items():
            cmd.append("-{}={}".format(*arg))

        return cmd


class CoverageCounting(FuzzingStage):
    NAME = "COVERAGE_COUNTING"
    COVERAGE_FOLDER = "coverage"
    DECRYPTED_COVERAGE_FILE = os.path.join(COVERAGE_FOLDER, "decrypted_coverage")

    def _command(self):
        make_folder(self.COVERAGE_FOLDER)
        cmd = [
            "ASAN_OPTIONS=coverage=1:coverage_dir={}:allocator_may_return_null=1".format(self.COVERAGE_FOLDER),
            "LSAN_OPTIONS=verbosity=1:log_threads=1",
            "ASAN_SYMBOLIZER_PATH={}".format(self.symbolizer_path()),
            str(self.component.binary_data.path),
            os.path.abspath(self.component.FUZZ_QUERIES_FOLDER),
            "-dump_coverage=1",
        ]
        cmd.extend(self.component.add_queries_paths)
        for arg in self.component.cmd_args.items():
            cmd.append("-{}={}".format(*arg))

        return cmd

    def collect_problems(self):
        """
            Coverage counting must not collect problems.
            Avoid missing some problems because of tasks race conditions
        """
        pass

    def custom_process(self):
        """Count coverage"""
        LOGGER.info("Try to count coverage")
        LOGGER.info("All cwd files: %s", os.listdir("."))
        coverage_res = fuzz_res.SanitizerFuzzyCoverage(
            self.component.task, "Sanitizer coverage for fuzzer", self.COVERAGE_FOLDER,
            fuzzer_type=type(self.component).__name__
        )
        coverage_res.path.mkdir(mode=0o755, exist_ok=True)

        files_list = os.listdir(self.COVERAGE_FOLDER)
        LOGGER.info("Coverage files: %s", files_list)
        for f_name in files_list:
            os.chmod(os.path.join(self.COVERAGE_FOLDER, f_name), 0o755)

        self.get_sancov_script()
        with sdk2.helpers.ProcessLog(self.component.task, logger="decrypt_sancov") as pl:
            sp.check_call(
                "./sancov.py print {} > {}".format(
                    " ".join([os.path.join(self.COVERAGE_FOLDER, f) for f in files_list]),
                    self.DECRYPTED_COVERAGE_FILE
                ),
                shell=True, stdout=pl.stdout, stderr=pl.stderr
            )
        with sdk2.helpers.ProcessLog(self.component.task, logger="human_readable_sancov") as pl:
            sp.check_call(
                "cat {} | addr2line -f -C -e {}".format(
                    self.DECRYPTED_COVERAGE_FILE,
                    self.component.binary_data.path
                ),
                shell=True, stdout=pl.stdout, stderr=pl.stderr
            )
        with sdk2.helpers.ProcessLog(self.component.task, logger="coverage_missing") as pl:
            sp.check_call(
                "./sancov.py missing --json {} < {}".format(
                    self.component.binary_data.path,
                    self.DECRYPTED_COVERAGE_FILE
                ),
                shell=True, stdout=pl.stdout, stderr=pl.stderr
            )
            try:
                missing_info_as_json = fu.json_load(str(pl.stderr.path))
                self.result["missing_info"] = missing_info_as_json
            except Exception:
                LOGGER.info("Unable to get missing info as json:\n%s", eh.shifted_traceback())

    @classmethod
    def get_sancov_script(cls):
        sdk2.svn.Arcadia.export("arcadia:/arc/trunk/arcadia/contrib/tools/sancov/sancov.py", "sancov.py")


class FuzzerComponentCommon(object):
    FUZZ_QUERIES_PACKED = "fuzzer_queries_main.tar.gz"
    FUZZ_QUERIES_FOLDER = "fuzz_queries_folder_main"
    FUZZ_QUERIES_FOLDER_ADD = "fuzz_queries_folder_add_"
    st_assignee = "mvel"  # by default
    problem_resources = {}
    rss_limit_mb = 40960  # 40 Gb

    def __init__(self, task, binary, queries, add_queries, count_coverage, cmd_args=None):
        """
            :param task: sdk2.Task object.
            :param binary: Fuzzer binary. Every component should have binary to fuzz.
            :param queries: Main queries to fuzz.
            :param add_queries: Additional queries to extend coverage.
            :param bool count_coverage: Defines fuzzing stages (children of FuzzingStage class)
            :param dict cmd_args: Parameters for fuzzer binary.
        """
        LOGGER.info("Init fuzzer")
        self.task = task
        self.binary_data = sdk2.ResourceData(binary)
        self.output_results = {}
        self.cmd_args = cmd_args or {}
        self.stages = (
            PreliminaryCheckCrash, PreliminaryCheckOom, MainFuzzing
        ) if not count_coverage else (CoverageCounting,)

        # It's important to deal with add queries before main because of moving paths
        self.add_queries_paths = self.get_add_queries_paths(add_queries)
        self._prev_problems = {}

        queries_data = sdk2.ResourceData(queries)
        queries_folder = unpack_queries(self.task, queries_data)
        transfer(self.task, queries_folder, self.FUZZ_QUERIES_FOLDER, cmd="cp -r", cmd_descr="copy_recursively")

    def prev_problem_queries(self, problem_name):
        if problem_name not in self._prev_problems:
            LOGGER.info("Try to get previous problems for %s", problem_name)
            prev_problems = list(sdk2.Resource.find(
                resource_type=self.problem_resources[problem_name],
                state=ctr.State.READY,
                attr_name="last_problem_queries",
                attr_value=problem_name
            ).limit(1))
            if prev_problems:
                LOGGER.info("Prev %s queries got: %s", problem_name, prev_problems[0])
                local_path = sdk2.ResourceData(prev_problems[0]).path
                self._prev_problems[problem_name] = (prev_problems[0], local_path, list(local_path.iterdir()))
            else:
                LOGGER.info("No previous problems found for %s", problem_name)
                self._prev_problems[problem_name] = (None, None, [])
        return self._prev_problems[problem_name]

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.collect_output_queries_corpus()
        LOGGER.info("Fuzzer output results: %s", self.output_results)
        self.task.Context.output_results = self.output_results

    @property
    def output_queries_resource(self):
        raise NotImplementedError("Output queries resource wasn't implemented for %s", type(self).__name__)

    def get_add_queries_paths(self, add_queries):
        """Prepare additional queries corpuses"""
        add_queries_paths = []
        for i, q in enumerate(add_queries):
            if q:
                add_queries_data = sdk2.ResourceData(q)
                add_queries_folder = unpack_queries(self.task, add_queries_data)
                add_q_path = self.FUZZ_QUERIES_FOLDER_ADD + str(i + 1)
                transfer(self.task, add_queries_folder, add_q_path, cmd="cp -r", cmd_descr="copy_recursively")
                add_queries_paths.append(os.path.abspath(add_q_path))
        return add_queries_paths

    def use_component(self):
        LOGGER.info("Current work directory contains: %s", os.listdir(os.path.abspath("")))
        LOGGER.info("Stages sequence: %s", ", ".join(map(str, self.stages)))
        for stage in self.stages:
            notify_on = SIGNIFICANT_PROBLEMS if self.cmd_args.get("runs") else set()  # SEARCH-8494
            s = stage(self, notify_on)
            with s:
                s.run()

            self.output_results[stage.NAME] = s.result
            if s.result["return_code"]:
                raise TaskFailure("Exception at {}".format(s.NAME))
            if any(s.result["problems"].get(name) for name in SIGNIFICANT_PROBLEMS):
                raise TaskFailure("Problem at {}: {}".format(s.NAME, s.result["problems"]))

    def collect_output_queries_corpus(self):
        pack_queries(self.task, self.FUZZ_QUERIES_FOLDER, self.FUZZ_QUERIES_PACKED)
        out_fuzzed_queries_res = self.output_queries_resource(self.task, "Fuzzed queries", self.FUZZ_QUERIES_PACKED)
        if self.cmd_args.get("runs"):
            # There is no sense to mark resources for no fuzzing runs
            out_fuzzed_queries_res.last_fuzzed_queries = tu.date_ymdhm(sep="_")
        sdk2.ResourceData(out_fuzzed_queries_res).ready()


class BegemotFuzzer(FuzzerComponentCommon):
    output_queries_resource = fuzz_res.BegemotQueriesFuzzy
    crash_queries_resource = fuzz_res.FuzzyBegemotCrashQueries
    slow_units_resource = fuzz_res.FuzzyBegemotSlowUnitQueries
    timeout_resource = fuzz_res.FuzzyBegemotTimeoutQueries
    st_assignee = "dmitryno"
    problem_resources = {
        "crash": fuzz_res.FuzzyBegemotCrashQueries,
        "slow-unit": fuzz_res.FuzzyBegemotSlowUnitQueries,
        "timeout": fuzz_res.FuzzyBegemotTimeoutQueries,
        "leak": fuzz_res.FuzzyBegemotLeakQueries,
        "mismatch": fuzz_res.FuzzyBegemotMismatchQueries,
        "oom": fuzz_res.FuzzyBegemotOomQueries,
    }

    def __init__(
        self, binary, queries,
        add_queries=(),  # tuple with additional queries resources
        cmd_args=None,
        count_coverage=False,
        task=None
    ):
        super(BegemotFuzzer, self).__init__(task, binary, queries, add_queries, count_coverage, cmd_args=cmd_args)


class WizardFuzzer(FuzzerComponentCommon):
    output_queries_resource = fuzz_res.WizardQueriesFuzzy
    crash_queries_resource = fuzz_res.FuzzyWizardCrashQueries
    slow_units_resource = fuzz_res.FuzzyWizardSlowUnitQueries
    timeout_resource = fuzz_res.FuzzyWizardTimeoutQueries
    st_assignee = "dmitryno"
    problem_resources = {
        "crash": fuzz_res.FuzzyWizardCrashQueries,
        "slow-unit": fuzz_res.FuzzyWizardSlowUnitQueries,
        "timeout": fuzz_res.FuzzyWizardTimeoutQueries,
        "leak": fuzz_res.FuzzyWizardLeakQueries,
        "mismatch": fuzz_res.FuzzyWizardMismatchQueries,
        "oom": fuzz_res.FuzzyWizardOomQueries,
    }

    def __init__(
        self, binary, config, data, queries,
        add_queries=(),  # tuple with additional queries resources
        cmd_args=None,
        count_coverage=False,
        task=None
    ):
        super(WizardFuzzer, self).__init__(task, binary, queries, add_queries, count_coverage, cmd_args=cmd_args)

        self.config_data = sdk2.ResourceData(config)
        self.data_data = sdk2.ResourceData(data)
        os.symlink(str(self.config_data.path), "wizard.cfg")  # name is hardcoded
        os.makedirs("wizard_data/wizard")
        os.symlink(str(self.data_data.path), "wizard_data/wizard/WIZARD_SHARD")  # name is hardcoded


class BlenderFuzzer(FuzzerComponentCommon):
    output_queries_resource = fuzz_res.WebBlenderQueriesFuzzy
    crash_queries_resource = fuzz_res.FuzzyBlenderCrashQueries
    slow_units_resource = fuzz_res.FuzzyBlenderSlowUnitQueries
    timeout_resource = fuzz_res.FuzzyBlenderTimeoutQueries
    st_assignee = "elshiko"
    problem_resources = {
        "crash": fuzz_res.FuzzyBlenderCrashQueries,
        "slow-unit": fuzz_res.FuzzyBlenderSlowUnitQueries,
        "timeout": fuzz_res.FuzzyBlenderTimeoutQueries,
        "leak": fuzz_res.FuzzyBlenderLeakQueries,
        "mismatch": fuzz_res.FuzzyBlenderMismatchQueries,
        "oom": fuzz_res.FuzzyBlenderOomQueries,
    }

    def __init__(
        self, binary, config, data, queries,
        add_queries=(),  # tuple with additional queries resources
        cmd_args=None,
        count_coverage=False,
        task=None
    ):
        super(BlenderFuzzer, self).__init__(task, binary, queries, add_queries, count_coverage, cmd_args=cmd_args)
        self.data_data = sdk2.ResourceData(data)
        self.config_data = sdk2.ResourceData(config)
        self._patch_cfg()

    def _patch_cfg(self):
        all_data_path = str(self.data_data.path)
        rd_dir = os.path.join(all_data_path, "rearrange.dynamic")
        rearrange_dir = os.path.join(all_data_path, "rearrange")
        q_cache_dir = "querycachedir"
        make_folder(q_cache_dir)
        fu.json_dump(
            "patch_for_blender_cfg",
            {
                "Server.Port": "${Port or 9080}",
                "Server.AppHostOptions": (
                    "Port=${AppHostPort or '9081'}, Threads=${AppHostThreads or '32'}, GrpcPort=+2, GrpcThreads=24"
                ),
                "Server.LoadLog": "loadlog",
                "Server.ServerLog": "serverlog",
                "Server.EventLog": "eventlog",
                "Collection.RearrangeDynamicDataDir": rd_dir,
                "Collection.RearrangeDataDir": rearrange_dir,
                "DNSCache": "__remove__",
            }
        )
        cfg_patcher.patch_cfg(
            self.task, str(self.config_data.path), "patch_for_blender_cfg", "noapache.cfg", "blender_fuzzer"
        )


class MiddlesearchFuzzer(FuzzerComponentCommon):
    output_queries_resource = fuzz_res.WebMiddlesearchQueriesFuzzy
    crash_queries_resource = fuzz_res.FuzzyMiddlesearchCrashQueries
    slow_units_resource = fuzz_res.FuzzyMiddlesearchSlowUnitQueries
    timeout_resource = fuzz_res.FuzzyMiddlesearchTimeoutQueries
    st_assignee = "ulyanin"
    problem_resources = {
        "crash": fuzz_res.FuzzyMiddlesearchCrashQueries,
        "slow-unit": fuzz_res.FuzzyMiddlesearchSlowUnitQueries,
        "timeout": fuzz_res.FuzzyMiddlesearchTimeoutQueries,
        "leak": fuzz_res.FuzzyMiddlesearchLeakQueries,
        "mismatch": fuzz_res.FuzzyMiddlesearchMismatchQueries,
        "oom": fuzz_res.FuzzyMiddlesearchOomQueries,
    }
    rss_limit_mb = 80 * 1024  # 80 Gb

    def __init__(
        self, binary, config, data, models, queries,
        basesearches=None,
        add_queries=(),  # tuple with additional queries resources
        cmd_args=None,
        count_coverage=False,
        task=None
    ):
        super(MiddlesearchFuzzer, self).__init__(task, binary, queries, add_queries, count_coverage, cmd_args=cmd_args)
        self.models_data = sdk2.ResourceData(models)
        self.rearrdata_data = sdk2.ResourceData(data)
        self.config_data = sdk2.ResourceData(config)
        self._patch_cfg(basesearches)

    def _patch_cfg(self, basesearches):
        init_cfg_path = str(self.config_data.path)
        if basesearches:
            # todo: use c++ config patcher for this
            tmp_cfg_path = "tmp_cfg_path"
            patched_cfg = sconf._replace_basesearches(fu.read_file(init_cfg_path), basesearches)
            fu.write_file(tmp_cfg_path, patched_cfg)
        else:
            tmp_cfg_path = init_cfg_path
        rearr_data_path = str(self.rearrdata_data.path)
        pure_dir = os.path.join(rearr_data_path, "pure")
        rearrange_dir = os.path.join(rearr_data_path, "rearrange")
        q_cache_dir = "querycachedir"
        make_folder(q_cache_dir)
        fu.json_dump(
            "patch_for_mmeta_cfg",
            {
                "Server.LoadLog": "loadlog",
                "Server.ServerLog": "serverlog",
                "Server.EventLog": "eventlog",
                "Collection.IndexDir": pure_dir,
                "Collection.MXNetFile": str(self.models_data.path),
                "Collection.RearrangeDataDir": rearrange_dir,
                "Collection.QueryCache[0].Dir": os.path.abspath(os.path.join(q_cache_dir, "tmpdir.0")),
                'Collection.QueryCache[1:]': "__remove__",
                "DNSCache": "__remove__",
            }
        )
        patched_cfg = cfg_patcher.patch_cfg(
            self.task, tmp_cfg_path, "patch_for_mmeta_cfg", "metasearch.cfg", "middlesearch_fuzzer"
        )
        LOGGER.debug("Patched config for middlesearch fuzzer:\n%s", patched_cfg)


class BasesearchFuzzer(FuzzerComponentCommon):
    _FUZZ_LOGS_FOLDER = "fuzz_logs"
    _LOAD_LOG_PATH = os.path.join(_FUZZ_LOGS_FOLDER, "fuzz_base_load.log")
    _SERVER_LOG_PATH = os.path.join(_FUZZ_LOGS_FOLDER, "fuzz_base_server.log")
    _PASSAGE_LOG_PATH = os.path.join(_FUZZ_LOGS_FOLDER, "fuzz_base_passage.log")

    def __init__(
        self, binary, config, models, database, queries,
        add_queries=(),  # tuple with additional queries resources
        cmd_args=None,
        count_coverage=False,
        task=None
    ):
        super(BasesearchFuzzer, self).__init__(task, binary, queries, add_queries, count_coverage, cmd_args=cmd_args)
        self.config_data = sdk2.ResourceData(config)
        self.models_data = sdk2.ResourceData(models)
        self.database_data = sdk2.ResourceData(database)
        self.setup_cfg()

    @property
    def cfg_filename(self):
        raise NotImplementedError("Filename for config must be specified")

    def setup_cfg(self):
        os.mkdir(self._FUZZ_LOGS_FOLDER)
        config_obj = sconf.BasesearchWebConfig.get_config_from_file(str(self.config_data.path))
        config_obj.apply_local_patch({
            "Collection/IndexDir": str(self.database_data.path),
            "Collection/MXNetFile": (str(self.models_data.path), True),
            "Server/LoadLog": self._LOAD_LOG_PATH,
            "Server/ServerLog": self._SERVER_LOG_PATH,
            "Server/PassageLog": self._PASSAGE_LOG_PATH,
        })
        config_obj.save_to_file(self.cfg_filename)


class WebBasesearchFuzzer(BasesearchFuzzer):
    output_queries_resource = fuzz_res.PlainTextQueriesFuzzy
    crash_queries_resource = fuzz_res.FuzzyBasesearchCrashQueries
    slow_units_resource = fuzz_res.FuzzyBasesearchSlowUnitQueries
    timeout_resource = fuzz_res.FuzzyBasesearchTimeoutQueries
    cfg_filename = "basesearch.cfg"
    st_assignee = "nikita-uvarov"
    problem_resources = {
        "crash": fuzz_res.FuzzyBasesearchCrashQueries,
        "slow-unit": fuzz_res.FuzzyBasesearchSlowUnitQueries,
        "timeout": fuzz_res.FuzzyBasesearchTimeoutQueries,
        "leak": fuzz_res.FuzzyBasesearchLeakQueries,
        "mismatch": fuzz_res.FuzzyBasesearchMismatchQueries,
        "oom": fuzz_res.FuzzyBasesearchOomQueries,
    }


class VideoBasesearchFuzzer(BasesearchFuzzer):
    output_queries_resource = fuzz_res.VideoBasesearchQueriesFuzzy
    crash_queries_resource = fuzz_res.FuzzyVideoBasesearchCrashQueries
    slow_units_resource = fuzz_res.FuzzyVideoBasesearchSlowUnitQueries
    timeout_resource = fuzz_res.FuzzyVideoBasesearchTimeoutQueries
    cfg_filename = "video_basesearch.cfg"
    st_assignee = "juver"
    problem_resources = {
        "crash": fuzz_res.FuzzyVideoBasesearchCrashQueries,
        "slow-unit": fuzz_res.FuzzyVideoBasesearchSlowUnitQueries,
        "timeout": fuzz_res.FuzzyVideoBasesearchTimeoutQueries,
        "leak": fuzz_res.FuzzyVideoBasesearchLeakQueries,
        "mismatch": fuzz_res.FuzzyVideoBasesearchMismatchQueries,
        "oom": fuzz_res.FuzzyVideoBasesearchOomQueries,
    }


def pack_queries(task, from_path, to_path):
    with sdk2.helpers.ProcessLog(task, logger="pack_queries_to_tar_gz") as pl:
        sp.Popen(
            ["tar", "-czvf", to_path, from_path],
            stdout=pl.stdout, stderr=pl.stderr,
        ).wait()


def unpack_queries(task, queries_data):
    queries_data_path = str(queries_data.path)
    if queries_data_path.endswith("tar.gz"):
        with sdk2.helpers.ProcessLog(task, logger="unpack_queries_from_tar_gz") as pl:
            sp.Popen(
                ["tar", "-xzvf", str(queries_data.path)],
                stdout=pl.stdout, stderr=pl.stderr
            ).wait()
            with open(str(pl.stdout.path)) as p_stdout:
                for line in p_stdout:
                    queries_folder = line.rstrip("/\n")
                    if queries_folder:
                        return queries_folder
    else:
        LOGGER.info("No need to unpack folder")
        return queries_data_path


def transfer(task, old_path, new_folder, cmd="mv", cmd_descr="move"):
    if old_path != new_folder:
        LOGGER.info("Try to %s: '%s' to '%s'", cmd_descr, old_path, new_folder)
        with sdk2.helpers.ProcessLog(task, logger="{}_path".format(cmd_descr)) as pl:
            retcode = sp.Popen(
                "{} {} {}".format(cmd, old_path, new_folder),
                shell=True, stdout=pl.stdout, stderr=pl.stderr
            ).wait()
            if retcode != 0:
                LOGGER.error("Something went wrong:\n%s", fu.read_lines(str(pl.stderr.path))[-20:])
    else:
        LOGGER.info("No need to %s folder: %s = %s", cmd_descr, old_path, new_folder)
