from collections import namedtuple
import datetime
import dateutil.parser
import logging
import pytz
import requests

from sandbox import sdk2
from sandbox.projects.maps.common.retry import retry
from sandbox.projects.garden.common import config as garden_config
from sandbox.sandboxsdk.environments import PipEnvironment


GARDEN_MODULE_STATISTICS_URL = "{}/module_statistics/"
GARDEN_STORAGE_URL = "{}/storage/"
OFFLINE_TILE_CACHE_MODULE = "offline_tile_cache"

SEND_TO_STAT = {
    1: {
        "name": "Moscow and Moscow Oblast",
        "locales": ["ru_RU", "en_RU"]
    },
    213: {
        "name": "Moscow",
        "locales": ["ru_RU", "en_RU"]
    },
    2: {
        "name": "Saint-Petersburg",
        "locales": ["ru_RU", "en_RU"]
    },
    123: {
        "name": "Finland",
        "locales": ["ru_RU", "en_RU"]
    },
    11232: {
        "name": "Yamalo-Nenets Autonomous Okrug",
        "locales": ["ru_RU", "en_RU"]
    },
    11309: {
        "name": "Krasnoyarsk Krai",
        "locales": ["ru_RU", "en_RU"]
    },
}


def _in_stat_dict(row):
    region_id = row["region_id"]
    if region_id not in SEND_TO_STAT.keys():
        return False
    return row["locale"] in SEND_TO_STAT[region_id]["locales"]


def _get_region_name(region_id):
    if region_id not in SEND_TO_STAT.keys():
        return "Unknown"
    return SEND_TO_STAT[region_id]["name"]


def _request_json(url, params):
    r = requests.get(url, params=params)
    r.raise_for_status()
    return r.json()


@retry(tries=5, delay=10)
def _get_module_builds(garden_url, module, from_date, contour):
    logging.info("Getting {} builds".format(module))
    return _request_json(GARDEN_MODULE_STATISTICS_URL.format(garden_url), {"module": module, "from": from_date, "contour": contour})


@retry(tries=5, delay=10)
def _get_build_resources(garden_url, build):
    logging.info("Getting {} resources".format(build["name"]))
    try:
        return _request_json(GARDEN_STORAGE_URL.format(garden_url), {
            "build": "{0}:{1}".format(build["name"], build["id"]),
            "limit": 20000
        })
    except Exception as e:
        logging.warning("Request has failed with error {}".format(e))
    return []


def _get_build_full_name(build):
    return "{0}:{1}".format(build["name"], build["id"])


def _to_ms(seconds):
    return int(seconds * 1000)


def _to_timestamp(d):
    if d.tzinfo:
        epoch = datetime.datetime(1970, 1, 1, tzinfo=pytz.UTC)
        dt = d.astimezone(pytz.UTC) - epoch
    else:
        epoch = datetime.datetime(1970, 1, 1)
        dt = d - epoch
    return int(dt.total_seconds())


def _gather_stat(build, resources):
    rows = []
    build_duration = _to_ms(
        (dateutil.parser.parse(build["completed_at"]) - dateutil.parser.parse(build["started_at"])).seconds)
    date = dateutil.parser.parse(build["completed_at"]).strftime("%Y-%m-%d %H:%M:00")
    created_at = _to_ms(_to_timestamp(dateutil.parser.parse(build["started_at"])))
    for r in resources:
        if not r["name"].startswith("cache_file"):
            continue
        properties = r["properties"]
        row = {
            "file_name": r["name"],
            "fielddate": date,
            "name" : _get_build_full_name(build),
            "duration": build_duration,
            "created_at": created_at,
            "environment": build["contour_name"],
            "domain": properties["domain"],
            "region_id": properties["region"],
            "locale": properties["locale"],
            "size": r["size"]["bytes"],
            "format": properties["layer"],
            "region_name": _get_region_name(properties["region"])
        }
        rows.append(row)
    return rows


def _merge_dicts(d1, d2):
    res = d1.copy()
    res.update(d2)
    return res


def _aggregate_stat(rows):
    GroupBy = namedtuple("GroupBy", ["name", "environment", "format", "domain"])
    d = {}
    for row in rows:
        key = GroupBy(**{k: row[k] for k in GroupBy._fields})
        if key not in d:
            d[key] = {
                "fielddate": row["fielddate"],
                "id": "_".join(map(str, key)),
                "duration": row["duration"],
                "tasks_count": 0,
                "error_count": 0,
                "created_at" : row["created_at"],
                "size": 0
            }
        d[key]["size"] += row["size"]

    return [_merge_dicts(k._asdict(), v) for k, v in d.items()]


class OfflineTileCacheToStat(sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = (PipEnvironment("python-statface-client", "0.142.0", use_wheel=False), )

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.String("Contour") as contour:
            contour.values.datatesting = "datatesting"
            contour.values.stable = contour.Value("stable", default=True)

        days_ago = sdk2.parameters.Integer("Days ago", required=True, default_value=1)

    def on_execute(self):
        import statface_client
        garden_url = "http://" + garden_config.server_hostname("stable")

        from_date = (datetime.datetime.now() - datetime.timedelta(days=self.Parameters.days_ago)).isoformat()
        data = []
        for b in _get_module_builds(garden_url, OFFLINE_TILE_CACHE_MODULE, from_date, self.Parameters.contour):
            logging.info("Processing build {0}".format(b["properties"]["release_name"]))
            data.extend(_gather_stat(b, _get_build_resources(garden_url, b)))

        logging.info("Aggregate stats")
        aggr_data = _aggregate_stat(data)

        logging.info("Filter regions")
        filtered_data = [r for r in data if _in_stat_dict(r)]

        sfcli = statface_client.StatfaceClient(host="upload.stat.yandex-team.ru", oauth_token=sdk2.Vault.data("MAPS_GARDEN", "robot-garden-stat-token"))
        logging.info("Uploading aggregated stat ({0} rows)".format(len(aggr_data)))
        stat_report = sfcli.get_report("ExtData/garden/offline_tile_cache_aggregated")
        stat_report.upload_data(scale="i", data=aggr_data)

        logging.info("Uploading regions stat ({0} rows)".format(len(filtered_data)))
        stat_report = sfcli.get_report("ExtData/garden/offline_tile_cache")
        stat_report.upload_data(scale="i", data=filtered_data)
