from collections import namedtuple
import datetime
import dateutil.parser
import json
import logging
import requests

from sandbox import sdk2
from sandbox.projects.maps.common.retry import retry
from sandbox.projects.garden.common import config as garden_config
from sandbox.sandboxsdk.environments import PipEnvironment


RenderBuildInfo = namedtuple("RenderBuildInfo", ["design_name", "start_time", "finish_time"])
GARDEN_MODULE_STATISTICS_URL = "{}/module_statistics/"
GARDEN_BUILD_HIERARCHY_URL = "{}/build_hierarchy/"
STYLEREPO_MODULE = "stylerepo_map_design_src"


def _request_json(url, params):
    r = requests.get(url, params=params)
    r.raise_for_status()
    return r.json()


@retry(tries=10, delay=5)
def _load_render_modules_dict(sfcli):
    d = json.loads(sfcli.get_stat_dict("Renderer_modules").download(language="en"))
    return {v: k for k, v in d.iteritems()}


@retry(tries=5, delay=10)
def get_render_builds(garden_url, module, from_date):
    logging.info("Getting {} builds".format(module))
    return _request_json(GARDEN_MODULE_STATISTICS_URL.format(garden_url), {"module": module, "from": from_date})


@retry(tries=5, delay=10)
def _get_hierarchy(garden_url, build):
    hierarchy_url = GARDEN_BUILD_HIERARCHY_URL.format(garden_url)
    return _request_json(hierarchy_url, {"module": build["name"], "build_id": build["id"]})


def _get_stylerepo_build(garden_url, render_build):
    hierarchy = _get_hierarchy(garden_url, render_build)
    for build in hierarchy:
        if build["name"] == STYLEREPO_MODULE:
            return build


def get_render_builds_info(garden_url, render_builds, render_modules):
    logging.info("Getting render builds info")
    result = {}
    visited_builds = set()
    for render_build in render_builds:
        if (render_build["name"], render_build["id"]) in visited_builds:
            continue

        stylerepo_build = _get_stylerepo_build(garden_url, render_build)

        design_name = stylerepo_build["properties"]["release_name"]
        hierarchy = _get_hierarchy(garden_url, stylerepo_build)

        for descendant in hierarchy:
            visited_builds.add((descendant["name"], descendant["id"]))
            started_at = descendant["started_at"]
            completed_at = descendant.get("completed_at")
            if descendant["name"] in render_modules and started_at and completed_at:
                result[(descendant["name"], descendant["id"])] = RenderBuildInfo(
                    design_name,
                    dateutil.parser.parse(started_at),
                    dateutil.parser.parse(completed_at))

    return result


def make_stats(render_builds_info, render_modules):
    data = []

    for key, b_info in render_builds_info.iteritems():
        name, build_id = key
        data.append({
            "fielddate": b_info.start_time.strftime("%Y-%m-%d %H:%M:00"),
            "build_type": int(render_modules.get(name)),
            "id": int(build_id),
            "design_name": b_info.design_name,
            "duration": int((b_info.finish_time - b_info.start_time).total_seconds())
        })

    return data


class RenderModulesToStat(sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = (PipEnvironment("python-statface-client", "0.142.0", use_wheel=False), )

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.String("Environment", multiline=True) as env:
            env.values.testing = env.Value("testing", default=True)
            env.values.production = "production"
        days_ago = sdk2.parameters.Integer("Days ago", required=True, default_value=1)

    def on_execute(self):
        import statface_client
        garden_url = "http://" + garden_config.server_hostname(self.Parameters.env)

        from_date = (datetime.datetime.now() - datetime.timedelta(days=self.Parameters.days_ago)).isoformat()
        sfcli = statface_client.StatfaceClient(host="upload.stat.yandex-team.ru", oauth_token=sdk2.Vault.data("MAPS_GARDEN", "robot-garden-stat-token"))
        render_modules = _load_render_modules_dict(sfcli)

        render_builds = []
        for module in render_modules.keys():
            render_builds.extend(get_render_builds(garden_url, module, from_date))

        render_builds_info = get_render_builds_info(garden_url, render_builds, render_modules)
        data = make_stats(render_builds_info, render_modules)

        logging.info("Uploading data ({0} rows)".format(len(data)))
        stat_report = sfcli.get_report("ExtData/garden/render-stats")
        stat_report.upload_data(scale="i", data=data)
