# -*- coding: utf-8 -*-
import json
import traceback

import sandbox.sdk2 as sdk2
import sandbox.sdk2.parameters as parameters
import sandbox.common.errors as errors

import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

import sandbox.projects.gencfg.mongo as mongo
import sandbox.projects.gencfg.helpers as helpers
import sandbox.projects.gencfg.semaphores as semaphores
import sandbox.projects.gencfg.environment as environment


class AllocateInDynamic(sdk2.Task):
    """ Allocate group in dynamic """

    class Requirements(sdk2.Task.Requirements):
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 10 * 1024, None)
        ram = 80 * 1024
        cores = 16
        client_tags = ctc.Tag.CUSTOM_GENCFG_BUILD
        semaphores = semaphores.gencfg.dynamic

    class Context(sdk2.Task.Context):
        svn_diff = ''
        exceptions = []
        commit = -1
        gencfg_revision = -1
        retry_count = 0

    class Parameters(sdk2.Task.Parameters):
        author = parameters.String('Request author')
        parent_group_name = parameters.String(
            'Parent group name', description='ALL_DYNAMIC, ALL_PERSONAL etc.', default='ALL_DYNAMIC', required=False
        )
        group_name = parameters.String(
            'Group name without LOC_', description='e.g. WEB_BASE, if you want to allocate MAN_WEB_BASE'
        )
        group_descr = parameters.String('Group purpose and other human-readable description')
        owners = parameters.List('List of owners')
        watchers = parameters.List('List of watchers')

        ctype = parameters.String('Cluster type (prod, prestable)', default='test')
        itype = parameters.String('Instance type', default='none')
        prj = parameters.List(
            'Project',
            description='e.g. web-jupiter or l7-balancer',
            default=['none'],
        )
        metaprj = parameters.String('Accounting metaproject', default='unknown')
        itags = parameters.List('Custom instance tags')
        dispenser_project_key = parameters.String('Dispenser project key', default=None)
        affinity_category = parameters.String('Affinity category', default=None, required=False)

        with parameters.Group('Requirements') as req:
            instance_count = parameters.Integer('Instance count', default=3)
            cores_per_instance = parameters.Integer('Cores (PER INSTANCE)', default=1)
            memory = parameters.Float('RAM (Gb)', default=1)
            disk = parameters.Float('Hard disk (not ssd) (Gb)', default=1)
            ssd = parameters.Float('SSD (Gb)', default=0)
            net = parameters.Float('Net (Gb)', default=0)

        locations = parameters.CheckGroup(
            'Location',
            choices=[
                ('Mantsala', 'man'),
                ('Sasovo', 'sas'),
                ('Vladimir', 'vla'),
                ('Moscow/Ivanteevka', 'msk_iva'),
                ('Moscow/Mytischi', 'msk_myt'),
            ]
        )

        volumes = parameters.JSON('Volumes', default=[], required=False)

        with parameters.Group('Additional') as additional:
            parent_macro = parameters.String('Parent macro', default='')
            ipv64_tunnel_v2 = parameters.Bool('IPv6to4 tunnel', default=False)
            internet_tunnel = parameters.Bool('Internet tunnel', default=False)
            mtn_export_to_cauth = parameters.Bool('Export MTN hostnames to CAUTH', default=False)
            use_mtn_in_config = parameters.Bool('Use MTN addresses in CONFIGS', default=False)
            portovm_mtn_addrs = parameters.Bool('Use MTN addresses in PORTOVM', default=False)
            hbf_slb_name = parameters.List('LSB names', default=[])

        commit_message = parameters.String('Commit message', default='', required=False)

        with parameters.Group('Execution') as execution:
            dry_run = sdk2.parameters.Bool(
                'Dry run',
                description='Run the whole procedure, but do not commit results',
                default=True
            )

            arcadia_revision = sdk2.parameters.String(
                'Base revision',
                description='Allocate, check and commit against this revision',
                required=False
            )

            use_last_resources = sdk2.parameters.Bool(
                'Use last released resources',
                required=True,
                default=False
            )

    def on_execute(self):
        self.Context.retry_count += 1

        try:
            gencfg = environment.GencfgEnvironment(self, self.Parameters.arcadia_revision, self.ramdrive.path)
            gencfg.prepare()
            gencfg.install(self.Parameters.use_last_resources)

            gencfg_revision = int(gencfg.info(gencfg.src_root)['commit_revision'])
            self.set_info('Task manipulates revision {}'.format(gencfg_revision))
            self.Context.gencfg_revision = gencfg_revision

            try:
                self.allocate_in_dynamic(gencfg, self.Parameters)
                if not self.save_svn_diff(gencfg):
                    self.set_info('Nothing to commit.')
                    return
            except Exception:
                self.set_info("allocate_in_dynamic exception: %s" % traceback.format_exc())
                raise errors.TaskFailure('Error in resource allocation process')

            try:
                gencfg.update()
            except Exception as e:
                raise errors.TemporaryError('{}: {}. Task will be restarted'.format(type(e).__name__, e))

            gencfg.run_script('scripts/gen-topology-check.sh')

            if not self.Parameters.dry_run:
                commit = self.commit(
                    gencfg,
                    u'[AllocateInDynamic] {}: {} (committed by {}@) '
                    'from https://sandbox.yandex-team.ru/task/{}/view'.format(
                        self.Parameters.group_name, self.Parameters.commit_message.encode('ascii', 'ignore'),
                        self.Parameters.author, self.id
                    ),
                    restart_on_fail=not bool(self.Parameters.arcadia_revision)
                )

                if not commit:
                    return

                self.Context.commit = int(commit)
                self.set_info(
                    'Request committed: <a href="https://a.yandex-team.ru/arc/commit/{0}">{0}</a>'.format(commit),
                    do_escape=False
                )

                self.insert_commit(commit, self.Parameters.author or self.author, self.id)

                gencfg.update(commit)
                gencfg.run_script('utils/mongo/populate_gencfg_trunk.py')
        except (errors.TaskFailure, errors.TemporaryError):
            raise
        except Exception as e:
            if self.Context.retry_count > 3:
                raise
            raise errors.TemporaryError('{}: {}'.format(type(e).__name__, e))

    def on_finish(self, prev_status, status):
        if status != ctt.Status.SUCCESS:
            self.save_exceptions()

    def on_break(self, prev_status, status):
        self.save_exceptions()

    def allocate_in_dynamic(self, gencfg, params):
        def format_list(lst):
            return ','.join(str(obj) for obj in lst)

        def format_size(param, unit):
            return '{}{}'.format(param, unit)

        def format_prj(param):
            return '[{}]'.format(','.join('"{}"'.format(obj) for obj in param))

        def format_python_list(param):
            return '[{}]'.format(','.join('"{}"'.format(obj) for obj in param))

        self.check_exist_allocating_groups(gencfg, params.group_name, set(params.locations))

        created_groups = []
        for location in set(params.locations):
            location = location.upper()
            location_group = (location + '_' + params.group_name).upper()
            min_power = params.cores_per_instance * 40 * params.instance_count

            allocate_command = [
                'optimizers/dynamic/main.py',
                '-y',
                '-m', params.parent_group_name,
                '-a', 'add',
                '-g', location_group,
                '-d', params.group_descr,
                '--owners', format_list(params.owners),
                '--watchers', format_list(params.watchers),
                '--ctype', params.ctype,
                '--itype', params.itype,
                '--prj', format_prj(params.prj),
                '--metaprj', params.metaprj,
                '--location', location,
                '--min_replicas', str(params.instance_count),
                '--max_replicas', str(params.instance_count),
                '--memory', format_size(params.memory, 'Gb'),
                '--min_power', str(min_power),
                '--equal_instances_power', str(True),
                '--disk', format_size(params.disk, 'Gb'),
                '--ssd', format_size(params.ssd, 'Gb'),
                '--net', format_size(params.net, 'Mbit'),
                # '--itag', params.itag,
            ]

            if params.affinity_category:
                allocate_command.extend(['--affinity_category', str(params.affinity_category)])

            gencfg.run_script(*allocate_command)
            created_groups.append(location_group)

            if params.volumes:
                modify_volumes_command = [
                    './utils/common/manipulate_volumes.py',
                    '-a', 'put',
                    '-g', location_group,
                    '-j', json.dumps(params.volumes),
                ]

                gencfg.run_process(modify_volumes_command, 'modify_volumes')

            for parameter, value in (
                ('properties.hbf_parent_macros', params.parent_macro),
                ('properties.ipip6_ext_tunnel_v2', params.ipv64_tunnel_v2),
                ('properties.internet_tunnel', params.internet_tunnel),
                ('properties.mtn.export_mtn_to_cauth', params.mtn_export_to_cauth),
                ('properties.mtn.use_mtn_in_config', params.use_mtn_in_config),
                ('properties.mtn.portovm_mtn_addrs', params.portovm_mtn_addrs),
                ('properties.mtn.tunnels.hbf_slb_name', format_python_list(params.hbf_slb_name)),
                ('dispenser.project_key', params.dispenser_project_key),
            ):
                if not value:
                    continue

                modify_group_card_command = [
                    './utils/common/update_card.py',
                    '-y',
                    '-g', location_group,
                    '-k', parameter,
                    '-v', str(value)
                ]

                gencfg.run_process(modify_group_card_command, 'modify_{}'.format(parameter))

    def get_has_groups(self, gencfg, groups):
        groups_param = ','.join(groups)
        output = gencfg.run_process_output(['./utils/common/has_group.py', groups_param], 'has_group')
        return [x.strip() == 'True' for x in output.split(' ')]

    def check_exist_allocating_groups(self, gencfg, group_name, locations):
        has_groups = self.get_has_groups(gencfg, [(x + '_' + group_name).upper() for x in locations])
        has_groups = {(x + '_' + group_name).upper(): has for x, has in zip(locations, has_groups)}

        errors = []
        for group, has in has_groups.items():
            if has:
                errors.append(group)

        if errors:
            raise Exception('Allocation request contains existing groups: {}'.format(', '.join(errors)))

    def save_svn_diff(self, gencfg):
        diff = gencfg.diff()
        self.Context.svn_diff = diff
        return diff

    def commit(self, gencfg, message, restart_on_fail=False):
        diff = gencfg.diff()
        if not diff:
            self.set_info('Nothing to commit.')
            return
        self.Context.svn_diff = diff

        try:
            return gencfg.commit(message)
        except Exception as e:
            if restart_on_fail:
                raise errors.TemporaryError(str(e))
            raise

    def insert_commit(self, commit, author, task_id):
        commit = mongo.normalize_commit(commit)

        mongo.commits().update(
            {'commit': commit},
            {
                '$set': {
                    'skip': {'mongo': True, 'clickhouse': False, 'hbf': False},
                    'author': format(author),
                    'task_id': task_id,
                    'interface': True
                },
            },
            upsert=True
        )

    def save_exceptions(self):
        self.set_info('RESOURCE ALLOCATION FAILED:\n    '
                      'If you need some help please create ticket at '
                      'https://st.yandex-team.ru/gencfg with link to that task in it.')
        helpers.print_errors_to_info(self, self.log_path())
        self.Context.exceptions.append('If you need some help please create ticket at '
                                       'https://st.yandex-team.ru/gencfg with link to that task in it.')
        self.Context.exceptions = helpers.get_list_errors(self.log_path())
