# -*- coding: utf-8 -*-

import os
import os.path
from distutils.dir_util import mkpath
import logging
import requests
import json
import re

from copy import deepcopy

from sandbox import sdk2
import sandbox.sdk2.helpers
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk import ssh
import sandbox.common.types.misc as ctm


class BackgroundGencfgGuiTasks(sdk2.Task):
    class Requirements(sdk2.Requirements):
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 10 * 1024, None)
        ram = 80 * 1024
        cores = 16

    class Parameters(sdk2.Task.Parameters):
        json_params = sdk2.parameters.String('json params')
        dry_run = sdk2.parameters.Bool('do not commit, run checks only', default=True)

    @staticmethod
    def fix_multiline_command(command):
        fixed_command = ' '.join(line.strip() for line in command.split('\n'))
        return fixed_command

    def get_gencfg_path(self):
        return os.path.join(str(self.ramdrive.path), "gencfg_source")

    def get_tmp_path(self):
        tmp_path = os.path.join(str(self.ramdrive.path), "tmp")
        if not os.path.exists(tmp_path):
            mkpath(tmp_path)

        return tmp_path

    def run_check_gencfg_if_not_checked_yet(self):
        local_revision = Arcadia.info(os.path.join(self.get_gencfg_path(), 'db'))['commit_revision']
        already_checked_and_ok = False
        try:
            r = requests.get('https://clusterstate.yandex-team.ru/gencfg/json', timeout=30)
            r.raise_for_status()
            for commit_info in r.json()['commits']:
                if int(commit_info['commit']) == local_revision:
                    already_checked_and_ok = commit_info['test_passed']
        except:
            logging.exception("Failed to get commit info from clusterstate")

        # if the commit was not checked
        # or if the commit was checked but check failed - recheck in case of false negative
        if not already_checked_and_ok:
            logging.info("Could not find a successfully checked commit, running a full gencfg check")
            self.run_check_gencfg()

    def run_check_gencfg(self):
        logging.info("Running gen.sh")
        try:
            with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("gen_sh")) as pl:
                sp.check_call(
                    'cd {gencfg_path} && TMP={tmp} TMPDIR={tmp} ./gen.sh'.format(gencfg_path=self.get_gencfg_path(), tmp=self.get_tmp_path()),
                    shell=True, stdout=pl.stdout, stderr=pl.stderr
                )
        except:
            Arcadia.cleanup(self.get_gencfg_path())
            raise

    def run_check_gencfg_light_zero_calories(self):
        logging.info("Running gen-topology-check.sh")
        try:
            with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("gen_topology_check_sh")) as pl:
                sp.check_call(
                    'cd {gencfg_path} && TMP={tmp} TMPDIR={tmp} ./scripts/gen-topology-check.sh'.format(gencfg_path=self.get_gencfg_path(), tmp=self.get_tmp_path()),
                    shell=True, stdout=pl.stdout, stderr=pl.stderr
                )
        except:
            Arcadia.cleanup(self.get_gencfg_path())
            raise

    def run_install_sh(self):
        logging.info("Running install.sh")
        with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("install_sh")) as pl:
            sp.check_call(
                'cd {gencfg_path} && TMP={tmp} TMPDIR={tmp} ./install.sh'.format(gencfg_path=self.get_gencfg_path(), tmp=self.get_tmp_path()),
                shell=True, stdout=pl.stdout, stderr=pl.stderr
            )

    def run_show_power(self, group_name):
        logging.info("Running show_power.py")
        with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("show_power")) as pl:
            show_power_json = sp.check_output(
                '{}/utils/common/show_power.py -g {} -t json'.format(self.get_gencfg_path(), group_name),
                shell=True, stderr=pl.stderr
            )
        return json.loads(show_power_json)

    def is_gencfg_outdated(self):
        local_revision = Arcadia.info(os.path.join(self.get_gencfg_path(), 'db'))['commit_revision']
        global_revision = Arcadia.info('arcadia:/arc/trunk/arcadia/gencfg')['commit_revision']
        logging.info("Local revision = {}".format(local_revision))
        logging.info("Global revision = {}".format(global_revision))

        return local_revision != global_revision

    def subtask_recluster_in_all_dynamic(self, group, target_instances_count=None, target_instance_power_units=None, target_instance_memory=None):
        show_power = self.run_show_power(group)

        current_instances_count = show_power[group]['total']['instances_count']
        current_total_power_units = show_power[group]['total']['instances_cpu']
        current_instance_power_units = current_total_power_units / current_instances_count

        target_instance_power_units = target_instance_power_units or current_instance_power_units

        delta_instances = target_instances_count - current_instances_count

        delta_total_power = delta_instances * current_instance_power_units +\
            target_instances_count * (target_instance_power_units - current_instance_power_units)

        logging.info("Running recluster.py")
        recluster_command = "{}/optimizers/dynamic/recluster.py -y -g {}".format(
            self.get_gencfg_path(),
            group
        )

        if delta_instances or delta_total_power:
            if delta_instances:
                recluster_command += ' --instances {}'.format(delta_instances)
            recluster_command += ' --power {}'.format(delta_total_power)

        if target_instance_memory:
            recluster_command += ' --memory "{} GB"'.format(target_instance_memory)

        with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("dynamic_recluster")) as pl:
            sp.check_call(recluster_command, shell=True, stdout=pl.stdout, stderr=pl.stderr)

    def subtask_allocate_in_all_dynamic(self, all_groups_port=None, all_groups_same_port=False, tier_name=None, **other_alloc_form):
        prepared_alloc_params = deepcopy(other_alloc_form)

        locations = []
        for k in prepared_alloc_params.keys():
            m = re.match('allocate_in_(.*)', k)
            if m:
                location_name = m.group(1)
                if prepared_alloc_params[k]:
                    locations.append(location_name.upper())
                del prepared_alloc_params[k]
            elif k == "group":
                group = prepared_alloc_params[k]
                del prepared_alloc_params[k]
            elif k in ["owners", "watchers"]:
                prepared_alloc_params[k] = ','.join(prepared_alloc_params[k])
            elif isinstance(prepared_alloc_params[k], list):
                prepared_alloc_params[k] = json.dumps(prepared_alloc_params[k])

        target_group_port = None
        if all_groups_same_port and not all_groups_port:
            find_most_unused_port_command = "{gencfg_path}/utils/common/find_most_unused_port.py -j -r 1".format(
                gencfg_path=self.get_gencfg_path()
            )
            with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("dynamic_allocate")) as pl:
                find_most_unused_port_output = json.loads(
                    sp.check_output(find_most_unused_port_command, shell=True, stderr=pl.stderr)
                )
            target_group_port = find_most_unused_port_output['port']
        elif all_groups_port:
            target_group_port = all_groups_port

        created_groups = []
        for location in locations:
            loc_group = location + "_" + group.upper()

            allocate_command = """{gencfg_path}/optimizers/dynamic/main.py -y -m ALL_DYNAMIC -a add -g {loc_group} -d "{description}"
                                  --owners '{owners}' --watchers '{watchers}'
                                  --ctype '{ctype}' --itype '{itype}' --prj '{prj}' --metaprj '{metaprj}'
                                  --location '{location}'
                                  --min_replicas '{min_instances}' --max_replicas '{max_instances}'
                                  --memory '{memory}' --min_power '{min_power}' --equal_instances_power '{equal_instances_power}'
                                  --disk '{disk}' --ssd '{ssd}' --l3enabled '{l3enabled}'
                               """.format(gencfg_path=self.get_gencfg_path(), loc_group=loc_group, location=location, **prepared_alloc_params)

            allocate_command = self.fix_multiline_command(allocate_command)
            with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("dynamic_allocate")) as pl:
                sp.check_call(allocate_command, shell=True, stdout=pl.stdout, stderr=pl.stderr)

            created_groups.append(loc_group)

        if target_group_port:
            for group in created_groups:
                change_port_command = "{gencfg_path}/utils/common/change_port.py -g {group} -p {port} --port-step 1".format(
                    gencfg_path=self.get_gencfg_path(),
                    group=group,
                    port=target_group_port
                )
                with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("change_port")) as pl:
                    sp.check_call(change_port_command, shell=True, stdout=pl.stdout, stderr=pl.stderr)

        if tier_name:
            for group in created_groups:
                set_tier_name_command = "{gencfg_path}/utils/pregen/generate_trivial_intlookup.py -g {group} -t '{tier_name}'".format(
                    gencfg_path=self.get_gencfg_path(),
                    group=group,
                    tier_name=tier_name
                )
                with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("set_tier_name")) as pl:
                    sp.check_call(set_tier_name_command, shell=True, stdout=pl.stdout, stderr=pl.stderr)

    def subtask_create_group(self, owners, template_group, parent_group, group_type, **other_subtask_kwargs):
        extra_p_params = ",properties.{}=True".format(group_type) if group_type != 'group_with_resources' else ""

        command = u"""{gencfg_path}/utils/common/update_igroups.py
                      -a addgroup -g {group} -d '{group_description}'
                      -o {owners} -p 'reqs.instances.memory_guarantee={instance_memory},legacy.funcs.instancePower=exactly{instance_power}{extra_p_params}'
                      -t '{{"metaprj": "{metaprj}" }}'
                      --instance-port-func {instance_port_func}
                   """.format(gencfg_path=self.get_gencfg_path(),
                              owners=','.join(owners),
                              extra_p_params=extra_p_params,
                              **other_subtask_kwargs)

        command = self.fix_multiline_command(command)

        if template_group:
            command += " -e '{template_group}'"

        if parent_group:
            command += " -m '{parent_group}'"

        with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("update_igroups")) as pl:
            sp.check_call(command, shell=True, stdout=pl.stdout, stderr=pl.stderr)

    def subtask_update_resources_in_master_group(self, group, hosts_to_add, hosts_to_remove, **resources):
        self.run_update_icards(group, **resources)

        add_command = "{gencfg_path}/utils/common/update_igroups.py -a movehosts -g {group} -s {hosts}".format(
            gencfg_path=self.get_gencfg_path(),
            group=group,
            hosts=','.join(hosts_to_add)
        )

        remove_command = "{gencfg_path}/utils/common/update_igroups.py -a movehosts -g ALL_UNWORKING_BUSY -s {hosts}".format(
            gencfg_path=self.get_gencfg_path(),
            hosts=','.join(hosts_to_remove)
        )

        with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("update_igroups")) as pl:
            if hosts_to_add:
                sp.check_call(add_command, shell=True, stdout=pl.stdout, stderr=pl.stderr)
            if hosts_to_remove:
                sp.check_call(remove_command, shell=True, stdout=pl.stdout, stderr=pl.stderr)

    def run_update_icards(self, group, **fields):
        gb_format = "{} GB".format
        field_to_key_and_template = {
            'l3enabled': ('reqs.hosts.l3enabled', json.dumps),
            'owners': ('owners', json.dumps),
            'itype': ('tags.itype', json.dumps),
            'ctype': ('tags.ctype', json.dumps),
            'metaprj': ('tags.metaprj', json.dumps),
            'memory_guarantee': ('reqs.instances.memory_guarantee', gb_format),
            'memory_overcommit': ('reqs.instances.memory_overcommit', gb_format),
            'disk': ('reqs.instances.disk', gb_format),
            'ssd': ('reqs.instances.ssd', gb_format)
        }

        with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("update_icards")) as pl:
            for field_key, field_value in fields.iteritems():
                card_key, card_value_func = field_to_key_and_template[field_key]
                card_value = card_value_func(field_value)
                modify_command = "{gencfg_path}/utils/common/update_card.py -g {group} -k {card_key} -v '{card_value}' -y".format(
                    gencfg_path=self.get_gencfg_path(),
                    group=group,
                    card_key=card_key,
                    card_value=card_value
                )
                sp.check_call(modify_command, shell=True, stdout=pl.stdout, stderr=pl.stderr)

    def subtask_modify_group_card(self, group, **fields):
        self.run_update_icards(group, **fields)

    def run_subtask(self, subtask_callable, commit_message, gencfg_check='run_check_gencfg', *subtask_args, **subtask_kwargs):
        logging.info("Checking out gencfg")
        Arcadia.checkout('arcadia:/arc/trunk/arcadia/gencfg/', self.get_gencfg_path())
        while True:
            logging.info("Updating gencfg")
            Arcadia.update(self.get_gencfg_path())

            try:
                self.run_install_sh()
                # self.run_check_gencfg_if_not_checked_yet()
                getattr(self, gencfg_check)()
            except:
                logging.exception("Exception while checking fresh gencfg")
                continue

            if self.is_gencfg_outdated():
                logging.info("Gencfg is outdated after running initial gen.sh -- updating")
                continue

            subtask_callable(*subtask_args, **subtask_kwargs)

            if self.is_gencfg_outdated():
                logging.info("Gencfg is outdated after running the subtask. Reverting, updating and re-checking")
                Arcadia.cleanup(self.get_gencfg_path())
                Arcadia.revert(self.get_gencfg_path())
                continue

            getattr(self, gencfg_check)()

            if self.is_gencfg_outdated():
                logging.info("Gencfg is outdated after running post-recluster gencfg check. Reverting, updating and re-checking")
                Arcadia.cleanup(self.get_gencfg_path())
                Arcadia.revert(self.get_gencfg_path())
                continue

            logging.info("Committing to gencfg")
            if self.Parameters.dry_run:
                logging.warning('DRY RUN, not commiting!')
            else:
                with ssh.Key(self, "robot-gencfg", "ROBOT_GENCFG_SSH_KEY"):
                    commit_out = Arcadia.commit(
                        self.get_gencfg_path(),
                        commit_message,
                        user='robot-gencfg'
                    )

                    if not commit_out.strip():
                        raise Exception('Empty output from Arcadia.commit -- nothing to commit?')
                    logging.info("Arcadia.commit output: {}".format(commit_out))
            break

    def on_execute(self):
        task_params = json.loads(self.Parameters.json_params)

        author = task_params['author']
        subtask_name = task_params['subtask']
        subtask_params = task_params['subtask_params']

        if subtask_name == "recluster_in_all_dynamic":
            commit_message = u"Guarantees change for {} by {}@".format(subtask_params['group'], author)
            self.run_subtask(
                self.subtask_recluster_in_all_dynamic,
                commit_message,
                gencfg_check='run_check_gencfg_light_zero_calories',
                **subtask_params
            )
        elif subtask_name == "allocate_in_all_dynamic":
            user_commit_message = subtask_params['commit_message']
            del subtask_params['commit_message']

            commit_message = u"Allocated group {} by {}@ : {}".format(subtask_params['group'], author, user_commit_message)
            self.run_subtask(
                self.subtask_allocate_in_all_dynamic,
                commit_message,
                gencfg_check='run_check_gencfg_light_zero_calories',
                **subtask_params
            )
        elif subtask_name == "create_group":
            commit_message = u"Create group {} by {}@".format(subtask_params['group'], author)
            self.run_subtask(
                self.subtask_create_group,
                commit_message,
                gencfg_check='run_check_gencfg_light_zero_calories',
                **subtask_params)
        elif subtask_name == "update_resources_in_master_group":
            commit_message = u"Modify hosts list of master group {} by {}@".format(subtask_params['group'], author)
            self.run_subtask(
                self.subtask_update_resources_in_master_group,
                commit_message,
                gencfg_check='run_check_gencfg_light_zero_calories',
                **subtask_params
            )
        elif subtask_name == "modify_group_card":
            commit_message = u"Modify group card for group {} by {}@".format(subtask_params['group'], author)
            self.run_subtask(
                self.subtask_modify_group_card,
                commit_message,
                gencfg_check='run_check_gencfg_light_zero_calories',
                **subtask_params
            )
        else:
            raise Exception("Unknown subtask name '{}'".format(subtask_name))
