# -*- coding: utf-8 -*-
import traceback

import sandbox.sdk2 as sdk2
import sandbox.sdk2.parameters as parameters
import sandbox.common.errors as errors

import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

import sandbox.projects.gencfg.mongo as mongo
import sandbox.projects.gencfg.helpers as helpers
import sandbox.projects.gencfg.environment as environment


class BaseGencfgGuiRequest(sdk2.Task):
    """ Base class for make Gencfg requests """

    class Requirements(sdk2.Task.Requirements):
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 10 * 1024, None)
        client_tags = ctc.Tag.CUSTOM_GENCFG_BUILD

    class Context(sdk2.Task.Context):
        svn_diff = ''
        exceptions = []
        commit = -1
        gencfg_revision = -1
        retry_count = 0

    class Parameters(sdk2.Task.Parameters):
        with parameters.Group('Base params') as base_paarams:
            author = parameters.String('Request author')
            commit_message = parameters.String('Commit message', default='', required=False)

        with parameters.Group('Execution') as execution:
            dry_run = sdk2.parameters.Bool(
                'Dry run',
                description='Run the whole procedure, but do not commit results',
                default=True
            )
            arcadia_revision = sdk2.parameters.String(
                'Base revision',
                description='Allocate, check and commit against this revision',
                required=False
            )
            use_last_resources = sdk2.parameters.Bool(
                'Use last released resources',
                required=True,
                default=False
            )

    def on_execute(self):
        self.Context.retry_count += 1

        try:
            gencfg = environment.GencfgEnvironment(self, self.Parameters.arcadia_revision, self.ramdrive.path)
            self.run_prepare(gencfg, self.Parameters)
            self.Context.gencfg_revision = int(gencfg.info(gencfg.src_root)['commit_revision'])
            self.set_info('Task manipulates revision {}'.format(self.Context.gencfg_revision))
        except Exception as e:
            helpers.print_log('\n' + traceback.format_exc(limit=20))
            raise Exception('Task failed at prepare gencfg environment.\n[{}]: {}'.format(type(e).__name__, e))

        try:
            self.run_subtask_payload(gencfg, self.Parameters)
            if not self.save_svn_diff(gencfg):
                self.set_info('Nothing to commit. Group already has selected params.')
                return
        except Exception as e:
            helpers.print_log('\n' + traceback.format_exc(limit=20))
            raise errors.TaskFailure('[{}]: {}'.format(type(e).__name__, e))

        try:
            gencfg.update()
        except Exception as e:
            raise errors.TemporaryError('Conflict merge. Task will be restarted.')

        try:
            self.run_full_checks(gencfg, self.Parameters)
        except Exception as e:
            helpers.print_log('\n' + traceback.format_exc(limit=20))
            raise Exception('Error is test changes. {}: {}'.format(type(e).__name__, e))

        if self.Parameters.dry_run:
            return

        try:
            gencfg.update()
        except Exception as e:
            raise errors.TemporaryError('Conflict merge. Task will be restarted.')

        try:
            self.run_post_update_checks(gencfg, self.Parameters)
        except Exception as e:
            raise errors.TemporaryError('Conflicting changes found. Task will be restarted.')

        try:
            commit = self.commit(
                gencfg,
                self.get_commit_message(gencfg, self.Parameters),
                restart_on_fail=not bool(self.Parameters.arcadia_revision)
            )

            if not commit:
                return

            self.insert_commit(commit)
            self.Context.commit = int(commit)

            self.set_info(
                'Request committed: <a href="https://a.yandex-team.ru/arc/commit/{0}">{0}</a>'.format(commit),
                do_escape=False
            )

            self.run_populate_commit(gencfg, self.Parameters, commit)

            self.set_info('Commit populated successful')
        except Exception as e:
            helpers.print_log('\n' + traceback.format_exc(limit=20))
            raise

    def on_finish(self, prev_status, status):
        if status != ctt.Status.SUCCESS:
            self.save_exceptions()

    def on_break(self, prev_status, status):
        self.save_exceptions()

    def run_prepare(self, gencfg, params):
        gencfg.prepare()
        gencfg.install(params.use_last_resources)

    def run_full_checks(self, gencfg, params):
        gencfg.run_script('scripts/gen-topology-check-dynamic.sh')

    def run_post_update_checks(self, gencfg, params):
        # gencfg.run_process(['./utils/check/check_ipv4tunnels.py'], 'check_ipv4tunnels')
        pass

    def get_commit_message(self, gencfg, params):
        return u'[{}] {} (committed by {}@) from https://sandbox.yandex-team.ru/task/{}/view'.format(
            type(self).__name__, params.commit_message.encode('ascii', 'ignore'),
            params.author, self.id
        )

    def run_subtask_payload(self, gencfg, params):
        raise NotImplementedError('This method need implement in child class')

    def run_populate_commit(self, gencfg, params, commit):
        gencfg.update(commit)
        gencfg.run_script('utils/mongo/populate_gencfg_trunk.py')

    def save_svn_diff(self, gencfg):
        diff = gencfg.diff()
        self.Context.svn_diff = diff
        return diff

    def commit(self, gencfg, message, restart_on_fail=False):
        try:
            return gencfg.commit(message)
        except Exception as e:
            if restart_on_fail:
                raise errors.TemporaryError(str(e))
            raise

    def insert_commit(self, commit):
        commit = mongo.normalize_commit(commit)

        mongo.commits().update(
            {'commit': commit},
            {
                '$set': {
                    'skip': {'mongo': True, 'clickhouse': False, 'hbf': False},
                    'author': str(self.Parameters.author),
                    'task_id': self.id,
                    'interface': True
                },
            },
            upsert=True
        )

    def save_exceptions(self):
        helpers.print_errors_to_info(self, self.log_path())
        self.Context.exceptions = helpers.get_list_errors(self.log_path())
