# coding: utf8
import re
import os
import time
import json
# import bson
# import pymongo
import logging
import sandbox.common.errors as errors
import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm
# import sandbox.common.types.client as ctc
import sandbox.projects.gencfg.mongo as mongo
import sandbox.projects.gencfg.helpers as helpers
import sandbox.projects.gencfg.environment as environment

from sandbox import sdk2
from datetime import datetime
from sandbox.projects.common.nanny import nanny
from sandbox.common.errors import TaskFailure
from sandbox.projects.resource_types import (
    CONFIG_BUILD_LOGS, GENCFG_DB_CACHE, CONFIG_ARCHIVE, CONFIG_SEARCHERLOOKUP,
    CONFIG_GENERATOR_CONFIGS, CONFIG_GENERATOR_GENERATED
)
from sandbox.projects.gencfg.BuildConfigsViaApi import BuildConfigsViaApi


SVN_COMMIT_KEY = 'Key'


def function_logger(func):
    """
    Count time executing func and logging exceptions to context
    """

    def wrapper(self, *args, **kwargs):
        s = time.time()
        try:
            result = func(self, *args, **kwargs)
        except Exception as e:
            self.Context.broken_goals += '[{}] {}: {}\n'.format(func.__name__, type(e).__name__, e)
            raise
        self.print_info('[TIME] {} {}'.format(func.__name__, time.time() - s))
        return result
    return wrapper


class BuildConfigGenerator2(sdk2.Task):
    """ Run gen.sh and update db """

    class Requirements(sdk2.Task.Requirements):
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 10 * 1024, None)
        # client_tags = ctc.Tag.CUSTOM_GENCFG_BUILD

    class Parameters(sdk2.Task.Parameters):
        generator_tag = sdk2.parameters.String('Svn tag', default='trunk')
        use_separate_task_for_configs = sdk2.parameters.Bool('Use separate task to build configs', default=True)
        generate_all = sdk2.parameters.Bool('Run generator script', default=True)
        generate_diff_to_prev = sdk2.parameters.String('Generate diff to previous tag', default=False)
        dry_run = sdk2.parameters.Bool('Run without populate/import', default=True)
        retry_count = sdk2.parameters.Integer('Count retry after TEMPORARY', default=2)
        use_last_resources = sdk2.parameters.Bool('Use last released resources', default=False)

    class Context(sdk2.Task.Context):
        exceptions = []
        executions_count = 0

    # LOGGING

    @staticmethod
    def set_log(info):
        logging.info('[{}] USER_LOG: {}'.format(datetime.now().strftime('%H:%M:%S'), info))

    def print_info(self, info):
        self.set_info('{}'.format(info))
        self.set_log('{}'.format(info))

    # PATHS

    def get_trunk_path(self):
        return self.ramdrive.path / 'tags'

    # BACKGROUND QUEUE

    def add_background_task(self, raise_on_fail, task_func, fargs, fkwargs):
        self.background_tasks.append({
            'name': task_func.__name__,
            'process': task_func(*fargs, **fkwargs),
            'raise_on_fail': raise_on_fail
        })

    def on_execute(self):
        with self.memoize_stage.first_stage:
            if self.Parameters.use_separate_task_for_configs:
                task_id = self.get_build_configs_via_api_task().id
                raise sdk2.WaitTask([task_id], ctt.Status.Group.FINISH + ctt.Status.Group.BREAK, wait_all=True)

        with self.memoize_stage.second_stage:
            gencfg = environment.GencfgEnvironment(self, tag=self.Parameters.generator_tag,
                                                   base_dir=self.get_trunk_path())
            gencfg.install()
            gencfg.install_db_cache_resource()
            gencfg.precalc_caches()

            extras = ['--full'] if self.tag_is_major() else []

            self.run_populate_searcher_lookup(extras)
            self.run_populate_search_map(extras)
            self.run_populate_gencfg_trunk()
            self.run_populate_staff_cache(self.get_vault_data('GENCFG', 'gencfg_default_oauth'))

            self.wait_background_gencfg_tasks()

            if self.Parameters.use_separate_task_for_configs:
                for subtask in self.get_finished_subtasks():
                    resources = sdk2.Resource['CONFIGS_DIRECTORY'].find(task=subtask).limit(1)
                    for resource in resources:
                        resource_path = gencfg.resource_path(resource)
                        gencfg.extract(resource_path, self.src_root)

                self.build_generator(gencfg, create_resources=True, run_build=False)
            else:
                self.build_generator(gencfg)

    def on_finish(self, prev_status, status):
        pass

    def on_break(self, prev_status, status):
        pass

    def on_release(self, additional_parameters):
        release_status = additional_parameters.get('release_status')
        # release_subject = additional_parameters.get('release_subject')

        for resource in self.list_resources():
            if resource.type.name != 'TASK_LOGS':
                # channel.sandbox.set_resource_attribute(resource.id, 'tag', self.Parameters.generator_tag)
                pass

        # send released resources to Nanny and add info about them to task's info field
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)

        if 'nannyfile_id' in self.ctx:
            nanny_diff_resource = self.sync_resource(self.ctx['nannyfile_id'])

            with open(nanny_diff_resource) as fp:
                nanny_diff = json.load(fp)
                nanny.ReleaseToNannyTask.on_gencfg_release(self, nanny_diff, additional_parameters)

        # announce chats
        if 'announce_id' in self.ctx:
            announce_resource = self.sync_resource(self.ctx['announce_id'])

            with open(announce_resource) as fp:
                announce_content = fp.read()
                if len(announce_content) > 0:
                    gencfg = environment.GencfgEnvironment(self, tag=self.Parameters.generator_tag,
                                                           base_dir=self.get_trunk_path())
                    gencfg.install()
                    gencfg.install_db_cache_resource()
                    gencfg.precalc_caches()
                    gencfg_path = os.path.join(gencfg.src_root, 'release_gencfg_path')
                    cmd = ['./utils/common/manipulate_telegram.py', '-a', 'announce', '-m', announce_content]
                    gencfg.run_process(cmd, work_dir=gencfg_path, log_name='manipulate_telegram')

        # leave last 4 major tags in mongo
        # self.remove_old_tags(4) temporarily disabled

        mongo.mark_tag_released(self.Parameters.generator_tag, self.ctx[SVN_COMMIT_KEY])
        mongo.mark_searcher_lookup(self.ctx[SVN_COMMIT_KEY], self.tag_is_major(), mongo.get_last_full_commit())
        self.mark_released_resources(release_status)

        # notify tasks and users about tag being released
        # self.report_mail_on_release(release_status, release_subject)
        # self.report_startrek_on_release(self.Parameters.generator_tag)

    def get_build_configs_via_api_task(self):
        BuildConfigsViaApi = sdk2.Task["BUILD_CONFIGS_VIA_API"]
        task = BuildConfigsViaApi(
            BuildConfigsViaApi.current,
            description="Build configs via api",
            owner=self.owner,
            priority=self.priority
        ).enqueue()
        return task

    def build_generator(self, gencfg, create_resources=True, run_only_checks=False, run_build=True):
        if run_build:
            self.gen_sh('run_check' if run_only_checks else None)

        if create_resources:
            for ResType, res_file in (
                (CONFIG_ARCHIVE, 'generator.configs.7z'),
                (CONFIG_SEARCHERLOOKUP, 'searcherlookup.conf'),
                (CONFIG_GENERATOR_CONFIGS, 'generator.configs.7z'),
                (CONFIG_GENERATOR_GENERATED, 'generator.generated.tar.gz')
            ):
                gencfg.create_resource(ResType, os.path.join(self.src_root, res_file), res_file)

            gencfg.run_process(['bash', './scripts/gen-configs-basesearch.sh', 'clean'], 'clean_configs-basesearch')
            gencfg.run_process(['bash', './scripts/gen-configs-web.sh', 'clean'], 'clean_configs-web')

    @function_logger
    def gen_sh(self, gencfg, option):
        """
        Run ./gen.sh run_checks for check gencfg
        """
        try:
            gencfg.gen_sh(option)
        except Exception as e:
            self.save_exceptions(os.path.join(gencfg.src_root, 'build'))
            raise TaskFailure('[{}] {}'.format(type(e).__name__, e))
        finally:
            gencfg.create_resource(CONFIG_BUILD_LOGS, os.path.join(gencfg.src_root, 'build'), 'build')
        gencfg.create_resource(GENCFG_DB_CACHE, os.path.join(gencfg.db_root, 'cache'), 'generator_cache')

    @helpers.disable_in_dry_run
    def run_populate_searcher_lookup(self, extras):
        self.add_background_task(
            raise_on_fail=True,
            task_func=self.gencfg.populate_searcher_lookup,
            fargs=(),
            fkwargs={'extras': extras, 'background': True}
        )

    @helpers.disable_in_dry_run
    def run_populate_search_map(self, extras):
        self.add_background_task(
            raise_on_fail=True,
            task_func=self.gencfg.populate_search_map,
            fargs=(),
            fkwargs={'extras': extras, 'background': True}
        )

    @helpers.disable_in_dry_run
    def run_populate_gencfg_trunk(self):
        self.add_background_task(
            raise_on_fail=True,
            task_func=self.gencfg.populate_gencfg_trunk,
            fargs=(),
            fkwargs={'background': True}
        )

    @helpers.disable_in_dry_run
    def run_populate_staff_cache(self, oauth_token):
        self.add_background_task(
            raise_on_fail=True,
            task_func=self.gencfg.populate_staff_cache,
            fargs=(),
            fkwargs={'oauth_token': oauth_token, 'background': True}
        )

    @function_logger
    def wait_background_gencfg_tasks(self):
        """
        Waits for tasks to complete and validates their code
        """
        for task in self.background_tasks:
            exitcode = task['process'].wait()
            if exitcode and task['raise_on_fail']:
                self.Context.exceptions = []
                self.save_exceptions()

                if self.Context.executions_count >= self.Parameters.retry_count - 1:
                    raise Exception('Task `{}` finished with exit code {}'.format(task['name'], exitcode))
                raise errors.TemporaryError('Task `{}` finished with exit code {}'.format(task['name'], exitcode))

    def tag_is_major(self):
        return re.match('^stable-\d+-r1$', self.Parameters.generator_tag) is not None

    def get_finished_subtasks(self):
        subtasks = self.find(BuildConfigsViaApi, status=ctt.Status.Group.FINISH + ctt.Status.Group.BREAK)
        for subtask in subtasks:
            yield subtask

    def save_exceptions(self, log_path=None):
        helpers.print_errors_to_info(self, log_path or self.log_path())
        self.Context.exceptions = helpers.get_list_errors(log_path or self.log_path())
