# -*- coding: utf-8 -*-
import sandbox.sdk2 as sdk2
import sandbox.sdk2.parameters as parameters
import sandbox.common.errors as errors

import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

import sandbox.projects.gencfg.mongo as mongo
import sandbox.projects.gencfg.helpers as helpers
# import sandbox.projects.gencfg.semaphores as semaphores
import sandbox.projects.gencfg.environment as environment


class CreateMacros(sdk2.Task):
    """ Create macros """

    class Requirements(sdk2.Task.Requirements):
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 10 * 1024, None)
        ram = 80 * 1024
        cores = 16
        client_tags = ctc.Tag.CUSTOM_GENCFG_BUILD
        # semaphores = semaphores.gencfg.dynamic

    class Context(sdk2.Task.Context):
        svn_diff = ''
        exceptions = []
        commit = -1
        gencfg_revision = -1
        retry_count = 0

    class Parameters(sdk2.Task.Parameters):
        author = parameters.String('Request author', default='', required=True)
        macros_name = parameters.String('Macros name', default='', required=True)
        parent_macros = parameters.String('Parent macros', default='', required=False)
        macros_descr = parameters.String('Macros purpose and other human-readable description')
        owners = parameters.List('List of owners')
        commit_message = parameters.String('Commit message', default='', required=False)

        dry_run = sdk2.parameters.Bool(
            'Dry run',
            description='Run the whole procedure, but do not commit results',
            default=True
        )

        arcadia_revision = sdk2.parameters.String(
            'Base revision',
            description='Allocate, check and commit against this revision',
            required=False
        )

        use_last_resources = sdk2.parameters.Bool(
            'Use last released resources',
            required=True,
            default=False
        )

    def on_execute(self):
        self.Context.retry_count += 1

        try:
            gencfg = environment.GencfgEnvironment(self, self.Parameters.arcadia_revision, self.ramdrive.path)
            gencfg.prepare()
            gencfg.install(self.Parameters.use_last_resources)

            gencfg_revision = int(gencfg.info(gencfg.src_root)['commit_revision'])
            self.set_info('Task manipulates revision {}'.format(gencfg_revision))
            self.Context.gencfg_revision = gencfg_revision

            try:
                self.create_macros(gencfg, self.Parameters)
                if not self.save_svn_diff(gencfg):
                    self.set_info('Nothing to commit.')
                    return
            except Exception as e:
                raise errors.TaskFailure('[{}]: {}'.format(type(e).__name__, e))

            gencfg.run_script('scripts/gen-topology-check-dynamic.sh')

            if not self.Parameters.dry_run:
                commit = self.commit(
                    gencfg,
                    u'[CreateMacros] {}: {} (committed by {}@) '
                    'from https://sandbox.yandex-team.ru/task/{}/view'.format(
                        self.Parameters.macros_name, self.Parameters.commit_message.encode('ascii', 'ignore'),
                        self.Parameters.author, self.id
                    ),
                    restart_on_fail=not bool(self.Parameters.arcadia_revision)
                )

                if not commit:
                    return

                self.Context.commit = int(commit)
                self.set_info(
                    'Request committed: <a href="https://a.yandex-team.ru/arc/commit/{0}">{0}</a>'.format(commit),
                    do_escape=False
                )

                self.insert_commit(commit)

                gencfg.update(commit)
                gencfg.run_script('utils/mongo/populate_gencfg_trunk.py')
        except (errors.TaskFailure, errors.TemporaryError):
            raise
        except Exception as e:
            if self.Context.retry_count > 3:
                raise
            raise errors.TemporaryError('{}: {}'.format(type(e).__name__, e))

    def on_finish(self, prev_status, status):
        if status != ctt.Status.SUCCESS:
            self.save_exceptions()

    def on_break(self, prev_status, status):
        self.save_exceptions()

    def create_macros(self, gencfg, params):
        def format_list(lst):
            return ','.join(str(obj) for obj in lst)

        remove_command = [
            './utils/common/manipulate_hbf.py',
            '-a', 'addmacros',
            '-m', '_GENCFG_{}_'.format(params.macros_name),
            '-d', params.macros_descr,
            '-o', format_list(params.owners)
        ]

        gencfg.run_process(remove_command, 'create_macros')

        if params.parent_macros != '':
            modify_macros_command = [
                './utils/common/manipulate_hbf.py',
                '-a', 'updatemacros',
                '-m', '_GENCFG_{}_'.format(params.macros_name),
                '-p', params.parent_macros
            ]

            gencfg.run_process(modify_macros_command, 'update_macros')

    def save_svn_diff(self, gencfg):
        diff = gencfg.diff()
        self.Context.svn_diff = diff
        return diff

    def commit(self, gencfg, message, restart_on_fail=False):
        try:
            return gencfg.commit(message)
        except Exception as e:
            if restart_on_fail:
                raise errors.TemporaryError(str(e))
            raise

    def insert_commit(self, commit):
        commit = mongo.normalize_commit(commit)

        mongo.commits().update(
            {'commit': commit},
            {
                '$set': {
                    'skip': {'mongo': True, 'clickhouse': False, 'hbf': False},
                    'author': str(self.Parameters.author),
                    'task_id': self.id,
                    'interface': True
                },
            },
            upsert=True
        )

    def save_exceptions(self):
        helpers.print_errors_to_info(self, self.log_path())
        self.Context.exceptions = helpers.get_list_errors(self.log_path())
