# coding: utf8

import os
import time
import logging
import requests

from datetime import datetime

from sandbox import sdk2
from sandbox.projects.common.gencfg import solomon


class GencfgApiSpeedTest(sdk2.Task):
    """ Speed Test for Gencfg API """

    GENCFG_API = 'http://api.gencfg.yandex-team.ru'

    class Context(sdk2.Task.Context):
        dump_test_json = ''

    class Parameters(sdk2.Task.Parameters):
        request_count = sdk2.parameters.Integer('Count request (per run)', required=True, default=3)
        group_list = sdk2.parameters.List(
            'List groups for test', default=['MAN_HOST_CPU_METRICS', 'SAS_HOST_CPU_METRICS', 'VLA_HOST_CPU_METRICS']
        )

    # LOGGING

    @staticmethod
    def set_log(info):
        logging.info('[{}] USER_LOG: {}'.format(
            datetime.now().strftime('%H:%M:%S'),
            info
        ))

    def writeln(self, info):
        self.set_info('{}'.format(info))
        self.set_log('{}'.format(info))

    # FUNCTIONS

    def make_url(self, url, **kwargs):
        full_url = '{}/{}'.format(self.GENCFG_API, url)
        for key, value in kwargs.iteritems():
            sep = '&'
            if '?' not in full_url:
                sep = '?'
            full_url += '{}{}={}'.format(sep, key, value)
        return full_url

    def http_get(self, url):
        try:
            start = time.time()
            response = requests.get(url, timeout=30)
            end = time.time()
            return datetime.now(), response.status_code, end - start
        except Exception:
            return datetime.now(), 0, 0

    def group(self, group_name):
        return self.http_get(self.make_url('trunk/groups/{}'.format(group_name)))

    def group_card(self, group_name):
        return self.http_get(self.make_url('trunk/groups/{}/card'.format(group_name)))

    def group_instances(self, group_name):
        return self.http_get(self.make_url('trunk/groups/{}/instances'.format(group_name)))

    def on_execute(self):
        os.environ['SOLOMON_TOKEN'] = sdk2.Vault.data('GENCFG', 'gencfg_default_oauth')

        results = {}
        for _ in range(self.Parameters.request_count):
            for group_name in self.Parameters.group_list:
                if group_name not in results:
                    results[group_name] = {}

                for func in [self.group, self.group_card, self.group_instances]:
                    if func.__name__ not in results[group_name]:
                        results[group_name][func.__name__] = []
                    results[group_name][func.__name__].append(func(group_name))

        # self.Context.dump_test_json = json.dumps(results, indent=4)

        client = solomon.SolomonClient(project='gencfg', cluster='monitoring', service='health')
        for group in results:
            for func in results[group]:
                for date_time, code, speed in results[group][func]:
                    client.add_sensor('gencfg_api_speed_test_{}'.format(func), speed, date_time)
                if len(client.sensors_collected) > 100:
                    client.flush()
        client.flush()
