# coding: utf8
import datetime
import json

from sandbox import sdk2
from sandbox.sandboxsdk import environments

import sandbox.common.types.misc as ctm
import sandbox.projects.gencfg.environment as gencfgenv
import sandbox.projects.gencfg.mongo as gencfg_mongo


class GencfgChartsUploadGroupsInfoToYt(sdk2.Task):
    """ Upload GenCfg groups info to YT """

    class Context(sdk2.Task.Context):
        gencfg_revision = ''

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('GenCfg') as gencfg:
            arcadia_revision = sdk2.parameters.String(
                'Base revision',
                description='Load mapping from with revision',
                required=False
            )
            use_last_resources = sdk2.parameters.Bool(
                'Use last released resources',
                required=True,
                default=False
            )
            exclude_abc_services = sdk2.parameters.List(
                'Exclude ABC services',
                default=[]
            )
            exclude_master_groups = sdk2.parameters.List(
                'Exclude master groups',
                default=[]
            )
            exclude_itypes = sdk2.parameters.List(
                'Exclude itypes',
                default=[]
            )
            exclude_slave_groups = sdk2.parameters.List(
                'Exclude slave groups',
                default=[]
            )

        with sdk2.parameters.Group('YT') as yt:
            yt_cluster = sdk2.parameters.String(
                'YT Cluster',
                default='arnold',
                required=True
            )
            yt_prefix = sdk2.parameters.String(
                'YT Prefix',
                default='//home/robot-dev/shotinleg/solomon',
                required=True
            )
            groups_info_table_name = sdk2.parameters.String(
                'Groups info table name',
                default='mapping',
                required=True
            )
            history_groups_info_table_name = sdk2.parameters.String(
                'History groups info table name',
                default='mapping_history',
                required=True
            )

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
        )
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 10 * 1024, None)

    def get_trunk_path(self):
        return self.ramdrive.path / 'trunk'

    def get_tag_path(self):
        return self.ramdrive.path / 'tag'

    def on_execute(self):
        import yt.wrapper as yt

        self.yt = yt
        self.yt_client = self.yt.YtClient(self.Parameters.yt_cluster, self.get_oauth_token())
        self.yt_client.config['pickling']['python_binary'] = '/skynet/python/bin/python'
        self.yt_client.config["pickling"]["module_filter"] = lambda module: "hashlib" not in getattr(module, "__name__", "") and getattr(module, "__name__", "") != "hmac"

        groups_info = self.get_gencfg_groups_info(self.get_trunk_path())
        self.upload_gencfg_groups_info_to_yt(groups_info)

        tag_info = self.find_last_tag()
        history_groups_info = self.get_gencfg_groups_info(self.get_tag_path(), tag_info["tag"])
        history_groups_info = self.convert_to_history_format(tag_info, history_groups_info)
        self.upload_history_gencfg_groups_info_to_yt(history_groups_info)

    def get_gencfg_groups_info(self, gencfg_path, tag=None):
        gencfg = gencfgenv.GencfgEnvironment(self, self.Parameters.arcadia_revision, gencfg_path, tag=tag)
        gencfg.prepare()
        gencfg.install(self.Parameters.use_last_resources)

        gencfg_revision = int(gencfg.info(gencfg.src_root)['commit_revision'])
        self.set_info('Task manipulates gencfg revision {}'.format(gencfg_revision))
        self.Context.gencfg_revision = gencfg_revision

        group_to_aggregate_keys_mapping = json.loads(gencfg.get_group_to_aggregate_keys_mapping())

        aggregation_data = []
        for slave_group, aggregation_keys in group_to_aggregate_keys_mapping.items():
            abc_service = aggregation_keys['abc_service']
            master_group = aggregation_keys['master_group']
            itype = aggregation_keys['itype']

            if abc_service not in self.Parameters.exclude_abc_services and \
                    master_group not in self.Parameters.exclude_master_groups and \
                    itype not in self.Parameters.exclude_itypes and \
                    slave_group not in self.Parameters.exclude_slave_groups:
                aggregation_data.append({
                    'slave_group': slave_group,
                    'abc': abc_service,
                    'master_group': master_group,
                    'itype': itype,
                })
        return aggregation_data

    def find_last_tag(self):
        tags_db = gencfg_mongo.get_collection("topology_commits", "tags")

        tags_info = [record for record in tags_db.find()]
        for tag_info in tags_info:
            tag_info.update({"creation_time": tag_info["_id"].generation_time})
        tags_info = sorted(tags_info, key=lambda x: x["creation_time"])
        return tags_info[-1]

    def convert_to_history_format(self, tag_info, groups_info):
        data = []
        for item in groups_info:
            data.append({
                'group': item['slave_group'],
                'abc_service': item['abc'],
                'itype': item['itype'],
                'tag': tag_info['tag'],
                'creation_time': tag_info["creation_time"].strftime("%Y-%m-%dT%H:%M")
            })
        return data

    def upload_history_gencfg_groups_info_to_yt(self, groups_info):
        yt_mapping_table = '{}/{}'.format(self.Parameters.yt_prefix, self.Parameters.history_groups_info_table_name)

        if not self.yt_client.exists(yt_mapping_table):
            self.set_info('Create table `{}`'.format(yt_mapping_table))

            self.yt_client.create('table', yt_mapping_table, attributes={
                'schema': [
                    {'name': 'tag', 'type': 'string'},
                    {'name': 'creation_time', 'type': 'string'},
                    {'name': 'group', 'type': 'string'},
                    {'name': 'abc_service', 'type': 'string'},
                    {'name': 'itype', 'type': 'string'},
                ],
            })

        yt_mapping_table = self.yt.TablePath(yt_mapping_table, append=True)
        self.yt_client.write_table(yt_mapping_table, groups_info)

        return yt_mapping_table

    def upload_gencfg_groups_info_to_yt(self, groups_info):
        yt_mapping_table = '{}/{}'.format(self.Parameters.yt_prefix, self.Parameters.groups_info_table_name)
        self.remove_table_if_needed(yt_mapping_table)

        self.set_info('Create table `{}`'.format(yt_mapping_table))
        self.yt_client.create('table', yt_mapping_table, attributes={
            'schema': [
                {'name': 'slave_group', 'type': 'string'},
                {'name': 'abc', 'type': 'string'},
                {'name': 'master_group', 'type': 'string'},
                {'name': 'itype', 'type': 'string'},
            ],
        })

        self.yt_client.write_table(yt_mapping_table, groups_info)

        return yt_mapping_table

    def remove_table_if_needed(self, yt_table):
        if self.yt_client.exists(yt_table):
            self.yt_client.remove(yt_table)

    def get_oauth_token(self):
        return sdk2.Vault.data(self.owner, 'yt-token-robot-gencfg2')
