# coding: utf8
import json
import time
import pymongo
import logging
import requests
import sandbox.common.errors as errors
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
import sandbox.common.types.notification as ctn
import sandbox.projects.gencfg.semaphores as semaphores

from sandbox import sdk2
from functools import wraps
from datetime import datetime, timedelta
from sandbox.projects.gencfg.AllocateInDynamic import AllocateInDynamic
from sandbox.projects.gencfg.ReallocateInDynamic import ReallocateInDynamic
from sandbox.projects.gencfg.ModifyGroupCard import ModifyGroupCard
from sandbox.projects.gencfg.RemoveGroup import RemoveGroup
from sandbox.projects.gencfg.CreateMacros import CreateMacros
from sandbox.projects.gencfg.RemoveMacros import RemoveMacros
from sandbox.projects.gencfg.ModifyMacros import ModifyMacros
from sandbox.projects.gencfg.ManipulateTypes import ManipulateTypes


def _get_datetime_from_timestamp(timestamp):
    result = ''
    for fmt in ('%Y-%m-%dT%H:%M:%S.%ZZ', '%Y-%m-%dT%H:%M:%S.%zZ', '%Y-%m-%dT%H:%M:%S.%fZ', '%Y-%m-%dT%H:%M:%SZ'):
        try:
            result = datetime.strptime(timestamp, fmt)
            result = result + timedelta(hours=3)
            break
        except Exception:
            pass

    return result


def retry(tries=1, delay=1):
    def deco_retry(func):
        @wraps(func)
        def func_retry(*args, **kwargs):
            my_tries, my_delay = tries, delay
            while my_tries > -1:
                try:
                    return func(*args, **kwargs)
                except Exception as err:
                    msg = '%s - %s. retrying in %d seconds' % (func.func_name,
                                                               str(err),
                                                               my_delay)
                    my_tries -= 1
                    if my_tries == -1:
                        return False
                    logging.debug(msg)
                    time.sleep(my_delay)
        return func_retry
    return deco_retry


class GencfgGuiBackendTask(sdk2.Task):
    """ Check updates and run tasks """
    ASYNC_TASKS = ('modify_group_card',)

    MONGO_SETTINGS = {
        'uri': ','.join([
            'myt0-4012.search.yandex.net:27017',
            'myt0-4019.search.yandex.net:27017',
            'sas1-6063.search.yandex.net:27017',
            'sas1-6136.search.yandex.net:27017',
            'vla1-3984.search.yandex.net:27017',
        ]),
        'replicaset': 'heartbeat_mongodb_c',
        'read_preference': pymongo.ReadPreference.PRIMARY
    }

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.CUSTOM_GENCFG_BUILD
        semaphore = semaphores.gencfg.dynamic

    class Context(sdk2.Task.Context):
        child_tasks = []

    class Parameters(sdk2.Task.Parameters):
        mongo_db = sdk2.parameters.String(
            'Mongo DB',
            required=True,
            default='shotinleg_ggdb'
        )
        mongo_collection = sdk2.parameters.String(
            'Mongo Collection',
            required=True,
            default='background_tasks'
        )
        retry_count = sdk2.parameters.Integer(
            'Count retry request (on exception)',
            required=False,
            default=3
        )
        dry_run = sdk2.parameters.Bool(
            'Dry Run',
            required=True,
            default=True
        )
        use_last_resources = sdk2.parameters.Bool(
            'Use last released resources',
            required=True,
            default=False
        )

    # LOGGING

    @staticmethod
    def set_log(info):
        logging.info('[{}] USER_LOG: {}'.format(
            datetime.now().strftime('%H:%M:%S'),
            info
        ))

    def writeln(self, info):
        self.set_info('{}'.format(info))
        self.set_log('{}'.format(info))

    # MONGO

    @staticmethod
    @retry(5, 1)
    def db_get_collection(mongo_db, mongo_collection):
        collection = pymongo.MongoReplicaSetClient(
            GencfgGuiBackendTask.MONGO_SETTINGS['uri'],
            connectTimeoutMS=5000,
            replicaSet=GencfgGuiBackendTask.MONGO_SETTINGS['replicaset'],
            w='majority',
            wtimeout=15000,
            read_preference=GencfgGuiBackendTask.MONGO_SETTINGS['read_preference']
        )[mongo_db][mongo_collection]
        return collection

    @retry(5, 1)
    def db_find_one(self, query, collection=None):
        if not collection:
            collection = self.collection
        request = collection.find(query).sort(
            '$natural', pymongo.ASCENDING
        ).limit(1)
        request = request[0] if request.count() else None
        return request

    @retry(5, 1)
    def db_find_all(self, query, collection=None):
        if not collection:
            collection = self.collection
        request = collection.find(query).sort(
            'added', pymongo.ASCENDING
        )
        request = [request[i] for i in range(request.count())] if request.count() else None
        return request

    @retry(5, 1)
    def db_update_one(self, query, updates, collection=None):
        if not collection:
            collection = self.collection
        collection.update_one(
            query,
            {"$set": updates}
        )

    def on_execute(self):
        self.writeln('TASK STARTED')
        self.mongo_db = self.Parameters.mongo_db
        self.mongo_collection = self.Parameters.mongo_collection
        self.collection = self.db_get_collection(
            self.mongo_db, self.mongo_collection
        )

        # Update child task status after restart parent(this) task
        self.update_subtasks()

        # Finding request for updating gencfg_db
        self.writeln('Find request in Mongo DB ({}/{})'.format(
            self.mongo_db, self.mongo_collection
        ))

        find_patterns = [
            {'status': 'executing'},
            {'status': 'enqueueing'},
            {'status': 'enqueued'}
        ]

        sync_tasks = 0
        for i, pattern in enumerate(find_patterns):
            new_requests = self.db_find_all(pattern)

            if not new_requests:
                self.set_info('{}: NOT FOUND'.format(pattern))
                continue

            for request in new_requests:
                if request['sandbox_task_data']['subtask'] not in self.ASYNC_TASKS and sync_tasks > 0:
                    self.set_info('{} - SKIP'.format(request['sandbox_task_data']['subtask']))
                    continue

                subtask = self.get_subtask(
                    request['_id'],
                    request['sandbox_task_data']['author'],
                    request['sandbox_task_data']['subtask'],
                    request['sandbox_task_data']['subtask_params']
                )
                sync_tasks += 1
                self.Context.child_tasks.append(subtask.id)

                subtask.enqueue()

                self.db_update_one({'_id': request['_id']}, {
                    'status': 'executing',
                    'enqueued': self.get_enqueued_time(subtask),
                    'sandbox_task_id': str(subtask.id)
                })

                self.set_info('CREATED {} USING {}: {}'.format(
                    subtask.id, request['sandbox_task_data']['subtask'], request['sandbox_task_data']['subtask_params']
                ))

        if not self.Context.child_tasks:
            self.writeln('INFO: No requests. Exiting.')
            return

        self.set_log('Created tasks: {}'.format(self.Context.child_tasks))

        # Waiting for child task finished
        raise sdk2.WaitTask(
            self.Context.child_tasks,
            ctt.Status.Group.FINISH + ctt.Status.Group.BREAK,
            wait_all=True
        )

    def get_subtask(self, subtask_id, subtask_author, subtask_name, subtask_params):
        if subtask_name == 'allocate_in_all_dynamic':
            subtask = AllocateInDynamic(
                self,
                description=u'{} created by {}'.format(subtask_params['description'], subtask_author)
            )

            # Fill params
            subtask.Parameters.author = subtask_author
            subtask.Parameters.group_name = subtask_params['group']
            subtask.Parameters.group_descr = subtask_params['description']
            subtask.Parameters.owners = subtask_params['owners']
            subtask.Parameters.watchers = subtask_params['watchers']

            subtask.Parameters.ctype = subtask_params['ctype']
            subtask.Parameters.itype = subtask_params['itype']
            subtask.Parameters.prj = subtask_params['prj'] or ['none']
            subtask.Parameters.metaprj = subtask_params['metaprj']
            subtask.Parameters.itags = ''
            subtask.Parameters.affinity_category = subtask_params.get('affinity_category')

            # Requirements
            subtask.Parameters.instance_count = subtask_params['instance_count']
            subtask.Parameters.cores_per_instance = subtask_params['cores_per_instance']
            subtask.Parameters.memory = subtask_params['memory']
            subtask.Parameters.disk = subtask_params['disk']
            subtask.Parameters.ssd = subtask_params['ssd']
            subtask.Parameters.net = subtask_params['net']

            # Locations
            locations = []
            if subtask_params['allocate_in_man']:
                locations.append('man')
            if subtask_params['allocate_in_vla']:
                locations.append('vla')
            if subtask_params['allocate_in_sas']:
                locations.append('sas')
            if subtask_params['allocate_in_msk_iva']:
                locations.append('msk_iva')
            if subtask_params['allocate_in_msk_myt']:
                locations.append('msk_myt')
            subtask.Parameters.locations = locations

            subtask.Parameters.volumes = json.loads(subtask_params.get('volumes', '[]'))
            subtask.Parameters.parent_macro = subtask_params.get('parent_macro', '')

            subtask.Parameters.commit_message = subtask_params.get('commit_message', '')
            subtask.Parameters.dry_run = self.Parameters.dry_run
            subtask.Parameters.arcadia_revision = ''
            subtask.Parameters.use_last_resources = self.Parameters.use_last_resources

            subtask.Parameters.notifications = [
                sdk2.Notification(
                    statuses=[ctt.Status.FAILURE, ctt.Status.SUCCESS],
                    recipients=[subtask_author],
                    transport=ctn.Transport.EMAIL,
                ),
            ]

            subtask.save()
            return subtask

        elif subtask_name == 'recluster_in_all_dynamic':
            subtask = ReallocateInDynamic(
                self,
                description="Created by {}".format(subtask_author)
            )

            # Fill params
            subtask.Parameters.author = subtask_author
            subtask.Parameters.group_names = \
                subtask_params['group_names'] if subtask_params.get('group_names') else [subtask_params['group']]
            subtask.Parameters.instances_count = subtask_params.get('target_instances_count')
            subtask.Parameters.instance_power_units = subtask_params.get('target_instance_power') * 40
            subtask.Parameters.instance_memory = subtask_params.get('target_instance_memory')
            subtask.Parameters.instance_disk = subtask_params.get('target_instance_disk')
            subtask.Parameters.instance_ssd = subtask_params.get('target_instance_ssd')
            subtask.Parameters.instance_net = subtask_params.get('target_instance_net')
            volumes = subtask_params.get('volumes')
            subtask.Parameters.volumes = json.loads(volumes) if volumes is not None else None
            subtask.Parameters.commit_message = subtask_params.get('commit_message', '')
            subtask.Parameters.dry_run = self.Parameters.dry_run
            subtask.Parameters.arcadia_revision = ''
            subtask.Parameters.use_last_resources = self.Parameters.use_last_resources

            subtask.Parameters.notifications = [
                sdk2.Notification(
                    statuses=[ctt.Status.FAILURE, ctt.Status.SUCCESS],
                    recipients=[subtask_author],
                    transport=ctn.Transport.EMAIL,
                ),
            ]

            subtask.save()
            return subtask

        elif subtask_name == 'modify_group_card':
            subtask = ModifyGroupCard(
                self,
                description="Created by {}".format(subtask_author)
            )

            # Fill params
            subtask.Parameters.author = subtask_author
            subtask.Parameters.group_names = \
                subtask_params['group_names'] if subtask_params.get('group_names') else [subtask_params['group']]
            subtask.Parameters.fields = {key.replace('!', '.'): value for key, value in subtask_params['changes'].items()}

            subtask.Parameters.commit_message = subtask_params.get('commit_message', '')
            subtask.Parameters.dry_run = self.Parameters.dry_run
            subtask.Parameters.arcadia_revision = ''
            subtask.Parameters.use_last_resources = self.Parameters.use_last_resources

            subtask.Parameters.notifications = [
                sdk2.Notification(
                    statuses=[ctt.Status.FAILURE, ctt.Status.SUCCESS],
                    recipients=[subtask_author],
                    transport=ctn.Transport.EMAIL,
                ),
            ]

            subtask.save()
            return subtask

        elif subtask_name == 'remove_group':
            subtask = RemoveGroup(
                self,
                description="Created by {}".format(subtask_author)
            )

            # Fill params
            subtask.Parameters.author = subtask_author
            subtask.Parameters.group_name = subtask_params['group']

            subtask.Parameters.commit_message = subtask_params.get('commit_message', '')
            subtask.Parameters.dry_run = self.Parameters.dry_run
            subtask.Parameters.arcadia_revision = ''
            subtask.Parameters.use_last_resources = self.Parameters.use_last_resources

            subtask.Parameters.notifications = [
                sdk2.Notification(
                    statuses=[ctt.Status.FAILURE, ctt.Status.SUCCESS],
                    recipients=[subtask_author],
                    transport=ctn.Transport.EMAIL,
                ),
            ]

            subtask.save()
            return subtask

        elif subtask_name == 'create_macros':
            subtask = CreateMacros(
                self,
                description="Created by {}".format(subtask_author)
            )

            # Fill params
            subtask.Parameters.author = subtask_author
            subtask.Parameters.macros_name = subtask_params['macros']
            subtask.Parameters.parent_macros = subtask_params.get('parent', '')
            subtask.Parameters.macros_descr = subtask_params['description']
            subtask.Parameters.owners = subtask_params['owners']

            subtask.Parameters.commit_message = subtask_params.get('commit_message', '')
            subtask.Parameters.dry_run = self.Parameters.dry_run
            subtask.Parameters.arcadia_revision = ''
            subtask.Parameters.use_last_resources = self.Parameters.use_last_resources

            subtask.Parameters.notifications = [
                sdk2.Notification(
                    statuses=[ctt.Status.FAILURE, ctt.Status.SUCCESS],
                    recipients=[subtask_author],
                    transport=ctn.Transport.EMAIL,
                ),
            ]

            subtask.save()
            return subtask

        elif subtask_name == 'remove_macros':
            subtask = RemoveMacros(
                self,
                description="Created by {}".format(subtask_author)
            )

            # Fill params
            subtask.Parameters.author = subtask_author
            subtask.Parameters.macros_name = subtask_params['macros']

            subtask.Parameters.commit_message = subtask_params.get('commit_message', '')
            subtask.Parameters.dry_run = self.Parameters.dry_run
            subtask.Parameters.arcadia_revision = ''
            subtask.Parameters.use_last_resources = self.Parameters.use_last_resources

            subtask.Parameters.notifications = [
                sdk2.Notification(
                    statuses=[ctt.Status.FAILURE, ctt.Status.SUCCESS],
                    recipients=[subtask_author],
                    transport=ctn.Transport.EMAIL,
                ),
            ]

            subtask.save()
            return subtask

        elif subtask_name == 'modify_macros':
            subtask = ModifyMacros(
                self,
                description="Created by {}".format(subtask_author)
            )

            # Fill params
            subtask.Parameters.author = subtask_author
            subtask.Parameters.macros_name = subtask_params['macros']
            subtask.Parameters.fields = {key.replace('!', '.'): value for key, value in subtask_params['changes'].items()}

            subtask.Parameters.commit_message = subtask_params.get('commit_message', '')
            subtask.Parameters.dry_run = self.Parameters.dry_run
            subtask.Parameters.arcadia_revision = ''
            subtask.Parameters.use_last_resources = self.Parameters.use_last_resources

            subtask.Parameters.notifications = [
                sdk2.Notification(
                    statuses=[ctt.Status.FAILURE, ctt.Status.SUCCESS],
                    recipients=[subtask_author],
                    transport=ctn.Transport.EMAIL,
                ),
            ]

            subtask.save()
            return subtask

        elif subtask_name == 'manipulate_types':
            subtask = ManipulateTypes(
                self,
                description="Created by {}".format(subtask_author)
            )

            # Fill params
            subtask.Parameters.author = subtask_author
            subtask.Parameters.type_class = subtask_params['type_class']
            subtask.Parameters.type_name = subtask_params['type_name']
            subtask.Parameters.action = subtask_params['action']
            subtask.Parameters.fields = subtask_params['fields']
            subtask.Parameters.commit_message = subtask_params.get('commit_message', '')
            subtask.Parameters.dry_run = self.Parameters.dry_run
            subtask.Parameters.arcadia_revision = ''
            subtask.Parameters.use_last_resources = self.Parameters.use_last_resources

            subtask.Parameters.notifications = [
                sdk2.Notification(
                    statuses=[ctt.Status.FAILURE, ctt.Status.SUCCESS],
                    recipients=[subtask_author],
                    transport=ctn.Transport.EMAIL,
                ),
            ]

            subtask.save()
            return subtask

        else:
            self.db_update_one({'_id': subtask_id}, {
                'status': 'skipping',
                'sandbox_task_id': ''
            })
            self.writeln('UNDEFINED SUBTASK: {}'.format(subtask_name))
            raise errors.TemporaryError('Undefined subtask: {}'.format(subtask_name))

    def update_subtasks(self):
        for TaskType in (AllocateInDynamic, ReallocateInDynamic, ModifyGroupCard, RemoveGroup, CreateMacros,
                         RemoveMacros, ModifyMacros, ManipulateTypes):
            subtasks = self.find(
                TaskType,
                status=ctt.Status.Group.FINISH + ctt.Status.Group.BREAK
            )
            for subtask in subtasks:
                if subtask.id in self.Context.child_tasks:
                    self.writeln('Updating task {} status'.format(subtask.id))
                    self.update_status(subtask)

    def update_status(self, subtask):
        if subtask.status == ctt.Status.SUCCESS:
            self.db_update_one({'sandbox_task_id': str(subtask.id)}, {
                'status': 'success',
                'started': self.get_start_time(subtask),
                'finished': self.get_finish_time(subtask),
                'commit': subtask.Context.commit
            })
        elif subtask.status == ctt.Status.FAILURE:
            self.db_update_one({'sandbox_task_id': str(subtask.id)}, {
                'status': 'failure',
                'started': self.get_start_time(subtask),
                'finished': self.get_finish_time(subtask),
                'exceptions': subtask.Context.exceptions
            })
        else:
            self.db_update_one({'sandbox_task_id': str(subtask.id)}, {
                'status': 'exception',
                'started': self.get_start_time(subtask),
                'finished': self.get_finish_time(subtask),
                'exceptions': subtask.Context.exceptions
            })
        self.Context.child_tasks.remove(subtask.id)

    @retry(tries=30, delay=3)
    def get_passed_commit(self):
        """
        Return number of last checked commit if it passed
        or raise Exception
        """
        r = requests.get('http://clusterstate.yandex-team.ru/gencfg/json')

        if r.status_code != 200:
            raise Exception('Cannot get last commit state')

        last_commit = {'commit': 0, 'test_passed': True}
        for commit in r.json()['commits']:
            if int(commit['commit']) > int(last_commit['commit']):
                last_commit['commit'] = commit['commit']
                last_commit['test_passed'] = commit.get('test_passed', None)

        if last_commit['test_passed'] is False:
            raise errors.TemporaryError('Last commit is not passed {}'.format(
                last_commit['commit']
            ))

        return last_commit['commit']

    def get_enqueued_time(self, _):
        try:
            return datetime.now()
        except Exception:
            return ''

    def get_start_time(self, task):
        try:
            data = requests.get('https://sandbox.yandex-team.ru/api/v1.0/task/{}'.format(task.id)).json()
            return _get_datetime_from_timestamp(data.get('execution', {}).get('started', None))
        except Exception:
            return ''

    def get_finish_time(self, task):
        try:
            data = requests.get('https://sandbox.yandex-team.ru/api/v1.0/task/{}'.format(task.id)).json()
            return _get_datetime_from_timestamp(data.get('execution', {}).get('finished', None))
        except Exception:
            return ''
