# coding: utf8
import json
import logging
import requests
import datetime
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
import sandbox.common.types.notification as ctn
import sandbox.projects.gencfg.mongo as mongo
import sandbox.projects.gencfg.helpers as helpers

from sandbox import sdk2
from sandbox.projects.gencfg.AllocateInDynamic import AllocateInDynamic
from sandbox.projects.gencfg.ReallocateInDynamic import ReallocateInDynamic
from sandbox.projects.gencfg.ModifyGroupCard import ModifyGroupCard
from sandbox.projects.gencfg.CreateMacros import CreateMacros
from sandbox.projects.gencfg.RemoveMacros import RemoveMacros
from sandbox.projects.gencfg.ModifyMacros import ModifyMacros
from sandbox.projects.gencfg.ManipulateTypes import ManipulateTypes
from sandbox.projects.gencfg.ManipulateHosts import ManipulateHosts
from sandbox.projects.gencfg.workflow_tasks.GencfgRemoveGroup import GencfgRemoveGroup


class AnyFromList(object):
    def __init__(self, *args):
        self.list_variants = args

    def __ne__(self, other):
        return other not in self.list_variants


class NotAnyFromList(object):
    def __init__(self, *args):
        self.list_variants = args

    def __ne__(self, other):
        return other in self.list_variants


class GencfgGuiRequestsScheduler(sdk2.Task):
    """ Find requests in mongo and start Sandbox tasks """

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.CUSTOM_GENCFG_BUILD

    class Parameters(sdk2.Task.Parameters):
        mongo_db = sdk2.parameters.String('Mongo DB', required=True, default='shotinleg_ggdb')
        mongo_collection = sdk2.parameters.String('Mongo Collection', required=True, default='gencfg_gui_requests')

        requests_types = sdk2.parameters.List('Requests types', required=True, default=[
            'allocate_in_all_dynamic', 'recluster_in_all_dynamic'
        ])
        async_requests_types = sdk2.parameters.List('Async requests types', required=True, default=[
            'modify_group_card', 'manipulate_types'
        ])
        skip_requests_types = sdk2.parameters.List('Skip requests types', default=[])

        full_limit = sdk2.parameters.Integer('Max count execution requests', required=True, default=8)
        error_requests_limit = sdk2.parameters.Integer('Max count error requests', required=True, default=3)
        sync_types_limit = sdk2.parameters.Integer('Max count SYNC requests', required=True, default=1)
        async_types_limits = sdk2.parameters.Integer('Max count ASYNC requests (same type)', required=True, default=3)

        dry_run = sdk2.parameters.Bool('Dry Run', required=True, default=True)
        use_last_resources = sdk2.parameters.Bool('Use last released resources', required=True, default=False)

    # LOGGING

    @staticmethod
    def set_log(info):
        logging.info('[{}] USER_LOG: {}'.format(datetime.datetime.now().strftime('%H:%M:%S'), info))

    def writeln(self, info):
        self.set_info('{}'.format(info))
        self.set_log('{}'.format(info))

    @helpers.retry(6, 10)
    def get_all_requests(self):
        background_tasks = mongo.get_collection(self.Parameters.mongo_db, self.Parameters.mongo_collection)
        subtasks = [
            {'type': x}
            for x in self.Parameters.requests_types + self.Parameters.async_requests_types
        ]
        all_requests = list(mongo.find_all(
            background_tasks,
            {'$or': subtasks},
            sort_key='time.added', sort_value=-1,
            limit=1000
        ))

        return list(reversed(all_requests))

    def select_requests_by_status(self, all_requests, status):
        return self.select_requests(all_requests, {'status': status})

    def select_requests_by_type(self, all_requests, request_type):
        return self.select_requests(all_requests, {'type': request_type})

    def check_requests_for_finish(self, executing_requests):
        executing_requests = executing_requests[:]
        for request in executing_requests:
            if request['status'] != 'executing' or not request.get('sb_task_id'):
                continue

            subtask = self.get_subtask(int(request['sb_task_id']))
            if subtask.status in ctt.Status.Group.FINISH + ctt.Status.Group.BREAK:
                status = 'exception'
                if subtask.status == ctt.Status.SUCCESS:
                    status = 'success'
                elif subtask.status == ctt.Status.FAILURE:
                    status = 'failure'

                # For local consistency
                request['status'] = status
                request['time']['started'] = self.get_started_time(subtask.id)
                request['time']['finished'] = self.get_finished_time(subtask.id)
                request['exceptions'] = subtask.Context.exceptions
                request['commit'] = subtask.Context.commit

                self.update_request_info(request, {
                    'status': request['status'],
                    'time.started': request['time']['started'],
                    'time.finished': request['time']['finished'],
                    'exceptions': request['exceptions'],
                    'commit': request['commit']
                })

        return executing_requests

    def filter_enqueued(self, all_requests, executing_requests):
        executing_requests_types = [x['type'] for x in executing_requests]
        enqueued_requests = [x for x in all_requests if x['type'] not in executing_requests_types]

        is_sync_requests_executing = any([x in self.Parameters.requests_types for x in executing_requests_types])
        if is_sync_requests_executing:
            # Remove all sync requests
            enqueued_requests = [x for x in enqueued_requests if x['type'] not in self.Parameters.requests_types]

        # Limits
        full_limit = self.Parameters.full_limit - len(executing_requests)  # Full limit less than 8 executing requests
        error_requests_limit = self.Parameters.error_requests_limit  # Limit of previous broken requests
        requests_types_limit = self.Parameters.sync_types_limit  # Limit for sync requests
        async_requests_types_limits = {
            x: self.Parameters.async_types_limits
            for x in self.Parameters.async_requests_types
        }  # Limits for async requests

        filtered_enqueued_requests = []
        for request in enqueued_requests:
            # Spec statuses control
            if request['status'] in ('skip',):
                self.writeln('SKIP REQUEST: {} becasue (status: {})'.format(request['id'], request['status']))
                continue

            if request['type'] in (self.Parameters.skip_requests_types or []):
                self.writeln('SKIP REQUEST: {} becasue (type: {})'.format(request['id'], request['type']))
                continue

            # Sync requests limit control
            if request['type'] in self.Parameters.requests_types and requests_types_limit <= 0:
                self.writeln('SKIP REQUEST: {} becasue (type: {}, limit: {})'.format(
                    request['id'], request['type'], requests_types_limit
                ))
                continue

            # Async requests limit control
            if request['type'] in async_requests_types_limits and async_requests_types_limits[request['type']] <= 0:
                self.writeln('SKIP REQUEST: {} becasue (type: {}, limit: {})'.format(
                    request['id'], request['type'], async_requests_types_limits[request['type']]
                ))
                continue

            # Shared limit control
            if error_requests_limit <= 0:
                self.writeln('SKIP REQUEST: {} becasue (status: {}, limit: {})'.format(
                    request['id'], request['status'], error_requests_limit
                ))
                continue

            # Shared limit control
            if full_limit <= 0:
                self.writeln('SKIP rest requests becasue (limit: {})'.format(full_limit))
                break

            filtered_enqueued_requests.append(request)

            # Decrement limits
            if request['type'] in self.Parameters.requests_types:
                requests_types_limit -= 1
            elif request['type'] in async_requests_types_limits:
                async_requests_types_limits[request['type']] -= 1

            if request['status'] in ('error',):
                error_requests_limit -= 1

            full_limit -= 1

        return filtered_enqueued_requests

    def on_execute(self):
        # Get not finished requests from Mongo
        not_finished_requests = self.select_requests_by_status(
            self.get_all_requests(), NotAnyFromList('success', 'failure', 'exception', 'skip')
        )

        # Update REAL finished request and remove it from list
        not_finished_requests = self.select_requests_by_status(
            self.check_requests_for_finish(not_finished_requests), NotAnyFromList('success', 'failure', 'exception')
        )

        self.writeln('NOT_FINISHED: {}'.format(
            ['{}: ({}, {})'.format(x['id'], x['type'], x['status']) for x in not_finished_requests]
        ))

        # Select NOW executing requests
        executing_requests = self.select_requests_by_status(not_finished_requests, 'executing')

        self.writeln('EXECUTING: {}'.format(
            ['{}: ({}, {})'.format(x['id'], x['type'], x['status']) for x in executing_requests]
        ))

        # Filter requests with collisions
        enqueued_requests = self.filter_enqueued(not_finished_requests, executing_requests)

        self.writeln('ENQUEUED: {}'.format(
            ['{}: ({}, {})'.format(x['id'], x['type'], x['status']) for x in enqueued_requests]
        ))

        # Start new subtasks
        for request in enqueued_requests:
            try:
                new_subtask = self.get_new_subtask(request)
                new_subtask.enqueue()

                # For local consistency
                request['status'] = 'executing'
                request['time']['enqueued'] = self.get_enqueued_time(new_subtask)
                request['sb_task_id'] = new_subtask.id

            except Exception as e:
                # For local consistency, if error in
                request['status'] = 'error'
                request['time']['enqueued'] = None
                request['sb_task_id'] = None

                self.writeln('ERROR: {} -> {}: {}'.format(request['id'], type(e).__name__, e))

            self.update_request_info(request, {
                'status': request['status'],
                'time.enqueued': request['time']['enqueued'],
                'sb_task_id': request['sb_task_id']
            })

    def select_requests(self, all_requests, filter_query):
        def get_field(path, request):
            keys = path.split('.')
            field = request
            for key in keys:
                field = field[key]
            return field

        filtered_requests = []
        for request in all_requests:
            invalid = False
            for path, value in filter_query.iteritems():
                if get_field(path, request) != value:
                    invalid = True
                    break

            if not invalid:
                filtered_requests.append(request)
        return filtered_requests

    def get_subtask(self, subtask_id):
        return sdk2.Task[subtask_id]

    def get_enqueued_time(self, _):
        try:
            return datetime.datetime.now()
        except Exception as e:
            self.writeln('EXCEPTION {}: {}'.format(type(e).__name__, e))

    @helpers.retry(6, 10)
    def api_sandbox_get(self, path):
        return requests.get('https://sandbox.yandex-team.ru/api/v1.0/{}'.format(path)).json()

    def get_started_time(self, task_id):
        try:
            data = self.api_sandbox_get('task/{}'.format(task_id))
            started = data.get('execution', {}).get('started', None)
            return helpers.get_datetime_from_timestamp(started)
        except Exception as e:
            self.writeln('EXCEPTION {}: {}'.format(type(e).__name__, e))

    def get_finished_time(self, task_id):
        try:
            data = self.api_sandbox_get('task/{}'.format(task_id))
            finished = data.get('execution', {}).get('finished', None)
            return helpers.get_datetime_from_timestamp(finished)
        except Exception as e:
            self.writeln('EXCEPTION {}: {}'.format(type(e).__name__, e))

    @helpers.retry(6, 10)
    def update_request_info(self, request, data):
        data['time.updated'] = datetime.datetime.now()
        background_tasks = mongo.get_collection(self.Parameters.mongo_db, self.Parameters.mongo_collection)
        mongo.update_one(background_tasks, {'id': request['id']}, data, upsert=False)
        self.writeln('UPDATED MONGO: {} -> ({})'.format(request['id'], data))

    def get_new_subtask(self, request):
        if request['type'] == 'allocate_in_all_dynamic':
            subtask = AllocateInDynamic(self, description=u'{} created by {}'.format(
                request['params'].get('description', ''),
                request['params'].get('author', '')
            ))

            # Fill params
            subtask.Parameters.author = request['params'].get('author', '')
            subtask.Parameters.group_name = request['params']['group']
            subtask.Parameters.group_descr = request['params']['description']
            subtask.Parameters.owners = request['params']['owners']
            subtask.Parameters.watchers = request['params']['watchers']

            # Tags
            subtask.Parameters.ctype = request['params']['ctype']
            subtask.Parameters.itype = request['params']['itype']
            subtask.Parameters.prj = request['params']['prj'] or ['none']
            subtask.Parameters.metaprj = request['params']['metaprj']
            subtask.Parameters.itags = ''
            subtask.Parameters.dispenser_project_key = request['params'].get('dispenser_project_key', None)

            # Requirements
            subtask.Parameters.instance_count = request['params']['instance_count']
            subtask.Parameters.cores_per_instance = request['params']['cores_per_instance']
            subtask.Parameters.memory = request['params']['memory']
            subtask.Parameters.disk = request['params']['disk']
            subtask.Parameters.ssd = request['params']['ssd']
            subtask.Parameters.net = request['params']['net']

            # Locations
            locations = []
            if request['params']['allocate_in_man']:
                locations.append('man')
            if request['params']['allocate_in_vla']:
                locations.append('vla')
            if request['params']['allocate_in_sas']:
                locations.append('sas')
            if request['params']['allocate_in_msk_iva']:
                locations.append('msk_iva')
            if request['params']['allocate_in_msk_myt']:
                locations.append('msk_myt')
            subtask.Parameters.locations = locations

            subtask.Parameters.volumes = json.loads(request['params'].get('volumes', '[]'))

            subtask.Parameters.parent_macro = request['params'].get('parent_macro', '')
            subtask.Parameters.ipv64_tunnel_v2 = request['params'].get('ipv64_tunnel_v2', False)
            subtask.Parameters.internet_tunnel = request['params'].get('internet_tunnel', False)
            subtask.Parameters.mtn_export_to_cauth = request['params'].get('mtn_export_to_cauth', False)
            subtask.Parameters.use_mtn_in_config = request['params'].get('use_mtn_in_config', False)
            subtask.Parameters.portovm_mtn_addrs = request['params'].get('portovm_mtn_addrs', False)
            subtask.Parameters.hbf_slb_name = request['params'].get('hbf_slb_name', [])

            subtask.Parameters.commit_message = request['params'].get('commit_message', '')
            subtask.Parameters.dry_run = self.Parameters.dry_run
            subtask.Parameters.arcadia_revision = ''
            subtask.Parameters.use_last_resources = self.Parameters.use_last_resources

            subtask.Parameters.notifications = [
                sdk2.Notification(
                    statuses=[ctt.Status.FAILURE, ctt.Status.SUCCESS],
                    recipients=[request['params'].get('author', '')],
                    transport=ctn.Transport.EMAIL,
                ),
            ]

            subtask.save()
            return subtask

        elif request['type'] == 'recluster_in_all_dynamic':
            subtask = ReallocateInDynamic(self, description=u'Created by {}'.format(
                request['params'].get('author', '')
            ))

            # Fill params
            subtask.Parameters.author = request['params'].get('author', '')
            subtask.Parameters.group_names = request['params']['group_names']

            # Requirements
            subtask.Parameters.instances_count = request['params'].get('target_instances_count')
            subtask.Parameters.instance_power_units = request['params'].get('target_instance_power') * 40
            subtask.Parameters.instance_memory = request['params'].get('target_instance_memory')
            subtask.Parameters.instance_disk = request['params'].get('target_instance_disk')
            subtask.Parameters.instance_ssd = request['params'].get('target_instance_ssd')
            subtask.Parameters.instance_net = request['params'].get('target_instance_net')
            volumes = request['params'].get('volumes')
            subtask.Parameters.volumes = json.loads(volumes) if volumes is not None else None

            subtask.Parameters.commit_message = request['params'].get('commit_message', '')
            subtask.Parameters.dry_run = self.Parameters.dry_run
            subtask.Parameters.arcadia_revision = ''
            subtask.Parameters.use_last_resources = self.Parameters.use_last_resources

            subtask.Parameters.notifications = [
                sdk2.Notification(
                    statuses=[ctt.Status.FAILURE, ctt.Status.SUCCESS],
                    recipients=[request['params'].get('author', '')],
                    transport=ctn.Transport.EMAIL,
                ),
            ]

            subtask.save()
            return subtask

        elif request['type'] == 'modify_group_card':
            subtask = ModifyGroupCard(self, description=u'Created by {}'.format(
                request['params'].get('author', '')
            ))

            # Fill params
            subtask.Parameters.author = request['params'].get('author', '')
            subtask.Parameters.group_names = request['params']['group_names']

            # Changes
            subtask.Parameters.fields = {
                key.replace('!', '.'): value
                for key, value in request['params']['changes'].items()
            }

            subtask.Parameters.commit_message = request['params'].get('commit_message', '')
            subtask.Parameters.dry_run = self.Parameters.dry_run
            subtask.Parameters.arcadia_revision = ''
            subtask.Parameters.use_last_resources = self.Parameters.use_last_resources

            subtask.Parameters.notifications = [
                sdk2.Notification(
                    statuses=[ctt.Status.FAILURE, ctt.Status.SUCCESS],
                    recipients=[request['params'].get('author', '')],
                    transport=ctn.Transport.EMAIL,
                ),
            ]

            subtask.save()
            return subtask

        elif request['type'] == 'remove_group':
            subtask = GencfgRemoveGroup(self, description=u'Created by {}'.format(
                request['params'].get('author', '')
            ))

            # Fill params
            subtask.Parameters.group_name_list = [request['params']['group']]

            subtask.Parameters.author = request['params'].get('author', '')
            subtask.Parameters.gen_sh_option = ''
            subtask.Parameters.commit_message = request['params'].get('commit_message', '')
            subtask.Parameters.dry_run = self.Parameters.dry_run
            subtask.Parameters.revision = 0
            subtask.Parameters.last_resources = self.Parameters.use_last_resources

            subtask.Parameters.notifications = [
                sdk2.Notification(
                    statuses=[ctt.Status.FAILURE, ctt.Status.SUCCESS],
                    recipients=[request['params'].get('author', '')],
                    transport=ctn.Transport.EMAIL,
                ),
            ]

            subtask.save()
            return subtask

        elif request['type'] == 'create_macros':
            subtask = CreateMacros(self, description=u'Created by {}'.format(
                request['params'].get('author', '')
            ))

            # Fill params
            subtask.Parameters.author = request['params'].get('author', '')
            subtask.Parameters.macros_name = request['params']['macros']
            subtask.Parameters.parent_macros = request['params'].get('parent', '')
            subtask.Parameters.macros_descr = request['params']['description']
            subtask.Parameters.owners = request['params']['owners']

            subtask.Parameters.commit_message = request['params'].get('commit_message', '')
            subtask.Parameters.dry_run = self.Parameters.dry_run
            subtask.Parameters.arcadia_revision = ''
            subtask.Parameters.use_last_resources = self.Parameters.use_last_resources

            subtask.Parameters.notifications = [
                sdk2.Notification(
                    statuses=[ctt.Status.FAILURE, ctt.Status.SUCCESS],
                    recipients=[request['params']],
                    transport=ctn.Transport.EMAIL,
                ),
            ]

            subtask.save()
            return subtask

        elif request['type'] == 'remove_macros':
            subtask = RemoveMacros(self, description=u'Created by {}'.format(
                request['params'].get('author', '')
            ))

            # Fill params
            subtask.Parameters.author = request['params'].get('author', '')
            subtask.Parameters.macros_name = request['params']['macros']

            subtask.Parameters.commit_message = request['params'].get('commit_message', '')
            subtask.Parameters.dry_run = self.Parameters.dry_run
            subtask.Parameters.arcadia_revision = ''
            subtask.Parameters.use_last_resources = self.Parameters.use_last_resources

            subtask.Parameters.notifications = [
                sdk2.Notification(
                    statuses=[ctt.Status.FAILURE, ctt.Status.SUCCESS],
                    recipients=[request['params'].get('author', '')],
                    transport=ctn.Transport.EMAIL,
                ),
            ]

            subtask.save()
            return subtask

        elif request['type'] == 'modify_macros':
            subtask = ModifyMacros(self, description=u'Created by {}'.format(
                request['params'].get('author', '')
            ))

            # Fill params
            subtask.Parameters.author = request['params'].get('author', '')
            subtask.Parameters.macros_name = request['params']['macros']

            subtask.Parameters.fields = {
                key.replace('!', '.'): value
                for key, value in request['params']['changes'].items()
            }

            subtask.Parameters.commit_message = request['params'].get('commit_message', '')
            subtask.Parameters.dry_run = self.Parameters.dry_run
            subtask.Parameters.arcadia_revision = ''
            subtask.Parameters.use_last_resources = self.Parameters.use_last_resources

            subtask.Parameters.notifications = [
                sdk2.Notification(
                    statuses=[ctt.Status.FAILURE, ctt.Status.SUCCESS],
                    recipients=[request['params'].get('author', '')],
                    transport=ctn.Transport.EMAIL,
                ),
            ]

            subtask.save()
            return subtask

        elif request['type'] == 'manipulate_types':
            subtask = ManipulateTypes(self, description=u'Created by {}'.format(
                request['params'].get('author', '')
            ))

            # Fill params
            subtask.Parameters.author = request['params'].get('author', '')
            subtask.Parameters.type_class = request['params']['type_class']
            subtask.Parameters.type_name = request['params']['type_name']
            subtask.Parameters.action = request['params']['action']

            subtask.Parameters.fields = request['params']['fields']

            subtask.Parameters.commit_message = request['params'].get('commit_message', '')
            subtask.Parameters.dry_run = self.Parameters.dry_run
            subtask.Parameters.arcadia_revision = ''
            subtask.Parameters.use_last_resources = self.Parameters.use_last_resources

            subtask.Parameters.notifications = [
                sdk2.Notification(
                    statuses=[ctt.Status.FAILURE, ctt.Status.SUCCESS],
                    recipients=[request['params'].get('author', '')],
                    transport=ctn.Transport.EMAIL,
                ),
            ]

            subtask.save()
            return subtask

        elif request['type'] == 'manipulate_hosts':
            subtask = ManipulateHosts(self, description=u'Created by {}'.format(
                request['params'].get('author', '')
            ))

            # Fill params
            subtask.Parameters.author = request['params'].get('author', '')

            subtask.Parameters.action = request['params']['action']
            subtask.Parameters.list_hosts = request['params']['list_hosts']
            subtask.Parameters.fields = request['params']['fields']

            subtask.Parameters.commit_message = request['params'].get('commit_message', '')
            subtask.Parameters.dry_run = self.Parameters.dry_run
            subtask.Parameters.arcadia_revision = ''
            subtask.Parameters.use_last_resources = self.Parameters.use_last_resources

            subtask.Parameters.notifications = [
                sdk2.Notification(
                    statuses=[ctt.Status.FAILURE, ctt.Status.SUCCESS],
                    recipients=[request['params'].get('author', '')],
                    transport=ctn.Transport.EMAIL,
                ),
            ]

            subtask.save()
            return subtask

        else:
            self.update_request_info(request, {
                'status': 'skip'
            })
            self.writeln('UNDEFINED SUBTASK: {}'.format(request['type']))
