import time
import logging
import datetime

import dateutil.parser

from sandbox.projects.common.gencfg import solomon
from sandbox.common import rest


def _from_utc_to_local(date_time):
    now_timestamp = time.time()
    offset = datetime.datetime.fromtimestamp(now_timestamp) - datetime.datetime.utcfromtimestamp(now_timestamp)
    return (date_time + offset).replace(tzinfo=None)


def _list_times(task_type, status, since, up_to):
    sandbox = rest.Client()
    offset = 0
    while True:
        tasks = sandbox.task.read(
            type=task_type,
            status=status,
            created=since.isoformat() + '..' + up_to.isoformat(),
            limit=500,
            offset=offset,
            children=True,
        )['items']
        if not tasks:
            return
        for task in tasks:
            offset += 1
            duration = task['execution']['current']
            started = dateutil.parser.parse(task['execution']['started'])
            started = _from_utc_to_local(started)
            yield started, duration


def _filter_by_duration(times):
    return [(dt, duration) for dt, duration in times if 3600 > duration > 120]


def _update_speed_chart(sensor_name, task_type, status, since, up_to):
    client = solomon.SolomonClient(project='gencfg', cluster='monitoring', service='health')
    all_times = _list_times(
        task_type,
        status,
        since=since,
        up_to=up_to,
    )

    all_times = _filter_by_duration(all_times)

    for date_time, duration in all_times:
        client.add_sensor(sensor_name, duration, date_time)
        if len(client.sensors_collected) > 100:
            client.flush()

    client.flush()
    logging.info('success with %s!', sensor_name)


def update_test_speed_chart(days_count):
    _update_speed_chart(
        sensor_name='test_speed',
        task_type='TEST_CONFIG_GENERATOR_2',
        status='SUCCESS',
        since=datetime.datetime.now() - datetime.timedelta(days=days_count),
        up_to=datetime.datetime.now(),
    )


def update_build_speed_chart(days_count):
    _update_speed_chart(
        sensor_name='build_speed',
        task_type='BUILD_CONFIG_GENERATOR',
        status='RELEASED',
        since=datetime.datetime.now() - datetime.timedelta(days=days_count),
        up_to=datetime.datetime.now(),
    )
