# -*- coding: utf-8 -*-

import re
import logging
import datetime
import subprocess
import collections

import bson
import dateutil.parser

from sandbox.projects.common.gencfg import task as gencfg_task
from sandbox.projects.common.gencfg import solomon


def _split_datetime(date_time):
    return (
        datetime.date(date_time.year, date_time.month, date_time.day),
        datetime.time(date_time.hour, date_time.minute, date_time.second),
    )


def _parse_svn_datetime(string):
    return dateutil.parser.parse(string).replace(tzinfo=None)


_LineWithCommitNumber = re.compile('^r\d+ \| ')


def _get_commit_and_date(line):
    line = line.split(' | ')
    commit, date = line[0], line[2]
    commit = int(commit.lstrip('r'))
    date = date.partition('(')[0].strip()
    date = _parse_svn_datetime(date)
    return commit, date


def _get_dates(commits_cnt, svn_gencfg_path):
    args = ['svn', 'log', '-l', str(commits_cnt), '-v', svn_gencfg_path]
    popen = subprocess.Popen(args, stdout=subprocess.PIPE)
    stdout = popen.communicate()[0]

    result = {}
    for line in stdout.split('\n'):
        if _LineWithCommitNumber.match(line):
            commit, date = _get_commit_and_date(line)
            result[commit] = date

    return result


def _load_commits(roughly_newer_than):
    result = {}
    for rec in gencfg_task.get_topology_mongo()['commits'].find(
        {
            '_id': {'$gt': bson.ObjectId.from_datetime(roughly_newer_than)},
            'test_passed': {'$exists': True},
        },
        {'commit': 1, 'test_passed': 1},
        sort=[('commit', 1)]
    ):
        result[int(rec['commit'])] = rec

    return result


def get_commits_and_dates(min_datetime, svn_gencfg_paths):
    commits = _load_commits(min_datetime - datetime.timedelta(days=7))
    dates = {}
    for path in svn_gencfg_paths:
        dates.update(_get_dates(max(commits) - min(commits), path))

    result = {}
    for commit in set(dates) & set(commits):
        result[commit] = {
            'test_passed': commits[commit]['test_passed'],
            'datetime': dates[commit],
        }
    return result


def find_less_or_eq(lst, obj):
    for item in sorted(lst, reverse=True):
        if item <= obj:
            return item


def _time_to_delta(x):
    return datetime.datetime.combine(datetime.date.min, x) - datetime.datetime.min


class DayStatistics(object):
    def __init__(self):
        self._points = {}

    def add(self, time, was_broken):
        self._points[time] = was_broken

    def was_broken_between(self, start_time=datetime.time.min, end_time=datetime.time.max):
        points = self._points.copy()
        assert datetime.time.min in points

        points[start_time] = points[find_less_or_eq(points, start_time)]
        points[end_time] = points[find_less_or_eq(points, end_time)]

        total_broken_time = datetime.timedelta()
        times = sorted(filter(lambda x: start_time <= x <= end_time, points))
        for i in range(len(times) - 1):
            a, b = times[i], times[i + 1]
            if points[a]:
                total_broken_time += _time_to_delta(b) - _time_to_delta(a)

        return total_broken_time


class Statistics(object):
    def __init__(self):
        self._current_date = None
        self._is_broken = None
        self._days = collections.defaultdict(DayStatistics)

    def consume(self, date_time, is_broken):
        date, time = _split_datetime(date_time)
        if self._is_broken is None:
            self._consume_first(date, time, is_broken)
        else:
            self._consume(date, time, is_broken)

    def _consume_first(self, date, time, is_broken):
        self._current_date = date
        self._is_broken = is_broken
        self._days[date].add(time, is_broken)

    def _consume(self, date, time, is_broken):
        while self._current_date != date:
            self._current_date += datetime.timedelta(days=1)
            self._days[self._current_date].add(datetime.time.min, self._is_broken)

        self._is_broken = is_broken
        self._days[date].add(time, is_broken)

    def __iter__(self):
        for day in sorted(self._days)[1:]:
            yield day, self._days[day]


def was_broken_statistics(commits_and_dates):
    statistics = Statistics()

    for commit in sorted(commits_and_dates):
        is_broken, date_time = not commits_and_dates[commit]['test_passed'], commits_and_dates[commit]['datetime']
        statistics.consume(date_time, is_broken)

    return statistics


def _was_not_broken_ratio(day_statistics, start_time=datetime.time.min, end_time=datetime.time.max):
    all_seconds = datetime.timedelta(days=1).total_seconds()
    broken_time_seconds = day_statistics.was_broken_between(start_time, end_time).total_seconds()
    not_broken_time_seconds = all_seconds - broken_time_seconds
    return not_broken_time_seconds / all_seconds


def update_solomon_chart(days_count):
    client = solomon.SolomonClient(project='gencfg', cluster='monitoring', service='health')
    statistics = was_broken_statistics(get_commits_and_dates(
        min_datetime=datetime.datetime.now() - datetime.timedelta(days=days_count),
        svn_gencfg_paths=[
            'svn+ssh://arcadia.yandex.ru/arc/trunk/data/gencfg_db',
            'svn+ssh://arcadia.yandex.ru/arc/tags/gencfg',
        ],
    ))

    for date, day_statistics in statistics:
        client.add_sensor(
            'not_broken_time',
            _was_not_broken_ratio(day_statistics),
            datetime.datetime.combine(date, datetime.time.min)
        )
        client.add_sensor(
            'not_broken_time_working_hours',
            _was_not_broken_ratio(day_statistics, datetime.time(9), datetime.time(22)),
            datetime.datetime.combine(date, datetime.time.min)
        )

    client.flush()
    logging.info('success!')
