import re
import sys
import time
import pytz
import pymongo
import logging
import datetime
import functools
import subprocess as sb
import xml.etree.ElementTree as ET

from collections import namedtuple


TAG_PATTERN = re.compile('^stable-(?P<branch>\d+)-r(?P<tag>\d+)$')
GENCFG_TAGS_SVN = 'svn+ssh://arcadia.yandex.ru/arc/tags/gencfg'

MongoStorage = namedtuple('MongoStorage', ['uri', 'replicaset', 'read_preference'])

ALL_HEARTBEAT_C_MONGODB = MongoStorage(
    uri=','.join([
        'myt0-4012.search.yandex.net:27017',
        'myt0-4019.search.yandex.net:27017',
        'sas1-6063.search.yandex.net:27017',
        'sas1-6136.search.yandex.net:27017',
        'vla1-3984.search.yandex.net:27017',
    ]),
    replicaset='heartbeat_mongodb_c',
    read_preference=pymongo.ReadPreference.SECONDARY_PREFERRED
)


def db_topology_commits():
    return pymongo.MongoReplicaSetClient(
        ALL_HEARTBEAT_C_MONGODB.uri,
        replicaset=ALL_HEARTBEAT_C_MONGODB.replicaset,
        read_preference=ALL_HEARTBEAT_C_MONGODB.read_preference
    )['topology_commits']


def _logging(text):
    logging.info(text)
    print(text)


def _retry(count, delay):
    def decorator(func):
        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            for i in xrange(count):
                try:
                    return func(*args, **kwargs)
                except Exception as e:
                    _logging('Exception ({}): {}: {}'.format(i + 1, type(e).__name__, e))
                    if i + 1 == count:
                        raise
                    time.sleep(delay)
        return wrapper
    return decorator


def _from_utc_to_local(date_time):
    now_timestamp = time.time()
    offset = datetime.datetime.fromtimestamp(now_timestamp) - datetime.datetime.utcfromtimestamp(now_timestamp)
    return (date_time + offset).replace(tzinfo=None)


def _parse_svn_log(output):
    result = []
    for line in output.split('\n'):
        if not line:
            continue

        parts = line.split(' | ')

        if len(parts) != 4:
            continue

        try:
            commit, user, time, _ = parts
            commit = int(commit[1:])
        except Exception:
            continue

        date, time, _ = time.split(' ', 2)

        result.append(
            (commit, user, datetime.datetime.strptime('{}T{}'.format(date, time), '%Y-%m-%dT%H:%M:%S'))
        )

    return result


@_retry(6, 10)
def _get_tags_from_mongo():
    data = {}
    for record in db_topology_commits().tags.find(sort=[('_id', -1)]):
        date = record['_id'].generation_time.astimezone(pytz.timezone('Europe/Moscow'))
        offset_aware_date = date.strftime('%Y-%m-%d %H:%M:%S')
        offset_native_date = datetime.datetime.strptime(offset_aware_date, '%Y-%m-%d %H:%M:%S')
        data[record['tag']] = {
            'time': offset_native_date
        }
    return data


@_retry(6, 10)
def _get_list_last_tags():
    xml_tags = sb.check_output(['svn', 'ls', '--xml', GENCFG_TAGS_SVN])
    tags_tree = ET.fromstring(xml_tags)
    tags_mongo = _get_tags_from_mongo()

    print(len(tags_mongo))

    filtered_tags = []
    for entry in tags_tree[0]:
        if not TAG_PATTERN.match(entry[0].text) or entry[0].text not in tags_mongo:
            continue

        filtered_tags.append((
            entry[0].text,  # tag_name
            int(entry[1].attrib['revision']),  # tag_revision
            tags_mongo[entry[0].text]['time']  # tag_released_time
        ))

    filtered_tags.sort(key=lambda x: x[1], reverse=True)

    return filtered_tags


@_retry(6, 10)
def _get_last_tag_commits(tag_name):
    tag_code_commits = _parse_svn_log(
        sb.check_output(['svn', 'log', '-l', '10', '{}/{}'.format(GENCFG_TAGS_SVN, tag_name)])
    )
    tag_code_commits = sorted(tag_code_commits[2:], reverse=True)

    tag_db_commits = _parse_svn_log(
        sb.check_output(['svn', 'log', '-l', '30', '{}/{}/db'.format(GENCFG_TAGS_SVN, tag_name)])
    )
    tag_db_commits = sorted(tag_db_commits[1:], reverse=True)

    return sorted(tag_code_commits + tag_db_commits, reverse=True)


def _get_list_tags_commits(limit=None, days_count=2):
    list_last_tags = []
    for i, (tag_name, tag_revision, tag_time) in enumerate(_get_list_last_tags()):
        if limit and i + 1 >= limit:
            break
        elif (datetime.datetime.now() - tag_time).days >= days_count:
            break
        list_last_tags.append((tag_name, tag_revision, tag_time))
    _logging('Found {} tags'.format(len(list_last_tags)))

    last_commit_last_tag = None
    list_commits_in_tag = []
    for i, (tag_name, tag_revision, tag_time) in enumerate(reversed(list_last_tags)):
        last_commits = _get_last_tag_commits(tag_name)
        _logging('Loaded {}/{} tag {} info'.format(i + 1, len(list_last_tags), tag_name))

        if last_commit_last_tag:
            last_commits = filter(lambda x: x[0] > last_commit_last_tag[0], last_commits)

        if last_commits:
            last_commit_last_tag = last_commits[0]
            list_commits_in_tag.append((tag_name, tag_time, last_commits))
            _logging('Added {}/{} tag {} info'.format(len(list_commits_in_tag), len(list_last_tags), tag_name))

    return reversed(list_commits_in_tag[1:])


def _get_tags_release_delay(days_count):
    tags_creation_delays = []
    for tag_name, tag_time, last_commits in _get_list_tags_commits(days_count=days_count):
        first_commit_in_tag = last_commits[-1]
        tag_creation_delay = (tag_time - first_commit_in_tag[2]).seconds
        tags_creation_delays.append((tag_time, tag_creation_delay))

        _logging('{}: {}({}) {}({}) {}({})'.format(
            tag_name, tag_creation_delay, tag_time.strftime('%d %H:%M'),
            last_commits[0][0], last_commits[0][2].strftime('%d %H:%M'),
            last_commits[-1][0], last_commits[-1][2].strftime('%d %H:%M'),
        ))
    return tags_creation_delays


def update_build_delay_chart(days_count):
    from sandbox.projects.common.gencfg import solomon

    client = solomon.SolomonClient(project='gencfg', cluster='monitoring', service='health')
    for tag_time, tag_creation_delay in _get_tags_release_delay(days_count):
        client.add_sensor('build_delay', tag_creation_delay, tag_time)
        if len(client.sensors_collected) > 100:
            client.flush()
            _logging('Flushed {} sensors'.format(len(client.sensors_collected)))
    client.flush()
    _logging('Flushed {} sensors'.format(len(client.sensors_collected)))
    _logging('success with build_delay!')


if __name__ == '__main__':
    if len(sys.argv) != 2:
        print('python tags_release_delay.py <days_count>')
        sys.exit(0)

    _get_tags_release_delay(int(sys.argv[1]))
