# -*- coding: utf-8 -*-
import sandbox.sdk2 as sdk2
import sandbox.sdk2.parameters as parameters
import sandbox.common.errors as errors

import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

import sandbox.projects.gencfg.mongo as mongo
import sandbox.projects.gencfg.helpers as helpers
import sandbox.projects.gencfg.environment as environment


class ManipulateTypes(sdk2.Task):
    """ Manipulate itype/ctype """

    class Requirements(sdk2.Task.Requirements):
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 10 * 1024, None)
        client_tags = ctc.Tag.CUSTOM_GENCFG_BUILD

    class Context(sdk2.Task.Context):
        svn_diff = ''
        exceptions = []
        commit = -1
        gencfg_revision = -1
        retry_count = 0

    class Parameters(sdk2.Task.Parameters):
        author = parameters.String('Request author')

        with parameters.RadioGroup('Type class') as type_class:
            type_class.values['itype'] = type_class.Value(value='itype', default=True)
            type_class.values['ctype'] = type_class.Value(value='ctype')

        type_name = parameters.String('Type name', required=True)

        with parameters.RadioGroup('Action') as action:
            action.values['add'] = action.Value(value='Create', default=True)

        fields = parameters.Dict('Action fields', required=False)
        commit_message = parameters.String('Commit message', default='', required=False)

        with parameters.Group('Execution') as execution:
            dry_run = sdk2.parameters.Bool(
                'Dry run',
                description='Run the whole procedure, but do not commit results',
                default=True
            )
            arcadia_revision = sdk2.parameters.String(
                'Base revision',
                description='Allocate, check and commit against this revision',
                required=False
            )
            use_last_resources = sdk2.parameters.Bool(
                'Use last released resources',
                required=True,
                default=False
            )

    def on_execute(self):
        self.Context.retry_count += 1

        try:
            gencfg = environment.GencfgEnvironment(self, self.Parameters.arcadia_revision, self.ramdrive.path)
            gencfg.prepare()
            gencfg.install(self.Parameters.use_last_resources)
            gencfg.precalc_caches()

            gencfg_revision = int(gencfg.info(gencfg.src_root)['commit_revision'])
            self.set_info('Task manipulates revision {}'.format(gencfg_revision))
            self.Context.gencfg_revision = gencfg_revision

            try:
                self.manipulate_types(gencfg, self.Parameters)
                if not self.save_svn_diff(gencfg):
                    self.set_info('Nothing to commit. Macros already has selected params.')
                    return
            except Exception as e:
                raise errors.TaskFailure('[{}]: {}'.format(type(e).__name__, e))

            gencfg.run_script('scripts/gen-topology-check.sh')

            if not self.Parameters.dry_run:
                gencfg.update()

                commit = self.commit(
                    gencfg,
                    u'[ManipulateTypes] {}: {} (committed by {}@) '
                    'from https://sandbox.yandex-team.ru/task/{}/view'.format(
                        self.Parameters.type_name, self.Parameters.commit_message.encode('ascii', 'ignore'),
                        self.Parameters.author, self.id
                    ),
                    restart_on_fail=not bool(self.Parameters.arcadia_revision)
                )

                if not commit:
                    return

                self.Context.commit = int(commit)
                self.set_info(
                    'Request committed: <a href="https://a.yandex-team.ru/arc/commit/{0}">{0}</a>'.format(commit),
                    do_escape=False
                )

                self.insert_commit(commit)

                gencfg.update(commit)
                gencfg.run_script('utils/mongo/populate_gencfg_trunk.py')
        except (errors.TaskFailure, errors.TemporaryError):
            raise
        except Exception as e:
            if self.Context.retry_count > 3:
                raise
            raise errors.TemporaryError('{}: {}'.format(type(e).__name__, e))

    def on_finish(self, prev_status, status):
        if status != ctt.Status.SUCCESS:
            self.save_exceptions()

    def on_break(self, prev_status, status):
        self.save_exceptions()

    def manipulate_types(self, gencfg, params):
        manipulate_types_command = [
            './utils/common/manipulate_{}s.py'.format(params.type_class),
            '-a', str(params.action),
            '--{}'.format(params.type_class), params.type_name
        ]

        filed_name_to_key = {
            'description': '-d'
        }
        for field_name, value in params.fields.items():
            manipulate_types_command.extend([filed_name_to_key[field_name], value])

        gencfg.run_process(manipulate_types_command, 'manipulate_{}'.format(params.type_class))

    def save_svn_diff(self, gencfg):
        diff = gencfg.diff()
        self.Context.svn_diff = diff
        return diff

    def commit(self, gencfg, message, restart_on_fail=False):
        try:
            return gencfg.commit(message)
        except Exception as e:
            if restart_on_fail:
                raise errors.TemporaryError(str(e))
            raise

    def insert_commit(self, commit):
        commit = mongo.normalize_commit(commit)

        mongo.commits().update(
            {'commit': commit},
            {
                '$set': {
                    'skip': {'mongo': True, 'clickhouse': False, 'hbf': False},
                    'author': str(self.Parameters.author),
                    'task_id': self.id,
                    'interface': True
                },
            },
            upsert=True
        )

    def save_exceptions(self):
        helpers.print_errors_to_info(self, self.log_path())
        self.Context.exceptions = helpers.get_list_errors(self.log_path())
