# -*- coding: utf-8 -*-
import sandbox.sdk2 as sdk2
import sandbox.sdk2.parameters as parameters
import sandbox.common.errors as errors

import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

import sandbox.projects.gencfg.mongo as mongo
import sandbox.projects.gencfg.helpers as helpers
import sandbox.projects.gencfg.environment as environment


class ModifyGroupCard(sdk2.Task):
    """ Modify group in dynamic """

    class Requirements(sdk2.Task.Requirements):
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 10 * 1024, None)
        ram = 80 * 1024
        cores = 16
        client_tags = ctc.Tag.CUSTOM_GENCFG_BUILD

    class Context(sdk2.Task.Context):
        svn_diff = ''
        exceptions = []
        commit = -1
        gencfg_revision = -1
        retry_count = 0

    class Parameters(sdk2.Task.Parameters):
        author = parameters.String('Request author')
        group_names = parameters.List('List of group names', required=True)
        fields = parameters.Dict('Fields to update', required=True)
        commit_message = parameters.String('Commit message', default='', required=False)

        with parameters.Group('Execution') as execution:
            dry_run = sdk2.parameters.Bool(
                'Dry run',
                description='Run the whole procedure, but do not commit results',
                default=True
            )
            arcadia_revision = sdk2.parameters.String(
                'Base revision',
                description='Allocate, check and commit against this revision',
                required=False
            )
            use_last_resources = sdk2.parameters.Bool(
                'Use last released resources',
                required=True,
                default=False
            )

    def on_execute(self):
        self.Context.retry_count += 1

        try:
            gencfg = environment.GencfgEnvironment(self, self.Parameters.arcadia_revision, self.ramdrive.path)
            gencfg.prepare()
            gencfg.install(self.Parameters.use_last_resources)

            gencfg_revision = int(gencfg.info(gencfg.src_root)['commit_revision'])
            self.set_info('Task manipulates revision {}'.format(gencfg_revision))
            self.Context.gencfg_revision = gencfg_revision

            try:
                self.modify_group_card(gencfg, self.Parameters)
                if not self.save_svn_diff(gencfg):
                    self.set_info('Nothing to commit. Group already has selected params.')
                    return
            except Exception as e:
                raise errors.TaskFailure('[{}]: {}'.format(type(e).__name__, e))

            try:
                gencfg.update()
            except Exception as e:
                raise errors.TemporaryError('{}: {}. Task will be restarted'.format(type(e).__name__, e))

            gencfg.run_script('scripts/gen-topology-check.sh')

            if self.Parameters.dry_run:
                return

            gencfg.update()

            # Run quick checks, to protect against conflicts
            self.run_post_update_checks(gencfg)

            commit = self.commit(
                gencfg,
                u'[ModifyGroupCard] {}: {} (committed by {}@) '
                'from https://sandbox.yandex-team.ru/task/{}/view'.format(
                    ', '.join(self.Parameters.group_names), self.Parameters.commit_message.encode('ascii', 'ignore'),
                    self.Parameters.author, self.id
                ),
                restart_on_fail=not bool(self.Parameters.arcadia_revision)
            )

            if not commit:
                return

            self.Context.commit = int(commit)
            self.set_info(
                'Request committed: <a href="https://a.yandex-team.ru/arc/commit/{0}">{0}</a>'.format(commit),
                do_escape=False
            )

            self.insert_commit(commit)

            gencfg.update(commit)
            gencfg.run_script('utils/mongo/populate_gencfg_trunk.py')
        except (errors.TaskFailure, errors.TemporaryError):
            raise
        except Exception as e:
            if self.Context.retry_count > 3:
                raise
            raise errors.TemporaryError('{}: {}'.format(type(e).__name__, e))

    def on_finish(self, prev_status, status):
        if status != ctt.Status.SUCCESS:
            self.save_exceptions()

    def on_break(self, prev_status, status):
        self.save_exceptions()

    def modify_group_card(self, gencfg, params):
        for group_name in params.group_names:
            for card_key, card_value in params.fields.items():
                if card_key == 'reqs.volumes':
                    modify_group_card_command = [
                        './utils/common/manipulate_volumes.py',
                        '-a', 'put',
                        '-g', group_name,
                        '-j', card_value,
                    ]
                elif card_key == 'reqs.shards.min_power':
                    modify_group_card_command = [
                        './utils/common/fix_cpu_guarantee.py',
                        '-g', group_name,
                        '-p', card_value,
                    ]
                elif card_key == 'properties.hbf_range':
                    modify_group_card_command = [
                        './utils/common/change_hbf_range.py',
                        '--group', group_name,
                        '--hbf-range', card_value
                    ]
                else:
                    modify_group_card_command = [
                        './utils/common/update_card.py',
                        '-y',
                        '-g', group_name,
                        '-k', card_key,
                        '-v', card_value,
                    ]
                gencfg.run_process(modify_group_card_command, 'update_card_{}'.format(group_name))

    def run_post_update_checks(self, gencfg):
        gencfg.run_process(['./utils/check/check_ipv4tunnels.py'], 'check_ipv4tunnels')

    def save_svn_diff(self, gencfg):
        diff = gencfg.diff()
        self.Context.svn_diff = diff
        return diff

    def commit(self, gencfg, message, restart_on_fail=False):
        try:
            return gencfg.commit(message)
        except Exception as e:
            if restart_on_fail:
                raise errors.TemporaryError(str(e))
            raise

    def insert_commit(self, commit):
        commit = mongo.normalize_commit(commit)

        mongo.commits().update(
            {'commit': commit},
            {
                '$set': {
                    'skip': {'mongo': True, 'clickhouse': False, 'hbf': False},
                    'author': str(self.Parameters.author),
                    'task_id': self.id,
                    'interface': True
                },
            },
            upsert=True
        )

    def save_exceptions(self):
        helpers.print_errors_to_info(self, self.log_path())
        self.Context.exceptions = helpers.get_list_errors(self.log_path())
