# coding: utf8
import re
import logging

import sandbox.common.errors as errors
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
import sandbox.projects.gencfg.semaphores as semaphores
import sandbox.projects.gencfg.environment as environment
import sandbox.projects.gencfg.mongo as mongo

from datetime import datetime

from sandbox import sdk2
from sandbox.common.types.task import ReleaseStatus


TAG_PATTERN = re.compile('^stable-(?P<branch>\d+)-r(?P<tag>\d+)$')
TAG_NAME_PATTERN = re.compile(r'(?P<tag_name>stable-\d+-r\d+)')
GENCFG_TAGS_URL = 'svn+ssh://arcadia.yandex.ru/arc/tags/gencfg'
BROKEN_BUILD_STATUSES = tuple(ctt.Status.Group.BREAK) + (ctt.Status.FAILURE, ctt.Status.NOT_RELEASED)
FINISHED_BUILD_STATUSES = ctt.Status.Group.FINISH + ctt.Status.Group.BREAK
RELEASE_BUILD_STATUSES = (ctt.Status.RELEASED, ctt.Status.NOT_RELEASED)
BUILD_TASK_TYPE_NAME = 'BUILD_CONFIG_GENERATOR'


class ReleaseConfigGenerator2(sdk2.Task):
    """ Check commit to gencfg, gencfg_db and run build configs """

    class Requirements(sdk2.Task.Requirements):
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 10 * 1024, None)
        client_tags = ctc.Tag.CUSTOM_GENCFG_BUILD
        semaphore = semaphores.GencfgReleaseGenerator.dynamic

    class Parameters(sdk2.Task.Parameters):
        mongo_db = sdk2.parameters.String('Mongo DB', default='ggdb')
        mongo_collection = sdk2.parameters.String('Mongo Collection', default='gencfg_gui_requests')

        generate_diff_to_prev = sdk2.parameters.Bool('Generate diff to previous tag', default=True)
        generate_improxy_configs = sdk2.parameters.Bool('Generate improxy configs', default=False)

        use_last_resources = sdk2.parameters.Bool('Use last released resources', default=False)

    class Context(sdk2.Task.Context):
        new_tag_name = None
        build_improxy_id = -1

    def set_log(self, info):
        logging.info('[{}] USER_LOG: {}'.format(
            datetime.now().strftime('%H:%M:%S'),
            info
        ))
        self.set_info(info)

    # PATH

    def get_trunk_path(self):
        return self.ramdrive.path / 'trunk'

    # MONGO

    def on_execute(self):
        try:

            with self.memoize_stage.first_stage:
                # Prepare gencfg
                gencfg = self.get_gencfg_environment()

                # Check the need to build the tag
                if not self.need_new_tag(gencfg):
                    self.set_info('Nothing to build')
                    return

                # Get new tag name and create it
                new_tag_name = self.get_new_tag_name(gencfg)
                self.create_tag(gencfg, new_tag_name)
                self.Context.new_tag_name = new_tag_name

                self.set_log('Created tag: {}'.format(self.Context.new_tag_name))

                # Run subtask BUILD_TASK_TYPE_NAME
                build_subtask = self.get_new_build_subtask(self.Context.new_tag_name, BUILD_TASK_TYPE_NAME)
                build_subtask.enqueue()
                self.update_build_improxy_status(status='executing', tag_name=self.Context.new_tag_name,
                                                 sb_task_id=build_subtask.id)

                raise sdk2.WaitTask([build_subtask.id], FINISHED_BUILD_STATUSES, wait_all=True)

            with self.memoize_stage.second_stage:
                build_subtask = self.get_build_subtask(BUILD_TASK_TYPE_NAME)
                self.verify_success_build_subtask(build_subtask)

                # Run RELEASING
                sdk2.Task.server.release(
                    task_id=build_subtask.id,
                    type=ReleaseStatus.STABLE,
                    subject='This is subject of releasing (TESTING)'
                )

                # Wait RELEASED or NOT_RELEASED status
                raise sdk2.WaitTask([build_subtask.id], RELEASE_BUILD_STATUSES, wait_all=True)

            with self.memoize_stage.third_stage:
                build_subtask = self.get_build_subtask(BUILD_TASK_TYPE_NAME)
                self.verify_success_build_subtask(build_subtask)

                # Prepare gencfg
                gencfg = self.get_gencfg_environment()

                self.update_tags_ranges(gencfg)
                self.update_build_improxy_status(status='success', tag_name=self.Context.new_tag_name)

        except Exception:
            # If tag was created remove it
            if self.Context.new_tag_name is not None:
                # Prepare gencfg
                gencfg = self.get_gencfg_environment()

                self.delete_tag(gencfg, self.Context.new_tag_name)
                self.set_log('Deleted tag: {}'.format(self.Context.new_tag_name))
                self.update_build_improxy_status(status='failure', tag_name=self.Context.new_tag_name)
            raise

    def get_gencfg_environment(self):
        gencfg = environment.GencfgEnvironment(self, None, self.get_trunk_path())
        gencfg.prepare()
        gencfg.install(self.Parameters.use_last_resources)
        return gencfg

    def last_commits_in_last_tag(self, gencfg):
        tag_last_log = gencfg.log(svn_path=GENCFG_TAGS_URL, fullpath=True)

        removed_tags = set()
        last_tag_name = None
        tag_db_revision = None
        tag_src_revision = None
        for commit in sorted(tag_last_log, key=lambda x: x['revision'], reverse=True):
            self.set_info('COMMIT: {}'.format(commit))

            # Get tag name
            tag_name = self.get_tag_name_from_msg(commit['msg'])
            if tag_name is None:
                continue

            # Check tag for removed
            if self.get_action_from_msg(commit['msg']) == 'remove':
                removed_tags.add(tag_name)
                continue
            elif tag_name in removed_tags:
                self.set_info('Found removed tag {} - skiping'.format(tag_name))
                continue

            # Sync code and db tag revisions
            if last_tag_name is not None and last_tag_name != tag_name:
                tag_src_revision = None
                tag_db_revision = None

            # Get commits from tag
            for path in commit['paths']:
                if 'copyfrom-path' not in path or 'copyfrom-rev' not in path:
                    continue

                if 'arcadia/gencfg' in path['copyfrom-path'] and tag_src_revision < int(path['copyfrom-rev']):
                    tag_src_revision = int(path['copyfrom-rev'])
                elif 'data/gencfg_db' in path['copyfrom-path'] and tag_db_revision < int(path['copyfrom-rev']):
                    tag_db_revision = int(path['copyfrom-rev'])

            # Check for completeness of information
            if tag_src_revision and tag_db_revision:
                self.set_info('Found created tag {} (code {}, db {})'.format(
                    tag_name, tag_src_revision, tag_db_revision
                ))
                break

        return tag_src_revision, tag_db_revision

    def need_new_tag(self, gencfg):
        # check if last commit is tagged
        tag_src_revision, tag_db_revision = self.last_commits_in_last_tag(gencfg)
        src_last_revision = int(gencfg.info(gencfg.src_root)['commit_revision'])
        db_last_revision = int(gencfg.info(gencfg.db_root)['commit_revision'])

        if src_last_revision <= tag_src_revision and db_last_revision <= tag_db_revision:
            self.set_info('Can not release commit {}: commit {} already released'.format(
                src_last_revision if src_last_revision > db_last_revision else db_last_revision,
                tag_src_revision if tag_src_revision > tag_db_revision else tag_db_revision,
            ))
            return False

        self.set_log('Commits:\n    Code {}, Db {}'.format(src_last_revision, db_last_revision))
        self.set_log('Tags commits:\n    Code {}, Db {}'.format(tag_src_revision, tag_db_revision))

        return True

    def get_new_build_subtask(self, tag_name, build_task_type_name):
        task_ctx = {
            'tag': tag_name,
            'build_bundle': False,
            'generate_diff_to_prev': self.Parameters.generate_diff_to_prev,
            'generate_improxy_configs': self.is_need_build_improxy(),
            'notify_if_finished': self.author,
            'notify_if_failed': self.author,
        }

        BuildConfigGenerator = sdk2.Task[build_task_type_name]
        subtask = BuildConfigGenerator(
            self,
            owner='GENCFG',
            description='Building {}'.format(tag_name),
            **task_ctx
        )
        subtask.save()
        return subtask

    def get_finished_subtasks(self, build_task_type_name):
        BuildTaskType = sdk2.Task[build_task_type_name]
        subtasks = self.find(BuildTaskType, status=FINISHED_BUILD_STATUSES)
        return subtasks

    def get_build_subtask(self, build_task_type_name):
        subtasks = self.get_finished_subtasks(build_task_type_name)
        return subtasks.first() if subtasks else None

    def verify_success_build_subtask(self, build_subtask):
        if build_subtask is None:
            raise errors.TaskFailure('{} - NOT FOUND'.format(BUILD_TASK_TYPE_NAME))
        elif build_subtask.status in BROKEN_BUILD_STATUSES:
            raise errors.TaskFailure('{} - BROKEN'.format(BUILD_TASK_TYPE_NAME))

    def get_new_tag_name(self, gencfg):
        output = gencfg.run_process_output(
            ['./utils/common/manipulate_gencfg_tags.py', 'new_tag_name'], 'get_new_tag_name'
        )
        for line in output.split('\n'):
            if not line:
                continue

            if TAG_PATTERN.match(line.strip()):
                return line.strip()

    def get_tag_name_from_msg(self, msg):
        tag_name = TAG_NAME_PATTERN.search(msg)
        if tag_name is None:
            return None
        return tag_name.groupdict()['tag_name']

    def get_action_from_msg(self, msg):
        if msg.strip().startswith('Created tag'):
            return 'create'
        return 'remove'

    def create_tag(self, gencfg, tag_name):
        gencfg.run_process(['./utils/common/manipulate_gencfg_tags.py', 'create_tag', tag_name], 'create_tag')

    def delete_tag(self, gencfg, tag_name):
        gencfg.run_process(['./utils/common/manipulate_gencfg_tags.py', 'delete_tag', tag_name], 'delete_tag')

    def update_tags_ranges(self, gencfg):
        gencfg.run_process(
            ['./utils/standalone/update_tags_commits_ranges_mongo.py', 'last', 'None', self.Context.new_tag_name],
            'update_tags_commits_ranges_mongo'
        )
        self.set_log('Tags ranges updated')

    def is_need_build_improxy(self):
        if self.Parameters.generate_improxy_configs:
            return True

        background_tasks = mongo.get_collection(self.Parameters.mongo_db, self.Parameters.mongo_collection)
        request = mongo.find_one(
            background_tasks,
            {'type': 'build_improxy'},
            sort_key='time.added', sort_value=-1,
        )

        if request is not None and request['status'] not in ('success', 'skip', 'error'):
            self.Context.build_improxy_id = request['id']
            return True
        return False

    def update_build_improxy_status(self, status=None, tag_name=None, sb_task_id=None):
        if self.Context.build_improxy_id == -1:
            return

        background_tasks = mongo.get_collection(self.Parameters.mongo_db, self.Parameters.mongo_collection)

        update_fields = {}
        if status is not None:
            update_fields.update({'status': status})
        if tag_name is not None:
            update_fields.update({'commit': tag_name})
        if sb_task_id is not None:
            update_fields.update({'sb_task_id': sb_task_id})

        mongo.update_one(background_tasks, {'id': self.Context.build_improxy_id}, update_fields)
