# coding: utf8
# import re
import os
import json
import logging
import requests
# import sandbox.common.errors as errors
# import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
import sandbox.projects.gencfg.mongo as mongo
# import sandbox.projects.gencfg.semaphores as semaphores
import sandbox.projects.gencfg.environment as environment

from datetime import datetime

from sandbox import sdk2
from sandbox import common
from sandbox.sandboxsdk.channel import channel


class ReleaseConfigNotificator(sdk2.Task):
    """ Get all CONFIG_MAIL_DIR resources and send mail notifications """

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.CUSTOM_GENCFG_BUILD
        # semaphore = semaphores.GencfgReleaseGenerator.dynamic

    class Context(sdk2.Task.Context):
        processed_resources = []

    class Parameters(sdk2.Task.Parameters):
        email_resources = sdk2.parameters.Bool('Send E-Mail', required=True, default=True)
        startrek_resources = sdk2.parameters.Bool('Send Startrek', required=True, default=False)
        count_resources = sdk2.parameters.Integer('Count resources', required=True, default=20)
        fill_mongo_run = sdk2.parameters.Bool('Only fill mongo run', required=True, default=False)
        dry_run = sdk2.parameters.Bool('Dry Run', required=True, default=True)

    def set_log(self, info):
        logging.info('[{}] USER_LOG: {}'.format(
            datetime.now().strftime('%H:%M:%S'),
            info
        ))
        self.set_info(info)

    def on_execute(self):
        tag_notify = mongo.get_collection('notifications', 'tags')

        resource_notification_handler = {}
        if self.Parameters.email_resources:
            resource_notification_handler['CONFIG_MAIL_DIR'] = self.send_email
        if self.Parameters.startrek_resources:
            resource_notification_handler['CONFIG_STARTREK_DIR'] = self.send_startrek

        for resource_type, send_notify in sorted(resource_notification_handler.iteritems()):
            notified = [
                x['resource_id']
                for x in (mongo.find_all(
                    tag_notify,
                    {'notified': True, 'resource_type': resource_type},
                    sort_key='resource_id', sort_value=-1,
                    limit=self.Parameters.count_resources) or []
                )
            ]
            notified_min_resource_id = min(notified or [0])

            partially_notified = {
                x['resource_id']: x
                for x in (mongo.find_all(
                    tag_notify,
                    {'notified': False, 'resource_type': resource_type},
                    sort_key='resource_id', sort_value=-1,
                    limit=self.Parameters.count_resources) or []
                )
            }

            self.set_log('Notified resources {}: {}'.format(resource_type, notified))
            self.set_log('Partially notified resources {}: {}'.format(resource_type, partially_notified))

            resources = sdk2.Resource[resource_type].find().limit(self.Parameters.count_resources)
            for resource in resources:
                if resource.task.status != ctt.Status.RELEASED:
                    continue

                if resource.id in notified or resource.id < notified_min_resource_id:
                    self.set_log('Skip resource: {} {}'.format(resource_type, resource.id))
                    mongo.update_one(tag_notify, {'resource_id': resource.id}, {'resource_type': resource_type})
                    continue

                tag_name = resource.task.Context.tag
                resource_path = environment.GencfgEnvironment.resource_path(resource)

                mongo.update_one(tag_notify, {'resource_id': resource.id},
                                 {'resource_type': resource_type, 'notified': False, 'tag_name': tag_name})
                processed_files = partially_notified.get(resource.id, {}).get('processed', [])

                for item in os.listdir(resource_path):
                    if item in processed_files:
                        continue

                    if not self.Parameters.fill_mongo_run:
                        send_notify(tag_name, resource.task.id, os.path.join(resource_path, item))
                    processed_files.append(item)

                    mongo.update_one(tag_notify, {'resource_id': resource.id}, {'processed': processed_files})
                mongo.update_one(tag_notify, {'resource_id': resource.id}, {'notified': True})
                self.Context.processed_resources.append(resource.id)
                self.set_log('{}: {} - DONE'.format(resource_type, resource.id))

    def send_email(self, tag_name, task_id, filepath):
        mail_prefix = (
            "Добрый день.\n\n"
            "Вы получили это письмо из-за изменений в некоторых GenCfg-группах. "
            "Более подробно про релиз можно посмотреть тут {} или спросить у одного из следующих людей: {}.\n\n"
            "Ниже -- полный diff групп, за которыми вы наблюдаете и которые поменялись.\n\n".format(
                common.utils.get_task_link(task_id), 'sereglond,shotinleg'
            )
        )

        with open(filepath, 'r') as user_resease_diff:
            mail_body = mail_prefix + user_resease_diff.read()

        mail_user, _ = os.path.splitext(os.path.basename(filepath))
        if mail_user.find('@') != -1:
            if mail_user.split('@', 1)[1] != 'yandex-team.ru':
                mail_user = None
            else:
                mail_user = mail_user.split('@', 1)[0]

        if not mail_user:
            return

        mail_type = '(full diff)' if mail_user == 'ALL' else '(personal to {})'.format(mail_user)
        mail_subj = '[stable] Building tag@{} {}'.format(tag_name, mail_type)

        if mail_user == 'ALL':
            if self.Parameters.dry_run:
                self.set_log('To: {}\nSubject: {}\n{}\n------'.format('osol', mail_subj, mail_body))
                return
            channel.sandbox.send_email('osol', '', mail_subj, mail_body)
            channel.sandbox.send_email('shotinleg', '', mail_subj, mail_body)
        else:
            if self.Parameters.dry_run:
                self.set_log('To: {}\nSubject: {}\n{}\n------'.format(mail_user, mail_subj, mail_body))
                return
            channel.sandbox.send_email(mail_user, '', mail_subj, mail_body)

    def send_startrek(self, tag_name, task_id, filepath):
        with open(filepath, 'r') as file:
            body = file.read()
        st_task_id = os.path.basename(filepath)

        text = 'Released gencfg tag <{0:s}> in task {1:s} :\n\n{2:s}'.format(
            tag_name, 'https://sandbox.yandex-team.ru/task/{}'.format(task_id), body
        )
        text = json.dumps({'text': text})

        url = 'https://st-api.yandex-team.ru/v2/issues/{}/comments'.format(st_task_id)
        headers = {
            'Authorization': 'OAuth {}'.format(sdk2.Vault.data('GENCFG', 'gencfg_default_oauth')),
            'Content-Type': 'application/json',
        }
        requests.post(url, data=text, headers=headers)
