# coding: utf8
import pymongo
import logging
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
import sandbox.projects.gencfg.environment as environment

from sandbox import sdk2
from datetime import datetime
from sandbox.projects.gencfg.TestConfigGenerator2 import TestConfigGenerator2


class SpawnTestConfigGenerator2(sdk2.Task):
    """ Check updates and run tasks """

    MONGO_SETTINGS = {
        'uri': ','.join([
            'myt0-4012.search.yandex.net:27017',
            'myt0-4019.search.yandex.net:27017',
            'sas1-6063.search.yandex.net:27017',
            'sas1-6136.search.yandex.net:27017',
            'vla1-3984.search.yandex.net:27017',
        ]),
        'replicaset': 'heartbeat_mongodb_c',
        'read_preference': pymongo.ReadPreference.PRIMARY
    }

    class Requirements(sdk2.Task.Requirements):
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 10 * 1024, None)
        client_tags = ctc.Tag.CUSTOM_GENCFG_BUILD

    class Parameters(sdk2.Task.Parameters):
        mongo_db = sdk2.parameters.String(
            'Mongo DB',
            required=True,
            default='shotinleg_topology_commits'
        )
        mongo_collection = sdk2.parameters.String(
            'Mongo Collection',
            required=True,
            default='commits'
        )
        gencfg_option = sdk2.parameters.String(
            'Gencfg option (<empty>, run_check, precommit)',
            default='run_checks'
        )
        retry_count = sdk2.parameters.Integer(
            'Count retry after TEMPORARY',
            default=2
        )
        hook_code_commits = sdk2.parameters.Bool(
            'Check commits to code',
            default=False
        )
        dry_run_child = sdk2.parameters.Bool(
            'Run without populate/import',
            default=True
        )
        update_commit_info = sdk2.parameters.Bool(
            'Update commit info in mongo',
            default=False
        )
        use_last_resources = sdk2.parameters.Bool(
            'Use last released resources',
            required=True,
            default=False
        )

    class Context(sdk2.Task.Context):
        list_subtasks_info = []
        last_started_subtask = None

    def set_log(self, info):
        logging.info('[{}] USER_LOG: {}'.format(
            datetime.now().strftime('%H:%M:%S'),
            info
        ))
        self.set_info(info)

    # PATH

    def get_trunk_path(self):
        return self.ramdrive.path / 'trunk'

    # MONGO

    @staticmethod
    def db_get_collection(mongo_db, mongo_collection):
        collection = pymongo.MongoReplicaSetClient(
            SpawnTestConfigGenerator2.MONGO_SETTINGS['uri'],
            connectTimeoutMS=5000,
            replicaSet=SpawnTestConfigGenerator2.MONGO_SETTINGS['replicaset'],
            w='majority',
            wtimeout=15000,
            read_preference=SpawnTestConfigGenerator2.MONGO_SETTINGS['read_preference']
        )[mongo_db][mongo_collection]
        return collection

    def db_find_one(self, query=None):
        query = query or {}
        request = self.collection.find(query).sort('$natural', pymongo.DESCENDING).limit(1)
        request = request[0] if request.count() else None
        return request

    def db_find(self, query=None, limit=100):
        query = query or {}
        requests = self.collection.find(query).sort('$natural', pymongo.DESCENDING).limit(limit)
        requests = [x for x in requests]
        return requests

    def db_update_one(self, query, updates):
        self.collection.update_one(
            query,
            {"$set": updates},
            upsert=True
        )

    def on_execute(self):
        self.mongo_db = self.Parameters.mongo_db
        self.mongo_collection = self.Parameters.mongo_collection
        self.collection = self.db_get_collection(
            self.mongo_db, self.mongo_collection
        )

        with self.memoize_stage.search_commits:
            self.gencfg = environment.GencfgEnvironment(self, None, self.get_trunk_path())
            self.gencfg.prepare()

            commits_for_test = self.get_commits_for_test()

            if not commits_for_test:
                self.set_log('Commits for test not found')
                return

            self.Context.list_subtasks_info = self.get_list_subtasks_info(commits_for_test)

        if self.Context.last_started_subtask is not None:
            self.update_subtasks(self.Context.last_started_subtask)
            self.Context.last_started_subtask = None

        if self.Context.list_subtasks_info:
            self.Context.last_started_subtask = self.start_next_suabtask()

            raise sdk2.WaitTask(
                [self.Context.last_started_subtask[1]],
                ctt.Status.Group.FINISH + ctt.Status.Group.BREAK,
                wait_all=True
            )

        self.set_log('All subtasks finished')

    def get_commits_for_test(self, limit=5):
        last_checked_commit = self.get_last_commit_from_mongo()
        list_new_commits = self.get_new_commits_from_svn(last_checked_commit, self.Parameters.hook_code_commits)
        list_not_tested_commits = self.get_not_tested_commits_from_mongo(last_checked_commit)

        self.set_log('last_checked_commit: {}'.format(last_checked_commit))
        self.set_log('list_new_commits: {}'.format(list_new_commits))
        self.set_log('list_not_tested_commits: {}'.format(list_not_tested_commits))

        list_commits_for_test = list_new_commits[:]
        for commit in list_not_tested_commits:
            if commit not in list_new_commits:
                list_commits_for_test.append(commit)
        list_commits_for_test.sort(reverse=True)

        if len(list_commits_for_test) > limit:
            list_commits_for_test = list_commits_for_test[0:limit]

        self.set_log('list_commits_for_test: {}'.format(list_commits_for_test))

        return list_commits_for_test

    def get_list_subtasks_info(self, commits_for_test):
        list_subtasks_info = []
        for commit in reversed(commits_for_test):
            subtask = self.get_subtask(commit, self.Parameters.gencfg_option)
            self.update_commit_info(commit, {'status': 'hooked', 'task_id': self.id})
            list_subtasks_info.append((commit, subtask.id))

        self.set_log('list_subtasks_info: {}'.format(list_subtasks_info))

        return list_subtasks_info

    def update_subtasks(self, finished_subtask_info):
        subtask = self.get_subtask_by_id(finished_subtask_info[1], ctt.Status.Group.FINISH + ctt.Status.Group.BREAK)

        if not subtask:
            return

        self.update_commit_info(finished_subtask_info[0], {'status': 'finished', 'task_id': subtask.id})
        self.Context.list_subtasks_info.remove(finished_subtask_info)

        if subtask.status != ctt.Status.SUCCESS:
            return

        for subtask_info in self.Context.list_subtasks_info:
            self.update_commit_info(subtask_info[0], {'status': 'finished'})
            self.remove_subtask(subtask_info[1])
        self.Context.list_subtasks_info = []

    def start_next_suabtask(self):
        last_started_subtask = self.Context.list_subtasks_info[-1]
        subtask = self.get_subtask_by_id(last_started_subtask[1], ctt.Status.Group.DRAFT)
        subtask.enqueue()

        self.set_log('start_next_suabtask: {}'.format(last_started_subtask))
        self.update_commit_info(last_started_subtask[0], {'status': 'testing', 'task_id': subtask.id})

        return last_started_subtask

    # DATA

    def get_last_commit_from_mongo(self):
        commits = self.db_find({'$or': [
            {'status': 'tested'},
            {'status': 'finished'}
        ]}, limit=20)

        self.set_log('get_last_commit_from_mongo: {}'.format(commits))

        last_commit = 0
        for commit in commits:
            if last_commit < int(commit['commit']):
                last_commit = int(commit['commit'])

        if not last_commit:
            raise Exception('Commit not found in mongo')

        return last_commit

    def get_new_commits_from_svn(self, start_commit, hook_code_commits=False):
        commits = []
        commits.extend(self.gencfg.log(svn_path=self.gencfg.db_root))
        if hook_code_commits:
            commits.extend(self.gencfg.log(svn_path=self.gencfg.src_root))

        list_new_commits = [x['revision'] for x in commits if x['revision'] > start_commit]
        list_new_commits.sort(reverse=True)

        return list_new_commits

    def get_not_tested_commits_from_mongo(self, start_commit):
        list_not_tested_commits = []
        commits = self.db_find()

        self.set_log('get_not_tested_commits_from_mongo: {}'.format(commits))

        for commit in commits:
            if commit.get('test_passed') in (True,):
                break
            elif commit.get('status') in ('tested', 'finished'):
                continue
            elif commit.get('test_passed') in (False,):
                continue
            list_not_tested_commits.append(int(commit['commit']))

        list_not_tested_commits = [commit for commit in list_not_tested_commits if commit < start_commit]
        list_not_tested_commits.sort(reverse=True)

        return list_not_tested_commits

    def get_subtask(self, commit, gen_sh_option):
        subtask = TestConfigGenerator2(
            self,
            description='[SpawnTestConfigGenerator2] '
                        'Revision: <a href="https://a.yandex-team.ru/arc/commit/{0}">{0}</a>'.format(commit)
        )

        # Fill params
        subtask.Parameters.mongo_db = self.Parameters.mongo_db
        subtask.Parameters.mongo_collection = self.Parameters.mongo_collection
        subtask.Parameters.revision = commit
        subtask.Parameters.gencfg_option = gen_sh_option
        subtask.Parameters.retry_count = self.Parameters.retry_count
        subtask.Parameters.dry_run = self.Parameters.dry_run_child
        subtask.Parameters.update_commit_info = self.Parameters.update_commit_info
        subtask.Parameters.use_last_resources = self.Parameters.use_last_resources

        subtask.Parameters.kill_timeout = 1800

        subtask.save()
        return subtask

    def update_commit_info(self, commit, update):
        if not self.Parameters.update_commit_info:
            return

        self.db_update_one({'commit': str(commit)}, update)
        self.set_log('update_commit_info: {} {}'.format({'commit': str(commit)}, update))

    def get_subtask_by_id(self, subtask_id, statuses):
        subtasks = self.find(TestConfigGenerator2)

        for task in subtasks:
            if task.id == subtask_id and task.status in statuses:
                return task
        return None

    def remove_subtask(self, subtask_id):
        subtasks = self.get_subtask_by_id(subtask_id, ctt.Status.Group.DRAFT)
        subtasks.delete()
        self.set_log('remove_subtask: {}'.format(subtask_id))
