# coding: utf8
import logging

import sandbox.sdk2.parameters as parameters
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
import sandbox.projects.gencfg.environment as environment


from datetime import datetime

from sandbox import sdk2


class SyncGencfgDbWithMongo(sdk2.Task):
    """ Get data from gencfg_db and push to mongo """

    class Requirements(sdk2.Task.Requirements):
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 10 * 1024, None)
        client_tags = ctc.Tag.CUSTOM_GENCFG_BUILD

    class Parameters(sdk2.Task.Parameters):
        with parameters.Group('Sync sections') as sections:
            hosts_to_groups = sdk2.parameters.Bool('hosts_to_groups', required=True, default=False)
            hosts_to_hardware = sdk2.parameters.Bool('hosts_to_hardware', required=True, default=False)

        with parameters.Group('Task settings') as settings:
            dry_run = sdk2.parameters.Bool('Dry Run', required=True, default=True)
            test_run = sdk2.parameters.Bool('Test Run', required=True, default=False)
            auto_clean = sdk2.parameters.Bool('Auto Clean', required=True, default=True)
            use_last_resources = sdk2.parameters.Bool(
                'Use last released resources',
                required=True,
                default=False
            )

    # PATHS

    def get_trunk_path(self):
        return self.ramdrive.path / 'trunk'

    def get_prev_trunk_path(self):
        return self.ramdrive.path / 'prev_trunk'

    # LOGGING

    @staticmethod
    def set_log(info):
        logging.info('[{}] USER_LOG: {}'.format(
            datetime.now().strftime('%H:%M:%S'),
            info
        ))

    def writeln(self, info):
        self.set_info('{}'.format(info))
        self.set_log('{}'.format(info))

    # MONGO

    def on_execute(self):
        self.writeln('TASK STARTED')
        gencfg = environment.GencfgEnvironment(self, None, self.get_trunk_path())
        self.prepare_gencfg_path(gencfg)

        if self.Parameters.hosts_to_groups:
            self.sync_hosts_info_with_mongo(gencfg, 'hosts_to_groups')
        if self.Parameters.hosts_to_hardware:
            self.sync_hosts_info_with_mongo(gencfg, 'hosts_to_hardware')

    def sync_hosts_info_with_mongo(self, gencfg, option):
        hosts_to_groups_cmd = [
            './utils/mongo/sync_hosts_info_with_mongo.py', '-s', option, '-v'
        ]
        if self.Parameters.dry_run:
            hosts_to_groups_cmd.append('--dry-run')
        elif self.Parameters.test_run:
            hosts_to_groups_cmd.append('--test-run')
        else:
            hosts_to_groups_cmd.append('--prod-run')

        if not self.Parameters.dry_run and self.Parameters.auto_clean:
            hosts_to_groups_cmd.append('--auto-clean')

        gencfg.run_process(hosts_to_groups_cmd, 'sync_hosts_info_with_mongo-{}'.format(option))

    def prepare_gencfg_path(self, gencfg):
        """
        Checkout gencfg on select revision
        """
        gencfg.prepare()
        gencfg.install(self.Parameters.use_last_resources)
