# coding: utf8
import json
import pymongo
import logging
import sandbox.common.types.misc as ctm
# import sandbox.common.types.client as ctc
import sandbox.projects.gencfg.environment as environment

from sandbox import sdk2
from datetime import datetime
from sandbox.common.errors import TaskFailure


class SyncHostDataMongo(sdk2.Task):
    """ Check diff and sync to mongo """

    MONGO_SETTINGS = {
        'uri': ','.join([
            'myt0-4012.search.yandex.net:27017',
            'myt0-4019.search.yandex.net:27017',
            'sas1-6063.search.yandex.net:27017',
            'sas1-6136.search.yandex.net:27017',
            'vla1-3984.search.yandex.net:27017',
        ]),
        'replicaset': 'heartbeat_mongodb_c',
        'read_preference': pymongo.ReadPreference.PRIMARY
    }

    class Requirements(sdk2.Task.Requirements):
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 10 * 1024, None)
        # client_tags = ctc.Tag.CUSTOM_GENCFG_BUILD

    class Context(sdk2.Task.Context):
        hosts_info_diff = ''

    class Parameters(sdk2.Task.Parameters):
        mongo_db = sdk2.parameters.String('Mongo DB', required=True, default='hosts_data')
        options = sdk2.parameters.List('List of options', default=['groups', 'hardware'])
        clean_run = sdk2.parameters.Bool('Clean run', required=True, default=False)
        dry_run = sdk2.parameters.Bool('Dry Run', required=True, default=True)
        use_last_resources = sdk2.parameters.Bool(
            'Use last released resources',
            required=True,
            default=False
        )

    # PATHS

    def get_trunk_path(self):
        return self.ramdrive.path / 'trunk'

    def get_prev_trunk_path(self):
        return self.ramdrive.path / 'prev_trunk'

    # LOGGING

    @staticmethod
    def set_log(info):
        logging.info('[{}] USER_LOG: {}'.format(
            datetime.now().strftime('%H:%M:%S'),
            info
        ))

    def writeln(self, info):
        self.set_info('{}'.format(info))
        self.set_log('{}'.format(info))

    # MONGO

    @staticmethod
    def get_monog_db(mongo_db):
        collection = pymongo.MongoReplicaSetClient(
            SyncHostDataMongo.MONGO_SETTINGS['uri'],
            connectTimeoutMS=5000,
            replicaSet=SyncHostDataMongo.MONGO_SETTINGS['replicaset'],
            w='majority',
            wtimeout=15000,
            read_preference=SyncHostDataMongo.MONGO_SETTINGS['read_preference']
        )[mongo_db]
        return collection

    def db_find_one(self, db, collection, query):
        request = db[collection].find(query).sort(
            '$natural', pymongo.ASCENDING
        ).limit(1)
        request = request[0] if request.count() else None
        return request

    def db_update_one(self, db, collection, query, updates):
        db[collection].update_one(
            query,
            {"$set": updates},
            upsert=True
        )

    def on_execute(self):
        self.writeln('TASK STARTED')
        db = self.get_monog_db(self.Parameters.mongo_db)
        hosts_info_revision = (self.db_find_one(db, 'meta', {'collection': 'hosts_info'}) or {}).get('revision')

        base_revision = None

        gencfg = environment.GencfgEnvironment(self, base_revision, self.get_trunk_path())
        self.prepare_gencfg_path(gencfg)
        gencfg_revision = int(gencfg.info(gencfg.src_root)['commit_revision'])
        self.writeln('GENCFG REVISION: {}'.format(gencfg_revision))

        gencfg_prev = None
        if hosts_info_revision >= gencfg_revision and not self.Parameters.clean_run:
            self.writeln('NOTHING TO SYNC. UPDATED TO LAST REVISION.')
            return
        elif hosts_info_revision and not self.Parameters.clean_run:
            gencfg_prev = environment.GencfgEnvironment(self, hosts_info_revision, self.get_prev_trunk_path())
            self.prepare_gencfg_path(gencfg_prev)

        for option in self.Parameters.options:
            self.set_log('Run `hosts_info` for `{}`'.format(option))
            self.hosts_info(option, gencfg, gencfg_prev, db)

        if not self.Parameters.dry_run:
            self.db_update_one(
                db, 'meta', {'collection': 'hosts_info'}, {'revision': gencfg_revision}
            )

    def hosts_info(self, option, gencfg, gencfg_prev, db):
        try:
            hosts_info = json.loads(gencfg.run_process_output(
                ['./utils/common/dump_hosts_info.py', option], 'dump_hosts_info_{}'.format(option)
            ))

            if gencfg_prev:
                hosts_info_prev = json.loads(gencfg_prev.run_process_output(
                    ['./utils/common/dump_hosts_info.py', option], 'dump_hosts_info_{}_prev'.format(option)
                ))
                hosts_info_diff = self.hosts_info_diff(hosts_info, hosts_info_prev)
            else:
                hosts_info_diff = hosts_info

            # Check diff empty
            if not hosts_info_diff:
                self.writeln('{}: NOTHING TO SYNC. DIFF IS EMPTY.'.format(option))
                return

            hosts_info_diff_dump = json.dumps(hosts_info_diff)
            if len(hosts_info_diff_dump) > 5000:
                hosts_info_diff_dump = 'Diff too much'
            self.Context.hosts_info_diff += 'HOST INFO ({})\n{}\n'.format(option, hosts_info_diff_dump)

            if self.Parameters.dry_run:
                return

            for hostname, data in hosts_info_diff.iteritems():
                self.db_update_one(db, 'hosts_info', {'hostname': hostname}, {option: data})

        except Exception as e:
            raise TaskFailure('[{}] {}'.format(type(e).__name__, e))

    def prepare_gencfg_path(self, gencfg):
        """
        Checkout gencfg on select revision
        """
        gencfg.prepare()
        gencfg.install()

    def hosts_info_diff(self, left_hosts_info, right_hosts_info):
        hosts_info_diff = {}
        for hostname in left_hosts_info:
            if hostname not in right_hosts_info:
                hosts_info_diff[hostname] = left_hosts_info[hostname]
            elif left_hosts_info[hostname] != right_hosts_info[hostname]:
                hosts_info_diff[hostname] = left_hosts_info[hostname]
        return hosts_info_diff
