# coding: utf8
import os
import time
import json
import shutil
import tarfile
import requests
import logging

from sandbox import sdk2
from datetime import datetime
from sandbox.sandboxsdk import ssh
from sandbox.sdk2.helpers import ProcessLog
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.gencfg.resource_types import GENCFG_PYTHON_VENV  # noqa
from sandbox.projects.gencfg.resource_types import GENCFG_BALANCER_PYTHON_VENV  # noqa


class BaseGencfgTask(sdk2.Task):
    """ Chech updates and run tasks """

    # INFORMING

    @staticmethod
    def set_log(info):
        logging.info('[{}] USER_LOG: {}'.format(
            datetime.now().strftime('%H:%M:%S'),
            info
        ))

    def writeln(self, info):
        self.set_info('{}'.format(info))
        self.set_log('{}'.format(info))

    # RUN SUBPROCESS FUNCTIONS

    def check_call(self, cmd, log_name, work_dir='.', env=dict()):
        self.set_log('check_call {}'.format(cmd))

        shell = False if isinstance(cmd, list) else True
        full_env = dict(os.environ.copy().items() + env.items())

        with ProcessLog(self, logger=log_name) as pl:
            sp.check_call(
                cmd, shell=shell, stdout=pl.stdout, stderr=pl.stderr,
                cwd=work_dir, env=full_env
            )

        self.set_log('check_call finished')

    def check_output(self, cmd, log_name, work_dir='.', env=dict()):
        self.set_log('check_call {}'.format(cmd))

        shell = False if isinstance(cmd, list) else True
        full_env = dict(os.environ.copy().items() + env.items())

        with ProcessLog(self, logger=log_name) as pl:
            output = json.loads(sp.check_output(
                cmd, shell=shell, stderr=pl.stderr,
                cwd=work_dir, env=full_env
            ))

        self.set_log('check_call finished')
        return output

    def background_call(self, cmd, log_name, work_dir='.', env=dict()):
        self.set_log('check_call {}'.format(cmd))

        shell = False if isinstance(cmd, list) else True
        full_env = dict(os.environ.copy().items() + env.items())

        log_stdout = str(self.log_path('{}.out.log'.format(log_name)))
        log_stderr = str(self.log_path('{}.out.log'.format(log_name)))

        with open(log_stdout, 'w') as stdout, open(log_stderr, 'w') as stderr:
            process = sp.Popen(
                cmd, shell=shell, stdout=stdout, stderr=stderr,
                cwd=work_dir, env=full_env
            )

        self.set_log('check_call finished')
        return process

    # RESOURCES

    def resource_by_type(self, resource_type):
        resource = sdk2.Resource[resource_type].find().first()
        if resource:
            resource_path = str(sdk2.ResourceData(resource).path)
            return resource_path
        return None

    def extract(self, tar_file, dest_path):
        with tarfile.open(tar_file) as a:
            a.extractall(dest_path)

    def install_venv(self, resource_type, install_path):
        resource = self.resource_by_type(resource_type)
        if resource:
            self.extract(resource, self.mkdir_clean(install_path, 'venv'))
        else:
            raise Exception('Cannot download resource {}'.format(resource_type))
        return os.path.join(install_path, 'venv', 'venv')

    def install_resource(self, resource_type, install_path):
        resource = self.resource_by_type(resource_type)
        if resource:
            self.extract(resource, install_path)
        else:
            raise Exception('Cannot download resource {}'.format(resource_type))

    # HELPERS FUNCTIONS

    @staticmethod
    def retry(retry_conut, delay, func, *args, **kwargs):
        for i in xrange(retry_conut):
            try:
                return func(*args, **kwargs)
            except Exception as e:
                if i == retry_conut - 1:
                    raise
                BaseGencfgTask.set_log(str(e))
                time.sleep(delay)

    @staticmethod
    def mkdir_clean(folder, dir_name):
        """
        If dir_name not exists in folder - make dir
        else remove dir_name and make dir
        """
        dst_dir = os.path.join(folder, dir_name)
        if os.path.exists(dst_dir):
            shutil.rmtree(dst_dir)
        os.mkdir(dst_dir)

        return dst_dir

    # SVN

    @staticmethod
    def checkout(dst_path, revision):
        """
        Checkout trunk root and gencfg with gencfg_db on selected revision
        return gencfg_path and gencfg_db_path
        """
        gencfg_path = os.path.join(dst_path, 'arcadia', 'gencfg')
        gencfg_db_path = os.path.join(dst_path, 'data', 'gencfg_db')

        sdk2.svn.Arcadia.checkout(
            'arcadia:/arc/trunk/', dst_path,
            depth=sdk2.svn.Svn.Depth.IMMEDIATES, revision=revision
        )

        sdk2.svn.Arcadia.update(
            gencfg_path, depth=sdk2.svn.Svn.Depth.INFINITY,
            revision=revision
        )

        sdk2.svn.Arcadia.update(
            gencfg_db_path, depth=sdk2.svn.Svn.Depth.INFINITY,
            revision=revision
        )

        # TODO: proper function
        try:
            os.remove(gencfg_path + '/db')
        except OSError:
            pass

        os.symlink(gencfg_db_path, gencfg_path + '/db')

        return gencfg_path, gencfg_db_path

    def commit_to_trunk(self, from_path, commit_message):
        with ssh.Key(self, "robot-gencfg", "ROBOT_GENCFG_SSH_KEY"):
            commit_out = sdk2.svn.Arcadia.commit(
                from_path,
                commit_message,
                user='robot-gencfg'
            )

            if not commit_out.strip():
                raise Exception('Empty output from Arcadia.commit')

    # GENCFG

    def install_sh(self, gencfg_path):
        """
        Preparing gencfg to start
        """
        # Unpack compiled cython scripts (*.so)
        self.install_resource('GENCFG_BIN_UTILS', gencfg_path)

        # Installing gencfg virtual enviroument
        self.install_venv('GENCFG_PYTHON_VENV', gencfg_path)

        # Installing balancer_gencfg virtual enviroument
        self.install_venv(
            'GENCFG_BALANCER_PYTHON_VENV', os.path.join(gencfg_path, 'custom_generators', 'balancer_gencfg')
        )

        # Making
        # self.check_call(['make', '-C', 'contrib/cJSON/'], 'make_cjson', gencfg_path)

        # Aliasing
        self.check_call(['ln', '-s', 'gaux', 'aux'], 'ln_gaux_aux', gencfg_path)

    def gen_sh(self, gencfg_path, option=''):
        """
        Run gen.sh with option:
            bash gen.sh
            bash gen.sh precommit
            bash gen.sh run_checks
        """
        cmd = './gen.sh {}'.format(option)
        self.check_call(cmd, 'gen_sh', gencfg_path)

    def get_passed_commit(self):
        """
        Return number of last checked commit if it passed
        or raise Exception
        """
        r = requests.get('http://clusterstate.yandex-team.ru/gencfg/json')

        if r.status_code != 200:
            raise Exception('Cannot get last commit state')

        last_commit = {'commit': 0, 'test_passed': True}
        for commit in r.json()['commits']:
            if int(commit['commit']) > int(last_commit['commit']):
                last_commit['commit'] = commit['commit']
                last_commit['test_passed'] = commit['test_passed']

        if not last_commit['test_passed']:
            raise Exception('Last commit is not passed {}'.format(
                last_commit['commit']
            ))

        return last_commit['commit']

    def on_execute(self):
        raise NotImplementedError('Need implement this method')
