import os
import re
import json
import shutil
import tarfile

from sandbox import sdk2
import sandbox.sdk2.helpers as helpers


ARCADIA_TRUNK_ROOT = 'arcadia:/arc/trunk'
ARCADIA_TAGS_ROOT = 'arcadia:/arc/tags'
COMMIT_RE = re.compile('^Committed revision (?P<revision>\d+)\.$', re.M)


# TODO: use SandboxEnvironment base class for resources
class GencfgEnvironment(object):
    def __init__(self, task, revision=None, base_dir=None, tag=None):
        self.task = task
        self.revision = None if not revision else int(revision)
        self.tag = tag
        self.base_dir = base_dir  # TODO: make sure type is good

    @property
    def arcadia_root(self):
        return self.base_dir / 'arcadia_root'

    @property
    def src_root(self):
        if self.tag:
            return str(self.arcadia_root / 'gencfg' / self.tag)
        return str(self.arcadia_root / 'arcadia' / 'gencfg')

    @property
    def db_root(self):
        if self.tag:
            return str(self.arcadia_root / 'gencfg' / self.tag / 'db')
        return str(self.arcadia_root / 'data' / 'gencfg_db')

    @property
    def tmp_root(self):
        tmp = self.base_dir / 'tmp'
        tmp.mkdir(exist_ok=True)

        return str(tmp)

    @property
    def svn_user(self):
        return 'zomb-sandbox-rw'

    def prepare(self):
        sdk2.svn.Arcadia.checkout(
            ARCADIA_TAGS_ROOT if self.tag else ARCADIA_TRUNK_ROOT,
            str(self.arcadia_root),
            depth=sdk2.svn.Svn.Depth.IMMEDIATES,
            revision=self.revision,
        )
        sdk2.svn.Arcadia.update(
            self.src_root,
            depth=sdk2.svn.Svn.Depth.INFINITY,
            revision=self.revision,
        )
        sdk2.svn.Arcadia.update(
            self.db_root,
            depth=sdk2.svn.Svn.Depth.INFINITY,
            revision=self.revision,
        )

        # use Path.symlink_to
        if not self.tag:
            os.symlink(self.db_root, self.src_root + '/db')

        shutil.copyfile(
            os.path.join(self.src_root, 'sandbox.resources'),
            os.path.join(self.src_root, 'sandbox.resources.installed')
        )

    def update(self, revision=None):
        sdk2.svn.Arcadia.update(str(self.arcadia_root), revision=revision)
        self.revision = revision

    def diff(self):
        return sdk2.svn.Arcadia.diff(str(self.arcadia_root))

    def log(self, revision=None, svn_path=None, fullpath=False):
        svn_path_for_logs = svn_path or str(self.arcadia_root)
        return sdk2.svn.Arcadia.log(svn_path_for_logs, revision, fullpath=fullpath, limit=30)

    def info(self, svn_path=None):
        svn_path_for_logs = str(svn_path or self.arcadia_root)
        return sdk2.svn.Arcadia.info(svn_path_for_logs)

    def commit(self, message):
        output = sdk2.svn.Arcadia.commit(
            self.db_root,
            message,
            user=self.svn_user,
        )
        match = COMMIT_RE.search(output)
        return match.group('revision')

    # TODO: resources in parameters
    # import resource types as classes
    def install(self, use_last_resources=False):
        resources_data = {}
        if not use_last_resources:
            resources_data = json.load(open(os.path.join(self.src_root, 'sandbox.resources')))

        binutils = resources_data.get('binutils', {}).get('resource')
        python_venv = resources_data.get('python_venv', {}).get('resource')
        balancer_python_venv = resources_data.get('balancer_python_venv', {}).get('resource')

        self.install_resource('GENCFG_BIN_UTILS', self.src_root, binutils)

        self.install_venv('GENCFG_PYTHON_VENV', self.src_root, python_venv)

        self.install_venv(
            'GENCFG_BALANCER_PYTHON_VENV', os.path.join(self.src_root, 'custom_generators', 'balancer_gencfg'),
            balancer_python_venv
        )

        os.symlink(self.src_root + '/gaux', self.src_root + '/aux')

    def install_db_cache_resource(self):
        self.install_resource('GENCFG_DB_CACHE', os.path.join(self.db_root, 'cache'))

    def gen_sh(self, option=''):
        cmd = '/usr/bin/time ./gen.sh {}'.format(option)
        self.run_process(cmd, 'gen_sh', self.src_root)

    def manipulate_hbf(self, background=False):
        manipulate_hbf_cmd = [
            './utils/common/manipulate_hbf.py', '-a', 'update', '-v'
        ]
        return self.run_process(manipulate_hbf_cmd, 'manipulate_hbf', background=background)

    def populate_gencfg_trunk(self, background=False):
        populate_gencfg_trunk_cmd = ['./utils/mongo/populate_gencfg_trunk.py']
        return self.run_process(populate_gencfg_trunk_cmd, 'populate_gencfg_trunk', background=background)

    def sync_hosts_info_with_mongo(self, section, background=False):
        sync_hosts_info_with_mongo_cmd = [
            './utils/mongo/sync_hosts_info_with_mongo.py',
            '-s', section,
            '--prod-run',
            '--auto-clean',
            '-v'
        ]
        return self.run_process(
            sync_hosts_info_with_mongo_cmd,
            'sync_hosts_info_with_mongo-{}'.format(section),
            background=background
        )

    def precalc_caches(self, option='--no-nanny', background=False):
        precalc_caches_cmd = ['./utils/common/precalc_caches.py', option]
        return self.run_process(precalc_caches_cmd, 'precalc_caches', background=background)

    def diffbuilder(self, gencfg_prev_path, background=False):
        diffbuilder_cmd = [
            './tools/diffbuilder/main.py', '-q', '-o',
            gencfg_prev_path, '-n', self.src_root
        ]
        return self.run_process(diffbuilder_cmd, 'diffbuilder', background=background)

    def populate_searcher_lookup(self, extras, background=False):
        populate_searcher_lookup_cmd = ['/skynet/python/bin/python', './utils/mongo/populate_searcher_lookup.py']
        populate_searcher_lookup_cmd.extend(extras)
        return self.run_process(populate_searcher_lookup_cmd, 'populate_searcher_lookup', background=background)

    def populate_search_map(self, extras, background=False):
        populate_search_map_cmd = ['/skynet/python/bin/python', './utils/mongo/populate_search_map.py']
        populate_search_map_cmd.extend(extras)
        return self.run_process(populate_search_map_cmd, 'populate_search_map', background=background)

    def populate_staff_cache(self, oauth_token, background=False):
        populate_staff_cache_cmd = ['./utils/mongo/populate_staff_cache.py', '--oauth-token', oauth_token]
        return self.run_process(populate_staff_cache_cmd, 'populate_staff_cache', background=background)

    def get_groups_info(self):
        get_groups_info_cmd = ['./utils/sandbox/get_groups_info.py']
        return self.run_process_output(get_groups_info_cmd, 'get_groups_info')

    def get_group_to_aggregate_keys_mapping(self):
        get_group_to_aggregate_keys_mapping_cmd = ['./charts/get_group_to_aggregate_keys_mapping.py']
        return self.run_process_output(get_group_to_aggregate_keys_mapping_cmd, 'get_group_to_aggregate_keys_mapping')

    # RUN SUBPROCESSES

    def run_script(self, *args):
        args = list(args)
        args[0] = os.path.join(self.src_root, args[0])

        env = os.environ.copy()
        env.update({'TMP': self.tmp_root, 'TMPDIR': self.tmp_root})

        with helpers.ProcessLog(task=self.task, logger=args[0].split('/')[-1]) as process_log:  # TODO: logger name
            process = helpers.subprocess.Popen(
                args,
                cwd=self.src_root,
                env=env,
                stdout=process_log.stdout,
                stderr=process_log.stderr,
            )
            process.wait()
            if process.returncode:
                raise Exception('{} failed with code {}'.format(args, process.returncode))  # TODO: exception type; restartable

    def run_process(self, cmd, log_name=None, work_dir=None, env={}, background=False):
        if background:
            return self._run_subprocess(helpers.subprocess.Popen, cmd, log_name, work_dir, env, True)
        return self._run_subprocess(helpers.subprocess.check_call, cmd, log_name, work_dir, env, True)

    def run_process_output(self, cmd, log_name=None, work_dir=None, env={}):
        return self._run_subprocess(helpers.subprocess.check_output, cmd, log_name, work_dir, env, False)

    def _run_subprocess(self, func, cmd, log_name=None, work_dir=None, env={}, stdout_file=True):
        shell = not isinstance(cmd, list)
        full_env = dict(os.environ.copy().items() + env.items())
        work_dir = work_dir or self.src_root
        log_name = log_name or get_log_name_from_cmd(cmd)

        log_stdout = str(self.task.log_path('{}.out.log'.format(log_name)))
        log_stderr = str(self.task.log_path('{}.err.log'.format(log_name)))

        with open(log_stdout, 'w') as stdout, open(log_stderr, 'w') as stderr:
            kwargs = {
                'shell': shell,
                'stderr': stderr,
                'cwd': work_dir,
                'env': full_env
            }

            if stdout_file:
                kwargs['stdout'] = stdout

            return func(cmd, **kwargs)

    @staticmethod
    def resource_by_type(resource_type, resource_id=None):
        if resource_id:
            resource = sdk2.Resource[resource_type].find(
                id=resource_id,
            ).first()
        else:
            resource = sdk2.Resource[resource_type].find().first()

        if resource:
            return str(sdk2.ResourceData(resource).path)
        return None

    @staticmethod
    def resource_path(resource):
        return str(sdk2.ResourceData(resource).path)

    @staticmethod
    def extract(tar_file, dest_path):
        with tarfile.open(tar_file) as a:
            a.extractall(dest_path)

    @staticmethod
    def install_resource(resource_type, install_path, resource_id=None):
        resource = GencfgEnvironment.resource_by_type(resource_type, resource_id)
        if resource:
            GencfgEnvironment.extract(resource, install_path)
        else:
            raise Exception('Cannot download resource {}'.format(resource_type))

    @staticmethod
    def install_venv(resource_type, install_path, resource_id=None):
        GencfgEnvironment.install_resource(resource_type, mkdir_clean(install_path, 'venv'), resource_id)
        return os.path.join(install_path, 'venv', 'venv')

    def create_resource(self, ResourceType, dir_path, resource_dir_name, descr=None):
        if not os.path.exists(dir_path):
            return

        resource = sdk2.ResourceData(ResourceType(
            self.task,
            descr or 'Test commit [{}]. Created from task [{}]'.format(self.revision, self.task.id),
            resource_dir_name
        ))
        resource.path.mkdir(0o755, parents=True, exist_ok=True)

        if os.path.exists(str(resource.path)):
            shutil.rmtree(str(resource.path))

        shutil.move(dir_path, str(resource.path))

        resource.ready()

    def create_text_resource(self, ResourceType, resource_name, data, descr=None):
        resource_data = sdk2.ResourceData(ResourceType(
            self.task,
            descr or 'Created from task [{}]'.format(self.task.id),
            resource_name,
        ))
        resource_data.path.write_bytes(data)
        resource_data.ready()


def mkdir_clean(folder, dir_name):
    """
    If dir_name not exists in folder - make dir
    else remove dir_name and make dir
    """
    dst_dir = os.path.join(folder, dir_name)
    if os.path.exists(dst_dir):
        shutil.rmtree(dst_dir)
    os.mkdir(dst_dir)

    return dst_dir


def get_log_name_from_cmd(cmd):
    if isinstance(cmd, list):
        return cmd[0].split('/')[-1]
    return cmd.split(' ')[0].split('/')[-1]
