import os
import re
import logging
import datetime
import functools


TIME_PATTERN = re.compile(r'(?P<user>\d+.\d+)user\s+(?P<system>\d+.\d+)system\s+(?P<elapsed>\d+:\d+.\d+)elapsed')


def print_log(info):
    logging.info('[{}] USER_LOG: {}'.format(datetime.datetime.now().strftime('%H:%M:%S'), info))


def retry(count, delay):
    def decorator(func):
        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            for i in xrange(count):
                try:
                    return func(*args, **kwargs)
                except Exception as e:
                    if i + 1 == count:
                        raise
                    print_log('EXCEPTION({}) {}: {}'.format(i + 1, type(e).__name__, e))

        return wrapper
    return decorator


def get_datetime_from_timestamp(timestamp):
    for fmt in ('%Y-%m-%dT%H:%M:%S.%ZZ', '%Y-%m-%dT%H:%M:%S.%zZ', '%Y-%m-%dT%H:%M:%S.%fZ', '%Y-%m-%dT%H:%M:%SZ'):
        try:
            return datetime.datetime.strptime(timestamp, fmt) + datetime.timedelta(hours=3)
        except Exception:
            pass
    return None


def disable_in_dry_run(func):
    """
    Do not start func if task started with dry_run == True
    """
    @functools.wraps(func)
    def wrapper(self, *args, **kwargs):
        if self.Parameters.dry_run:
            return None
        return func(self, *args, **kwargs)
    return wrapper


def print_errors_to_info(task, log_dir, depth=0):
    error = ''
    for _, tracebacks in get_error_messages_from_log(log_dir, depth).items():
        for traceback in tracebacks:
            error += '    {}\n'.format(traceback)
    if error:
        task.set_info('FAILED REASONS\n{}'.format(error))


def get_list_errors(log_path, depth=0):
    list_erros = []
    for _, tracebacks in get_error_messages_from_log(log_path, depth).items():
        list_erros.extend(tracebacks)
    return list_erros


def get_error_messages_from_log(log_path, depth=0):
    errors = {}

    log_path = str(log_path)
    list_items = os.listdir(log_path)
    for item in list_items:
        if item in ['debug.log', 'common.log']:
            continue

        item = os.path.join(log_path, item)
        if os.path.isfile(item):
            traceback = parse_tracebacks(item)
            if traceback:
                errors[item] = traceback
        elif os.path.isdir(item) and depth > 0:
            errors.update(get_error_messages_from_log(item, depth - 1))

    return errors


def parse_tracebacks(filepath):
    errors = []
    with open(str(filepath), 'r') as file:
        traceback = False
        for line in file:
            if line.startswith('Traceback'):
                traceback = True
            elif traceback and not line.startswith(' '):
                errors.append(line.strip())
                traceback = False
            elif not traceback and 'failed ... exiting' in line:
                errors.append(line.strip())
    return errors


def _to_seconds(work_time):
    minutes, seconds = 0, 0
    if ':' in work_time:
        parts = work_time.split(':')
        minutes = float(parts[0])
        work_time = parts[1]
    if '.' in work_time:
        parts = work_time.split('.')
        seconds = float(parts[0])
    return minutes * 60 + seconds


def get_bin_time_output(filepath):
    sections = {}

    if not os.path.exists(filepath):
        return sections

    with open(filepath, 'r') as data:
        for line in data:
            line = line.strip()

            if not line:
                continue

            matched = TIME_PATTERN.match(line)
            if matched:
                sections = {k: _to_seconds(v) for k, v in matched.groupdict().iteritems()}
    return sections
