import pymongo
import collections


MongoStorage = collections.namedtuple('MongoStorage', ['uri', 'replicaset'])

HEARTBEAT_C = MongoStorage(
    uri=','.join([
        'myt0-4012.search.yandex.net:27017',
        'myt0-4019.search.yandex.net:27017',
        'sas1-6063.search.yandex.net:27017',
        'sas1-6136.search.yandex.net:27017',
        'vla1-3984.search.yandex.net:27017',
    ]),
    replicaset='heartbeat_mongodb_c'
)
HEARTBEAT_D = MongoStorage(
    uri=','.join([
        'myt0-4013.search.yandex.net:27017',
        'myt0-4020.search.yandex.net:27017',
        'sas4-4916.search.yandex.net:27017',
        'sas4-4917.search.yandex.net:27017',
        'vla1-6006.search.yandex.net:27017',
    ]),
    replicaset='heartbeat_mongodb_d'
)


def commits():
    return topology()['commits']


def topology():
    return pymongo.MongoReplicaSetClient(
        HEARTBEAT_C.uri,
        connectTimeoutMS=15000,
        replicaSet=HEARTBEAT_C.replicaset,
        w='majority',
        wtimeout=15000,
    )['topology_commits']


def normalize_commit(commit):
    commit = str(commit)
    if commit.startswith('r'):
        commit = commit[1:]

    return commit


def get_collection(db_name, collection_name, heartbeat=None):
    heartbeat = heartbeat or HEARTBEAT_C

    return pymongo.MongoReplicaSetClient(
        heartbeat.uri,
        connectTimeoutMS=15000,
        replicaSet=heartbeat.replicaset,
        w='majority',
        wtimeout=15000,
    )[db_name][collection_name]


def find_one(collection, query, sort_key=None, sort_value=None):
    sort_key = sort_key or '$natural'
    sort_value = sort_value or pymongo.ASCENDING
    requests = collection.find(query).sort(sort_key, sort_value).limit(1)
    request = requests[0] if requests.count() else None
    return request


def find_all(collection, query, sort_key=None, sort_value=None, limit=100):
    sort_key = sort_key or '$natural'
    sort_value = sort_value or pymongo.ASCENDING
    requests = collection.find(query).sort(sort_key, sort_value).limit(limit)
    requests = [item for item in requests] if requests.count() else None
    return requests


def update_one(collection, query, updates, upsert=True):
    collection.update_one(query, {"$set": updates}, upsert=upsert)


def get_last_full_commit():
    collection = get_collection('topology_commits', 'tags')
    for tag in find_all(collection, {}, sort_key='commit', sort_value=-1):
        if tag.get('fullstate', False):
            return tag['commit']


def mark_tag_released(self, tag, commit):
    commit = normalize_commit(commit)
    collection = get_collection('topology_commits', 'tags')
    update_one(collection, {'tag': tag}, {'commit': commit})


def mark_searcher_lookup(commit, full=False, diff_to=None):
    commit = normalize_commit(commit)
    collection = get_collection('topology_commits', 'tags')
    if full:
        update_one(collection, {'commit': commit}, {'fullstate': 1})
    elif diff_to:
        update_one(collection, {'commit': commit}, {'diff_to': diff_to})
    else:
        raise Exception('not full and not diff_to')
