# coding: utf8
import os
import re
import json
import shutil
import tarfile
import functools

import sandbox.projects.gencfg.workflow.helpers as helpers

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess
from sandbox.projects.gencfg.resource_types import GENCFG_PYTHON_VENV  # noqa
from sandbox.projects.gencfg.resource_types import GENCFG_BALANCER_PYTHON_VENV  # noqa


ARCADIA_TRUNK_ROOT = 'arcadia:/arc/trunk'
ARCADIA_TAGS_ROOT = 'arcadia:/arc/tags'
GENCFG_SVN_URL = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/gencfg'
COMMIT_RE = re.compile('^Committed revision (?P<revision>\d+)\.$', re.M)
DYNAMIC_MASTER_GROUPS = ('ALL_DYNAMIC', 'ALL_PERSONAL', 'ALL_SOX')
DEFAULT_HOSTS_FILTER = 'lambda host: True'


DEFAULT_VOLUMES = [
    {
        "symlinks": [],
        "guest_mp": "",
        "host_mp_root": "/place",
        "quota": "1.0 Gb"
    },
    {
        "symlinks": [],
        "guest_mp": "/",
        "host_mp_root": "/place",
        "quota": "1.5 Gb"
    },
    {
        "symlinks": ["/usr/local/www/logs"],
        "guest_mp": "/logs",
        "host_mp_root": "/place",
        "quota": "1.0 GB"
    }
]


def procedure_call_logging(func):
    @functools.wraps(func)
    def procedure(self, *args, **kwargs):
        if 'by_subprocess' in kwargs:
            mode = 'native' if kwargs['by_subprocess'] else 'NON-native'
            self.writeln('Run {} in {} mode'.format(func.__name__, mode))
        else:
            self.writeln('Run {} with ({}), ({})'.format(func.__name__, args, kwargs))
        return func(self, *args, **kwargs)
    return procedure


class Gencfg(object):
    def __init__(self, basedir, logpath, revision=None, tag=None):
        self.revision = None if not revision else int(revision)
        self.tag = tag
        self.basedir = basedir
        self.logpath = logpath

        self.__gencfg_native_mode = False

    @property
    def trunk_path(self):
        return os.path.join(self.basedir, 'trunk_root')

    @property
    def arcadia_path(self):
        return os.path.join(self.trunk_path, 'arcadia')

    @property
    def data_path(self):
        return os.path.join(self.trunk_path, 'data')

    @property
    def gencfg_path(self):
        return os.path.join(self.arcadia_path, 'gencfg')

    @property
    def gencfg_db_path(self):
        if self.__gencfg_native_mode:
            return os.path.join(self.gencfg_path, 'db')
        return os.path.join(self.data_path, 'gencfg_db')

    @property
    def gencfg_balancer_path(self):
        return os.path.join(self.gencfg_path, 'custom_generators', 'balancer_gencfg')

    @property
    def gencfg_db_cahce_path(self):
        return os.path.join(self.gencfg_db_path, 'cache')

    @property
    def gencfg_build_path(self):
        return os.path.join(self.gencfg_path, 'build')

    @property
    def svn_user(self):
        # return 'shotinleg'
        return 'zomb-sandbox-rw'

    @property
    def gencfg_default_oauth(self):
        # return {'GENCFG_DEFAULT_OAUTH': '<token-for-debug>'}
        return {'GENCFG_DEFAULT_OAUTH': sdk2.Vault.data('GENCFG', 'gencfg_default_oauth')}

    def writeln(self, text):
        with open(os.path.join(self.logpath, 'gencfg.Gencfg.log'), 'a') as out:
            out.write('{}\n'.format(text))

    @procedure_call_logging
    def checkout(self, by_subprocess=False):
        if by_subprocess:
            return self.__checkout_subprocess()
        return self.__checkout_sandbox_sdk()

    @procedure_call_logging
    def update(self, revision=None, by_subprocess=False):
        if by_subprocess:
            return self.__update_subprocess(revision)
        return self.__update_sandbox_sdk(revision)

    @procedure_call_logging
    def diff(self, revision=None, subpath='', by_subprocess=False):
        if by_subprocess:
            return self.__diff_subprocess(revision, subpath)
        return self.__diff_sandbox_sdk(revision, subpath)

    @procedure_call_logging
    def log(self, revision=None, subpath='', fullpath=False, limit=30, by_subprocess=False):
        if by_subprocess:
            return self.__log_subprocess(revision, subpath, fullpath, limit)
        return self.__log_sandbox_sdk(revision, subpath, fullpath, limit)

    @procedure_call_logging
    def info(self, subpath='', by_subprocess=False):
        if by_subprocess:
            return self.__info_subprocess(subpath)
        return self.__info_sandbox_sdk(subpath)

    @procedure_call_logging
    def commit(self, message, subpath='', by_subprocess=False):
        if by_subprocess:
            return self.__commit_subprocess(message, subpath)
        return self.__commit_sandbox_sdk(message, subpath)

    @procedure_call_logging
    def install(self, use_last_released_resources=False, by_subprocess=False):
        if by_subprocess:
            return self.__install_subprocess(use_last_released_resources)
        return self.__install_sandbox_sdk(use_last_released_resources)

    @procedure_call_logging
    def install_sandbox_resource(self, ResourceType, install_path, resource_id=None):
        return install_sandbox_resource(ResourceType, install_path, resource_id)

    @procedure_call_logging
    def create_sandbox_resource(self, ResourceType, datapath, task, folder_name=None, descr=None):
        return create_sandbox_resource(ResourceType, datapath, task, folder_name, descr)

    @procedure_call_logging
    def run_custom_gencfg_util(self, cmd, env=None, background=False, output=False):
        return run_subprocess(cmd, get_log_name_from_cmd(cmd), self.logpath,
                              cwd=self.gencfg_path, env=env, background=background, output=output)

    @procedure_call_logging
    def precalc_caches(self, option='--no-nanny'):
        cmd = ['./utils/common/precalc_caches.py', option]
        return run_subprocess(cmd, 'precalc_caches', self.logpath, cwd=self.gencfg_path)

    @procedure_call_logging
    def gen_sh(self, *options):
        run_subprocess(['./gen.sh'] + list(options), logpath=self.logpath, cwd=self.gencfg_path)

    @procedure_call_logging
    def manipulate_volumes(self, action, group_name_list, volumes):
        cmd = [
            './utils/common/manipulate_volumes.py',
            '-a', action,
            '-g', helpers.comma_list(group_name_list),
            '-j', json.dumps(volumes) if isinstance(volumes, list) else volumes
        ]
        return run_subprocess(cmd, 'manipulate_volumes-{}'.format(action), self.logpath, cwd=self.gencfg_path)

    @procedure_call_logging
    def manipulate_hbf_ranges(self, action, verbose=True, apply_changes=False, upload_to_rt=False, upload_limit=None):
        cmd = ['./utils/common/manipulate_hbf_ranges.py', '-a', action, '--prefix', '']
        cmd += ['-v'] if verbose else []
        cmd += ['--apply'] if apply_changes else []
        cmd += ['--upload'] if upload_to_rt else []
        cmd += ['--upload-limit', str(upload_limit)] if upload_limit is not None else []

        return run_subprocess(cmd, 'manipulate_hbf_ranges-{}'.format(action), self.logpath,
                              cwd=self.gencfg_path, env=self.gencfg_default_oauth)

    @procedure_call_logging
    def manipulate_dispenser(self, action, verbose=True, apply_changes=False, upload_to_dispenser=False):
        cmd = ['./utils/common/manipulate_dispenser.py', '-a', action]
        cmd += ['-v'] if verbose else []
        cmd += ['--apply'] if apply_changes else []
        cmd += ['--upload'] if upload_to_dispenser else []

        env = self.gencfg_default_oauth
        env.update({'PYTHONPATH': '/skynet'})

        return run_subprocess(cmd, 'manipulate_dispenser-{}'.format(action), self.logpath,
                              cwd=self.gencfg_path, env=env)

    @procedure_call_logging
    def update_card(self, group_name_list, key, value):
        if key in ('owners', 'tags.prj'):
            value = helpers.str_format(value)

        cmd = [
            './utils/common/update_card.py',
            '-g', helpers.comma_list(group_name_list),
            '-k', key,
            '-v', str(value),
            '-y'
        ]
        return run_subprocess(cmd, 'modify_groups_cards-{}'.format(key.replace('.', '_')), self.logpath, cwd=self.gencfg_path)

    @procedure_call_logging
    def update_igroups(self, action, group_name, **kwargs):
        cmd = ['./utils/common/update_igroups.py', '-a', action, '-g', group_name]
        for key, value in kwargs.items():
            cmd.extend(['--{}'.format(key), str(value)])
        return run_subprocess(cmd, 'update_igroup-{}-{}'.format(action, group_name), self.logpath, cwd=self.gencfg_path)

    @procedure_call_logging
    def allocate_in_dynamic(self, **kwargs):
        cmd = ['./optimizers/dynamic/main.py', '--verbose']
        for key, value in kwargs.items():
            if key in ('owners',):
                value = helpers.comma_list(value)
            elif key in ('prj',):
                value = helpers.str_format(value)
            cmd.extend(['--{}'.format(key), str(value)])
        return run_subprocess(cmd, 'optimizers_dynamic_main', self.logpath, cwd=self.gencfg_path)

    @procedure_call_logging
    def recluster_dynamic_group(self, **kwargs):
        cmd = ['./utils/resources/recluster_dynamic_group.py', '--verbose']
        for key, value in kwargs.items():
            cmd.extend(['--{}'.format(key), str(value)])
        return run_subprocess(cmd, 'recluster_dynamic_group', self.logpath, cwd=self.gencfg_path)

    @procedure_call_logging
    def find_hosts_by_params(self, group_name, hosts_count, filename=None, host_dc=['man', 'sas', 'vla', 'myt', 'iva'],
                             host_pu=0, host_memory=0, host_hdd=0, host_ssd=0, host_net=0,
                             free_host_pu=0, free_host_memory=0, free_host_hdd=0, free_host_ssd=0, free_host_net=0, sort_by_free=False,
                             hosts_per_switch=None, hosts_per_queue=None, exclude_groups=None, exclude_hosts=None, hosts_filter=None):
        cmd = [
            './utils/resources/find_hosts.py',
            '-g', group_name,
            '--count', str(hosts_count),
            '--pu', str(host_pu),
            '--memory', str(host_memory),
            '--hdd', str(host_hdd),
            '--ssd', str(host_ssd),
            '--net', str(host_net),
            '--free-pu', str(free_host_pu),
            '--free-memory', str(free_host_memory),
            '--free-hdd', str(free_host_hdd),
            '--free-ssd', str(free_host_ssd),
            '--free-net', str(free_host_net),
            '--dc', helpers.comma_list(host_dc),
            '--hosts-per-queue', str(hosts_per_queue or hosts_count),
            '--hosts-per-switch', str(hosts_per_switch or hosts_count)
        ]
        if exclude_groups:
            cmd += ['--exclude-groups', helpers.comma_list(exclude_groups)]
        if exclude_hosts:
            cmd += ['--exclude-hosts', helpers.comma_list(exclude_hosts)]
        if hosts_filter is not None:
            cmd += ['--filter', hosts_filter]
        if sort_by_free:
            cmd += ['--sort-by-free']

        output = run_subprocess(cmd, 'find_hosts_by_params', self.logpath, cwd=self.gencfg_path, output=True)

        if filename is not None:
            with open(os.path.join(self.gencfg_path, filename), 'a') as hosts:
                hosts.write(output)
        return [x.strip() for x in output.split('\n') if x.strip()]

    @procedure_call_logging
    def show_group_card(self, group_name_list, fields_list):
        """
        Return dict of group card in format
        {
            "GROUP_NAME_1": {
                "field_name_1": <value>,
                "field_name_2": <value>
            },
            "GROUP_NAME_2": {
                "field_name_1": <value>,
                "field_name_2": <value>
            }
        }
        """
        cmd = [
            './utils/common/show_group_card.py',
            '-g', helpers.comma_list(group_name_list),
            '-f', helpers.comma_list(fields_list)
        ]
        return json.loads(run_subprocess(cmd, 'show_group_card', self.logpath, cwd=self.gencfg_path, output=True))

    @procedure_call_logging
    def populate_gencfg_trunk(self, background=False):
        cmd = ['./utils/mongo/populate_gencfg_trunk.py']
        return run_subprocess(cmd, 'populate_gencfg_trunk', self.logpath, background=background)

    @procedure_call_logging
    def sync_hosts_info_with_mongo(self, section, background=False):
        cmd = ['./utils/mongo/sync_hosts_info_with_mongo.py', '-s', section, '--prod-run', '--auto-clean', '-v']
        return run_subprocess(cmd, 'sync_hosts_info_with_mongo-{}'.format(section), self.logpath,
                              cwd=self.gencfg_path, background=background)

    @procedure_call_logging
    def diffbuilder(self, gencfg_prev_path, background=False):
        cmd = ['./tools/diffbuilder/main.py', '-q', '-o', gencfg_prev_path, '-n', self.gencfg_path]
        return run_subprocess(cmd, 'diffbuilder', self.logpath, cwd=self.gencfg_path, background=background)

    def __checkout_sandbox_sdk(self):
        sdk2.svn.Arcadia.checkout(
            ARCADIA_TAGS_ROOT if self.tag else ARCADIA_TRUNK_ROOT,
            str(self.trunk_path),
            depth=sdk2.svn.Svn.Depth.IMMEDIATES,
            revision=self.revision,
        )
        sdk2.svn.Arcadia.update(
            self.gencfg_path,
            depth=sdk2.svn.Svn.Depth.INFINITY,
            revision=self.revision,
        )
        sdk2.svn.Arcadia.update(
            self.gencfg_db_path,
            depth=sdk2.svn.Svn.Depth.INFINITY,
            revision=self.revision,
        )

        # use Path.symlink_to
        os.symlink(self.gencfg_db_path, os.path.join(self.gencfg_path, 'db'))
        self.__gencfg_native_mode = False

    def __checkout_subprocess(self):
        if not os.path.exists(self.arcadia_path):
            os.makedirs(self.arcadia_path)
        run_subprocess(['svn', 'checkout', GENCFG_SVN_URL, self.gencfg_path], 'svn_checkout', self.logpath)
        self.__gencfg_native_mode = True

    def __update_sandbox_sdk(self, revision):
        sdk2.svn.Arcadia.update(self.trunk_path, revision=revision)
        self.revision = revision

    def __update_subprocess(self, revision):
        cmd = ['svn', 'up']
        if revision is not None:
            cmd.extend(['-r', revision])

        run_subprocess(cmd, 'svn_up_code', self.logpath, cwd=self.gencfg_path)

        gencfg_db_path = os.path.join(self.gencfg_path, 'db')
        if os.path.exists(gencfg_db_path):
            run_subprocess(cmd, 'svn_up_db', self.logpath, cwd=gencfg_db_path)
        self.revision = revision

    def __diff_sandbox_sdk(self, revision, subpath):
        svn_path = cast_gencfg_subpath(self.gencfg_path, self.gencfg_db_path, subpath)
        return sdk2.svn.Arcadia.diff(svn_path)

    def __diff_subprocess(self, revision, subpath):
        svn_path = os.path.join(self.gencfg_path, subpath)
        cmd = ['svn', 'diff']
        return run_subprocess(cmd, cwd=svn_path, output=True)

    def __log_sandbox_sdk(self, revision, subpath, fullpath, limit):
        svn_path = cast_gencfg_subpath(self.gencfg_path, self.gencfg_db_path, subpath)
        return sdk2.svn.Arcadia.log(svn_path, revision, fullpath=fullpath, limit=limit)

    def __log_subprocess(self, revision, subpath, fullpath, limit):
        svn_path = os.path.join(self.gencfg_path, subpath)
        cmd = ['svn', 'log', '-l', str(limit)]
        return run_subprocess(cmd, cwd=svn_path, output=True)

    def __info_sandbox_sdk(self, subpath):
        svn_path = cast_gencfg_subpath(self.gencfg_path, self.gencfg_db_path, subpath)
        return sdk2.svn.Arcadia.info(svn_path)

    def __info_subprocess(self, subpath):
        svn_path = os.path.join(self.gencfg_path, subpath)
        cmd = ['svn', 'info']
        return parse_svn_info(run_subprocess(cmd, cwd=svn_path, output=True))

    def __commit_sandbox_sdk(self, message, subpath):
        svn_path = cast_gencfg_subpath(self.gencfg_path, self.gencfg_db_path, subpath)

        self.writeln('__commit_sandbox_sdk commit: {} {} {}'.format(svn_path, message, self.svn_user))
        output = sdk2.svn.Arcadia.commit(svn_path, message, user=self.svn_user)

        self.writeln('__commit_sandbox_sdk output:\n{}'.format(output))

        matched = COMMIT_RE.search(output)
        if matched:
            return int(matched.group('revision'))
        return None

    def __commit_subprocess(self, message, subpath):
        svn_path = os.path.join(self.gencfg_path, subpath)
        cmd = ['svn', 'commit', '-m', message]

        self.writeln('__commit_subprocess commit: {} {}'.format(cmd, svn_path))
        output = run_subprocess(cmd, cwd=svn_path, output=True)

        self.writeln('__commit_subprocess output:\n{}'.format(output))

        matched = COMMIT_RE.search(output)
        if matched:
            return int(matched.group('revision'))
        return None

    def __install_sandbox_sdk(self, use_last_released_resources):
        resources_data = {}
        if not use_last_released_resources:
            with open(os.path.join(self.gencfg_path, 'sandbox.resources'), 'r') as sandbox_resources:
                resources_data = json.load(sandbox_resources)

        binutils_id = resources_data.get('binutils', {}).get('resource')
        wheels_id = resources_data.get('wheels', {}).get('resource')

        self.install_sandbox_resource(sdk2.Resource['GENCFG_BIN_UTILS'], self.gencfg_path, binutils_id)
        self.install_sandbox_resource(sdk2.Resource['GENCFG_WHEELS'], self.gencfg_path, wheels_id)

        cmd = ['./install.sh']
        run_subprocess(cmd, 'install.sh', self.logpath, cwd=self.gencfg_path)

    def __install_subprocess(self, use_last_released_resources):
        cmd = ['./install.sh']
        run_subprocess(cmd, 'install.sh', self.logpath, cwd=self.gencfg_path)


def cast_gencfg_subpath(gencfg_path, gencfg_db_path, subpath):
    casted_path = os.path.join(gencfg_path, subpath)
    if subpath.startswith('db/') or subpath == 'db':
        subpath = '' if subpath != 'db' else subpath[3:]
        casted_path = os.path.join(gencfg_db_path, subpath)
    return casted_path


def get_log_name_from_cmd(cmd):
    if isinstance(cmd, list):
        return cmd[0].split('/')[-1]
    return cmd.split(' ')[0].split('/')[-1]


def parse_svn_info(svn_info_output):
    result = {
        'commit_revision': None
    }

    if not svn_info_output:
        return result

    for line in svn_info_output.split('\n'):
        line = line.strip()

        if not line:
            continue
        elif line.startswith('Revision:'):
            result['commit_revision'] = int(line.split(' ')[-1])

    return result


def run_subprocess_output(cmd, stderr, cwd, env, shell):
    return subprocess.check_output(cmd, stderr=stderr, cwd=cwd, env=env, shell=shell)


def run_subprocess_check(cmd, stdout, stderr, cwd, env, shell):
    return subprocess.check_call(cmd, stdout=stdout, stderr=stderr, cwd=cwd, env=env, shell=shell)


def run_subprocess_popen(cmd, stdout, stderr, cwd, env, shell):
    return subprocess.Popen(cmd, stdout=stdout, stderr=stderr, cwd=cwd, env=env, shell=shell)


def run_subprocess(cmd, logname=None, logpath=None, cwd=None, env=None, background=False, output=False):
    shell = not isinstance(cmd, list)
    logname = logname or get_log_name_from_cmd(cmd)
    logpath = logpath or os.getcwd()
    cwd = cwd or os.getcwd()
    fullenv = os.environ.copy()
    fullenv.update(env or {})

    log_stdout = os.path.join(logpath, '{}.out.log'.format(logname))
    log_stderr = os.path.join(logpath, '{}.err.log'.format(logname))

    with open(log_stderr, 'a') as stderr:
        stderr.write('\n\n[CMD]: {}\n\n'.format(' '.join(map(str, cmd))))
        if output:
            return run_subprocess_output(cmd, stderr, cwd, env, shell)

        with open(log_stdout, 'a') as stdout:
            stdout.write('\n\n[CMD]: {}\n\n'.format(' '.join(map(str, cmd))))
            if background:
                return run_subprocess_popen(cmd, stdout, stderr, cwd, env, shell)
            return run_subprocess_check(cmd, stdout, stderr, cwd, env, shell)


def extract_sandbox_resource(resource_path, extract_path):
    with tarfile.open(resource_path) as archive:
        archive.extractall(extract_path)


def install_sandbox_resource(ResourceType, install_path, resource_id=None):
    resource = ResourceType.find(id=resource_id).first() if resource_id else \
        ResourceType.find().first()

    if resource is None:
        raise RuntimeError('Can not download resource {} {}'.format(ResourceType, ResourceType.__class__.__name__))

    resource = str(sdk2.ResourceData(resource).path)
    extract_sandbox_resource(resource, install_path)


def create_sandbox_resource(ResourceType, datapath, task, folder_name=None, descr=None):
    if not os.path.exists(datapath):
        return

    folder_name = folder_name or os.path.basename(datapath)
    descr = descr or 'Resource created from task [{}]'.fromat(task.id)

    resource = sdk2.ResourceData(ResourceType(task, descr, folder_name))
    resource.path.mkdir(0o755, parents=True, exist_ok=True)

    if os.path.exists(str(resource.path)):
        shutil.rmtree(str(resource.path))
    shutil.move(datapath, str(resource.path))

    resource.ready()


def parse_checkers_from_error_log(filename):
    with open(filename, 'r') as rfile:
        for line in rfile:
            if 'ALL FAILED CHECKERS' not in line:
                continue
            _, checkers = [x.strip() for x in line.split(':')]
            return checkers.split(',')
    return []


def parse_not_enough_resources_from_error_log(filename):
    not_enough = None
    with open(filename, 'r') as rfile:
        for line in rfile:
            if 'hosts with enough resources, while required' not in line:
                continue

            all_line_numbers = map(int, re.findall(r'[0-9]+', line))
            if len(all_line_numbers) == 3:
                not_enough = all_line_numbers[2] - all_line_numbers[1]
    return not_enough


def get_reserved_by_location_name(location_name):
    if location_name.startswith('MSK'):
        return 'MSK_RESERVED'
    return '{}_RESERVED'.format(location_name)


def get_dc_by_location_name(location_name):
    if location_name.startswith('MSK'):
        return location_name.split('_')[1].lower()
    return location_name.lower()
