# coding: utf8
import re

CAST_UNIT_TABLE = {
    'b': {
        'to_bytes': lambda b: b,
        'from_bytes': lambda b: b
    },
    'kb': {
        'to_bytes': lambda kb: kb * 1024.,
        'from_bytes': lambda b: b / 1024.
    },
    'mb': {
        'to_bytes': lambda mb: mb * 1024. * 1024.,
        'from_bytes': lambda b: b / 1024. / 1024.
    },
    'gb': {
        'to_bytes': lambda gb: gb * 1024. * 1024. * 1024.,
        'from_bytes': lambda b: b / 1024. / 1024. / 1024.
    },
    'bit': {
        'to_bytes': lambda bit: bit / 8.,
        'from_bytes': lambda b: b * 8.
    },
    'kbit': {
        'to_bytes': lambda kbit: (kbit * 1024.) / 8.,
        'from_bytes': lambda b: (b * 8.) / 1024.
    },
    'mbit': {
        'to_bytes': lambda mbit: (mbit * 1024. * 1024.) / 8.,
        'from_bytes': lambda b: (b * 8.) / 1024. / 1024.
    },
    'gbit': {
        'to_bytes': lambda gbit: (gbit * 1024. * 1024. * 1024.) / 8.,
        'from_bytes': lambda b: (b * 8.) / 1024. / 1024. / 1024.
    }
}

UNIT_STR_FORMAT = re.compile(r'(?P<value>[0-9]*[.]*[0-9]*)\s*(?P<unit>[A-Za-z]+)')


def comma_list(list_items):
    return ','.join(map(str, list_items))


def str_format(value):
    if isinstance(value, list):
        list_items = map(lambda x: '"{}"'.format(x), value)
        return '[{}]'.format(', '.join(list_items))
    return None


def text_to_list(raw_text, uniq=False, raise_on_not_uniq=False, splitters=None):
    if isinstance(raw_text, list):
        return raw_text
    splitters = splitters or ['\n', ',']

    data_list = [raw_text]
    for splitter in splitters:
        splited = []
        for item in data_list:
            for split_item in item.split(splitter):
                if split_item not in splited:
                    splited.append(split_item)
        data_list = splited

    if uniq:
        uniq_data_list = []
        for item in data_list:
            if item not in uniq_data_list:
                uniq_data_list.append(item)
        return uniq_data_list

    if raise_on_not_uniq:
        uniq_data_list = set(data_list)
        if len(uniq_data_list) != len(data_list):
            raise ValueError('Raw text has not uniq items')

    return data_list


def cast_bytes_size(unit, integer=False, **kwargs):
    if len(kwargs) != 1:
        raise ValueError('Unit value must be only one.')

    current_unit, current_value = kwargs.items()[0]
    unit, current_unit = unit.lower(), current_unit.lower()

    if unit not in CAST_UNIT_TABLE or current_unit not in CAST_UNIT_TABLE:
        error_unit = unit if unit not in CAST_UNIT_TABLE else current_unit
        raise ValueError('{} is not supported, supported {}'.format(error_unit, CAST_UNIT_TABLE.keys()))

    if current_value is None:
        return None

    if unit == current_unit:
        return int(current_value) if integer else current_value

    bytes_value = CAST_UNIT_TABLE[current_unit]['to_bytes'](current_value)
    new_unit_value = CAST_UNIT_TABLE[unit]['from_bytes'](bytes_value)

    return int(new_unit_value) if integer else new_unit_value


def format_bytes_size(unit, integer=False, **kwargs):
    new_unit_value = cast_bytes_size(unit, integer, **kwargs)
    return '{} {}'.format(new_unit_value, unit.upper())


def parse_bytes_size(unit, unit_format, integer=False):
    parsed = UNIT_STR_FORMAT.match(unit_format).groupdict()
    current_value, current_unit = float(parsed['value']), parsed['unit'].lower()

    return cast_bytes_size(unit, integer, **{current_unit: current_value})
