# coding: utf8
import traceback
import sandbox.sdk2 as sdk2
import sandbox.sdk2.parameters as parameters
import sandbox.common.errors as errors
import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
import sandbox.projects.gencfg.helpers as helpers
import sandbox.projects.gencfg.workflow.gencfg as gencfg


class GencfgSdkTaskBase(sdk2.Task):
    """ Base class for make Gencfg SDK tasks """

    class Requirements(sdk2.Task.Requirements):
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 10 * 1024, None)
        client_tags = ctc.Tag.CUSTOM_GENCFG_BUILD

    class Context(sdk2.Task.Context):
        db_revision = -1
        svn_diff = ''
        commit = -1
        exceptions = []
        runtime_error_count = 0

    class Parameters(sdk2.Task.Parameters):
        with parameters.Group('Repository') as repository:
            revision = sdk2.parameters.Integer('Revison', required=False)

        with parameters.Group('Testing') as testing:
            gen_sh_option = sdk2.parameters.String('./gen.sh option', default='run_checks')

        with parameters.Group('Execution') as execution:
            native_mode = sdk2.parameters.Bool('Native mode', default=False)
            dry_run = sdk2.parameters.Bool('Dry run', default=True)

        with parameters.Group('Settings') as execution:
            last_resources = sdk2.parameters.Bool('Use last released resources', default=True)
            use_ramdrive = sdk2.parameters.Bool('Use ramdrive', default=True)

        with parameters.Group('Base params') as base_paarams:
            author = parameters.String('Request author')
            commit_message = parameters.String('Commit message', default='', required=False)

    def fill_task_parameters(self, params):
        pass

    def on_execute(self):
        revision = self.Parameters.revision if self.Parameters.revision > 0 else None
        basedir = str(self.ramdrive.path / 'arc') if self.Parameters.use_ramdrive else str(self.log_path('arc'))

        try:
            # Prepare gencfgenv
            try:
                gencfgenv = gencfg.Gencfg(basedir, str(self.log_path()), revision)
                gencfgenv.checkout(by_subprocess=self.Parameters.native_mode)
                gencfgenv.install(self.Parameters.last_resources, by_subprocess=self.Parameters.native_mode)
            except Exception as e:
                self.save_task_traceback()
                raise RuntimeError('An error occurred while preparing the environment\n[{}]: {}'.format(
                    type(e).__name__, e
                ))

            # Collect information
            try:
                self.Context.db_revision = self.get_current_revision(gencfgenv, self.Parameters)
                self.set_info('Task manipulates revision {}'.format(self.Context.db_revision))
            except Exception as e:
                self.save_task_traceback()
                raise RuntimeError('An error occurred while collecting information\n[{}]: {}'.format(
                    type(e).__name__, e
                ))

            # Task logic
            try:
                self.run_task_logic(gencfgenv, self.Parameters)
            except Exception as e:
                self.save_task_traceback()
                raise errors.TaskFailure('An error occurred while executing task logic\n[{}]: {}'.format(
                    type(e).__name__, e
                ))

            # Collect diff
            try:
                if not self.save_and_check_svn_diff(gencfgenv, self.Parameters):
                    gencfgenv.writeln('Nothing to commit. Group already has selected params.')
                    self.set_info('Nothing to commit. Group already has selected params.')
                    return
            except Exception as e:
                self.save_task_traceback()
                raise RuntimeError('An error occurred while collecting diff\n[{}]: {}'.format(
                    type(e).__name__, e
                ))

            # Commit changes section (try commit 10 times)
            for try_num in xrange(10):
                try:
                    # Merge changes
                    try:
                        gencfgenv.update(by_subprocess=self.Parameters.native_mode)
                    except Exception as e:
                        self.save_task_traceback()
                        raise errors.TemporaryError('Conflict merge\n[{}]: {}'.format(type(e).__name__, e))

                    # Base testing
                    try:
                        self.run_base_testing(gencfgenv, self.Parameters)
                    except Exception as e:
                        self.save_task_traceback()
                        raise RuntimeError('Base test - NOT CHECKED\n[{}]: {}'.format(type(e).__name__, e))

                    # Custom testing
                    try:
                        self.run_custom_testing(gencfgenv, self.Parameters)
                    except Exception as e:
                        self.save_task_traceback()
                        raise RuntimeError('Custom test - NOT CHECKED\n[{}]: {}'.format(type(e).__name__, e))

                    # Merge changes
                    try:
                        gencfgenv.update(by_subprocess=self.Parameters.native_mode)
                    except Exception as e:
                        self.save_task_traceback()
                        raise errors.TemporaryError('Conflict merge\n[{}]: {}'.format(type(e).__name__, e))

                    # Post update testing
                    try:
                        self.run_postupdate_testing(gencfgenv, self.Parameters)
                    except Exception as e:
                        self.save_task_traceback()
                        raise RuntimeError('Post update test - NOT CHECKED\n[{}]: {}'.format(type(e).__name__, e))

                    # CHECK FOR DRY RUN
                    if self.Parameters.dry_run:
                        return

                    # Commit changes
                    try:
                        commit = gencfgenv.commit(
                            message=self.get_commit_message(gencfgenv, self.Parameters),
                            subpath='db',
                            by_subprocess=self.Parameters.native_mode
                        )
                    except Exception as e:
                        self.save_task_traceback()
                        raise errors.TemporaryError('Conflict commit\n[{}]: {}'.format(
                            type(e).__name__, e
                        ))

                    if not commit:
                        return

                    self.Context.commit = int(commit)

                    self.set_info(
                        'Request committed: <a href="https://a.yandex-team.ru/arc/commit/{0}">{0}</a>'.format(commit),
                        do_escape=False
                    )

                    break
                except errors.TemporaryError as e:
                    gencfgenv.writeln('{}'.format(e))

            # Post commit task logic
            try:
                self.run_postcommit_logic(gencfgenv, self.Parameters)
            except Exception as e:
                self.save_task_traceback()
                raise Exception('An error occurred while executing post commit task logic\n[{}]: {}'.format(
                    type(e).__name__, e
                ))
        except RuntimeError as e:
            self.save_task_traceback()
            self.Context.runtime_error_count += 1
            if self.Context.runtime_error_count >= 3:
                raise Exception(e)
            raise errors.TemporaryError(str(e))

    def on_finish(self, prev_status, status):
        if status != ctt.Status.SUCCESS:
            self.save_exceptions()

    def on_break(self, prev_status, status):
        self.save_exceptions()

    def get_current_revision(self, gencfgenv, params):
        output = gencfgenv.info('db', by_subprocess=params.native_mode)
        return int(output['commit_revision'])

    def run_task_logic(self, gencfgenv, params):
        pass

    def save_and_check_svn_diff(self, gencfgenv, params):
        diff = gencfgenv.diff(subpath='db', by_subprocess=params.native_mode)
        self.Context.svn_diff = diff
        return diff

    def run_base_testing(self, gencfgenv, params):
        try:
            gencfgenv.gen_sh(params.gen_sh_option)
        finally:
            gencfgenv.create_sandbox_resource(
                sdk2.Resource['CONFIG_BUILD_LOGS'],
                gencfgenv.gencfg_build_path,
                self,
                'build',
                'Created form r{} by {}'.format(self.Context.db_revision, self.id)
            )

    def run_custom_testing(self, gencfgenv, params):
        pass

    def run_postupdate_testing(self, gencfgenv, params):
        pass

    def get_commit_message(self, gencfgenv, params):
        return u'[{}] {} (committed by {}@) from https://sandbox.yandex-team.ru/task/{}/view'.format(
            type(self).__name__, params.commit_message.encode('ascii', 'ignore'),
            params.author, self.id
        )

    def run_postcommit_logic(self, gencfgenv, params):
        pass

    def save_exceptions(self):
        helpers.print_errors_to_info(self, self.log_path())
        self.Context.exceptions = helpers.get_list_errors(self.log_path())

    def save_task_traceback(self):
        with open(str(self.log_path('{}.traceback'.format(self.__class__.__name__))), 'a') as out:
            out.write(traceback.format_exc(limit=20))
