# -*- coding: utf-8 -*-

import logging
import os
import shutil

import lxml.etree as ET

from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.geoadv.task import GeoadvTask, JugglerStatus, GeoadvEnvironment
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.environments import PipEnvironment


"""class GEOADV_ADVERTS(sdk2.Resource):
    releasers = resource_releasers.tycoon
    releasable = True"""


# class TycoonStyles(sdk2.Resource):
#    """ Tycoon pre-defined styles """


class XsdResolver(ET.Resolver):
    def __init__(self, catalog_dir):
        self.catalogDir = catalog_dir
        self.xsdByUrl = {
            "http://www.w3.org/2001/xml.xsd":  "xml.xsd",
            "http://docs.oasis-open.org/election/external/xAL.xsd": "xAL.xsd"
        }

    def resolve(self, url, pubid, context):
        logging.debug("Resolving %s" % url)
        xsdFile = self.xsdByUrl.get(url)
        if xsdFile:
            logging.debug("Resolved %s" % url)
            return self.resolve_filename(os.path.join(self.catalogDir, xsdFile), context)


class GeoadvAdverts(GeoadvTask):
    """
    Provide Tycoon adverts to Maps Search
    """

    GEN_DIR = "gen"

    class Requirements(sdk2.Requirements):
        environments = [
            PipEnvironment("yandex-yt"),
            PipEnvironment("yandex-yt-yson-bindings-skynet"),
        ]

    class Parameters(GeoadvTask.Parameters):
        description = "Tycoon adverts"

        createRelease = sdk2.parameters.Bool("Create release", default=False)

        validateXml = sdk2.parameters.Bool("Validate XML", default=True)
        with validateXml.value[True]:
            schemasUrl = sdk2.parameters.ArcadiaUrl(
                description="Maps doc schemas",
                required=True,
                default_value="arcadia:/arc/trunk/arcadia/maps/doc/schemas"
            )

        # genChainAds = sdk2.parameters.Bool("Generate chain ads", default=False)

        # newTycoonAdverts = sdk2.parameters.Bool("TYCOON_ADVERTS in the new format", default=False)

        # generateTycoonAds = sdk2.parameters.Bool("Generate TYCOON_ADS", default=False)

        """auxAdverts = sdk2.parameters.LastReleasedResource(
            "Auxiliary adverts (old AdvertInfos format)",
            resource_type=resource_types.TYCOON_AUX_ADVERTS_SOURCE,
            state=(ctr.State.READY),
            required=False,
            multiple=True
        )
        auxAdvertsNew = sdk2.parameters.Resource(
            "Auxiliary adverts (new AdvertData format)",
            resource_type=TycoonAds,
            state=(ctr.State.READY),
            required=False,
            multiple=True
        )"""

        """styles = sdk2.parameters.Resource(
            "Styles",
            resource_type=TycoonStyles,
            state=(ctr.State.READY),
            required=True
        )"""
        # withNavi = sdk2.parameters.Bool("Generate ads for navigator", default=False)

    def validate_adverts_xml(self, xml):
        if not self.Parameters.validateXml:
            return
        schemas_path = sdk2.svn.Arcadia.get_arcadia_src_dir(self.Parameters.schemasUrl)
        parser = ET.XMLParser()
        parser.resolvers.add(XsdResolver(os.path.join(schemas_path, "catalog")))
        schema = ET.XMLSchema(ET.parse(os.path.join(schemas_path, "biz", "advert", "1.x", "advert.xsd"), parser))
        schema.assertValid(xml)

    def ttl_by_env(self):
        if self.Parameters.environment == GeoadvEnvironment.PRODUCTION:
            return 5
        else:
            return 2

    def gen_tycoon_adverts(self):
        logging.info("Generate TYCOON_ADVERTS")

        xml = ET.parse(os.path.join(self.GEN_DIR, "ads.xml"))
        self.validate_adverts_xml(xml)

        res_data = sdk2.ResourceData(resource_types.TYCOON_ADVERTS(
            self, self.Parameters.description, "tycoon_ads", ttl=self.ttl_by_env()
        ))
        res_data.path.mkdir(0o755, parents=True, exist_ok=True)

        xml.write(
            str(os.path.join(str(res_data.path), "ads.xml")),
            encoding="UTF-8",
            xml_declaration=True,
            pretty_print=True
        )

        shutil.copy(os.path.join(self.GEN_DIR, "styles.tar.gz"), str(res_data.path))

        res_data.ready()

    """def gen_tycoon_ads(self, styles_path):
        logging.info("Generate TYCOON_ADS")
        xml = ET.parse(os.path.join(self.GEN_DIR, "ads.xml"))
        aux_resources = self.Parameters.auxAdvertsNew
        if aux_resources:
            for auxSrc in aux_resources:
                logging.info("Mixing aux resource {}".format(auxSrc))
                aux_xml = ET.parse(str(sdk2.ResourceData(auxSrc).path))
                assert aux_xml.getroot().tag == "{http://maps.yandex.ru/advert/1.x}AdvertData"
                xml.getroot().append(aux_xml.getroot())
        self.validate_adverts_xml(xml)
        res_data = sdk2.ResourceData(TycoonAds(
            self, self.Parameters.description, "tycoon_ads", ttl=self.ttl_by_env()
        ))
        res_data.path.mkdir(0o755, parents=True, exist_ok=True)
        ads_path = os.path.join(str(res_data.path), "ads.xml")
        xml.write(ads_path, encoding="UTF-8", xml_declaration=True, pretty_print=True)
        shutil.copy(os.path.join(styles_path, "styles.tar.gz"), str(res_data.path))
        res_data.ready()"""

    def on_execute(self):
        env = os.environ.copy()
        env["ENVIRONMENT_TYPE"] = self.Parameters.environment
        env["YT_PROXY"] = "hahn.yt.yandex.net"
        env["YT_TOKEN"] = sdk2.Vault.data(self.owner, "yt_token")

        # styles_path = str(sdk2.ResourceData(self.Parameters.styles).path)
        # shutil.copy(os.path.join(styles_path, "Styles.xml"), "Styles.xml")

        process.run_process(
            cmd=[
                '/skynet/python/bin/python',
                os.path.join(os.path.dirname(__file__), "generate_adverts.py"),
                self.GEN_DIR
            ],
            log_prefix="generate_adverts",
            wait=True,
            check=True,
            environment=env
        )

        self.gen_tycoon_adverts()

        if self.Parameters.createRelease:
            self.mark_released_resources(self.release_status(), ttl=self.ttl_by_env())
            self.notify_juggler("adverts-export", JugglerStatus.OK)
