#!/usr/bin/env python
# -*- coding: utf-8 -*-

import argparse
import logging
import json
import os
import random
import lxml.etree as ET
import yt.wrapper as yt
import unicodedata
import tarfile
import requests

# ENV_TYPE = os.getenv("ENVIRONMENT_TYPE")
ENV_TYPE = "production"

NS_MAP = {
    None: "http://maps.yandex.ru/advert/1.x",
    "xml": "http://www.w3.org/XML/1998/namespace",
    "xsi": "http://www.w3.org/2001/XMLSchema-instance"
}

empty_advert_text = "Приоритетное размещение на Яндекс.Картах".decode('utf-8')

# TODO fetch translations from tanker
# @see https://tanker.yandex-team.ru/?project=sprav&branch=master&keyset=core%3A_main
# @see https://tanker.yandex-team.ru/?project=direct&branch=to-islands-v4&keyset=core:_BannerFlags
DISCLAIMERS = {
    "abortion": {
        "ru": "Есть противопоказания. Посоветуйтесь с врачом. Возможен вред здоровью.",
        "uk": "Є протипоказання. Порадьтеся з лікарем. Можлива шкода здоров'ю.",
        "en": "Contraindications apply. Consult your doctor. May cause possible adverse side effects.",
        "tr": "Yan etkiler görülebilir. Doktorunuza danışın. Sağlığa zararlı olabilir.",
    },
    "medicine": {
        "ru": "Есть противопоказания. Посоветуйтесь с врачом.",
        "uk": "Є протипоказання. Порадьтеся з лікарем.",
        "en": "Contraindications apply. Consult your doctor.",
        "tr": "Yan etkiler görülebilir. Doktorunuza danışın.",
    },
    "alcohol": {
        "ru": "Чрезмерное потребление вредно.",
        "uk": "Надмірне споживання шкідливе.",
        "tr": "Aşırı tüketilmesi zararlıdır.",
        "en": "Enjoy in moderation.",
    },
    "tobacco": {
        "ru": "Курение вредит вашему здоровью",
        "uk": "Паління шкодить вашому здоров'ю",
        "tr": "Sigara sağlığa zararlıdır",
        "en": "Smoking kills",
    },
    "dietary_supplements": {
        "ru": "БАД",
        "uk": "БАД",
        "tr": "Besin desteği",
        "en": "Dietary supplement",
    },
    "pseudo_weapon": {
        "ru": "Конструктивно сходные с оружием изделия",
        "uk": "Конструктивно схожі на зброю вироби",
        "tr": "Yapısal olarak silaha benzeyen ürünler",
        "en": "Item's design bears resemblance to weapons",
    }
}

DEFAULT_PLACEMENT = ["default", "experiment", "maps"]
NAVI_PLACEMENT = ["navi", "navi/datatesting"]

ORDERS_TEMP_TABLE = "//tmp/adverts_for_export_{:}".format(ENV_TYPE)
ALTAY_TABLE = "//home/sprav/altay/prod/snapshot/company"
COMPANY_NAMES_TABLE = "//tmp/companies_for_geo_advert_{:}".format(ENV_TYPE)
ADVERTS_WITH_COMPANY_NAMES_TABLE = "//tmp/adverts_with_company_names_{:}".format(ENV_TYPE)
PHONES_TABLE = "//home/geoadv/export/production/redirect_phone"

default_icons = [
    "http://storage.mds.yandex.net/get-geoadv/225293/__gp_style_logo_v2.svg",
    "http://storage.mds.yandex.net/get-geoadv/1244551/__gp_style_pin.svg",
    "http://storage.mds.yandex.net/get-geoadv/1336908/__gp_style_logo.svg"
]

mobile_maps_icons = [{
    "name": "Dust",
    "path": ["dust_search", "standard"],
    "anchor": "0.5,0.5",
    "size": "18,18"
}, {
    "name": "DustVisited",
    "path": ["dust_search", "visited"],
    "anchor": "0.5,0.5",
    "size": "18,18"
}, {
    "name": "DustHover",
    "path": ["dust_search", "hover"],
    "anchor": "0.5,0.5",
    "size": "18,18"
}, {
    "name": "Icon",
    "path": ["drop_search", "standard"],
    "anchor": "0.5,0.89",
    "size": "32,38"
}, {
    "name": "IconVisited",
    "path": ["drop_search", "visited"],
    "anchor": "0.5,0.89",
    "size": "32,38"
}, {
    "name": "IconHover",
    "path": ["drop_search", "hover"],
    "anchor": "0.5,0.89",
    "size": "32,38"
}, {
    "name": "Selected",
    "path": ["pin_search", "standard"],
    "anchor": "0.5,0.97",
    "size": "60,68"
}]


def make_tarfile(path, source_dir):
    with tarfile.open(path, "w:gz") as tar:
        tar.add(source_dir, arcname="/")


def download_image(path, url):
    img_data = requests.get(url).content
    file_name = url.split("/")[-1]
    with open(path + "/" + file_name, 'wb') as handler:
        handler.write(img_data)
    return file_name


def get_icon_url(data, icons_meta):
    url = data
    for key in icons_meta["path"]:
        url = url[key]
    return url


def remove_control_characters(s):
    return "".join(ch for ch in s if unicodedata.category(ch)[0] != "C")


def add_element(data, parent_elem, elem_name):
    if elem_name in data and data[elem_name]:
        elem = ET.SubElement(parent_elem, elem_name)
        elem.text = data[elem_name].decode('utf-8')


def add_mandatory_element(data, parent_elem, elem_name, alt_text):
    if elem_name in data or alt_text:
        elem = ET.SubElement(parent_elem, elem_name)
        if elem_name in data:
            elem.text = data[elem_name].decode('utf-8')
        else:
            elem.text = alt_text


def add_stub_text_element(parent_elem, elem_name, text):
    elem = ET.SubElement(parent_elem, elem_name)
    elem.text = remove_control_characters(text)


def is_banner_moderated(advert):
    return "new_moderation" in advert \
           and advert["new_moderation"] \
           and advert["new_moderation"]["status"] == "accepted"


def add_disclaimers(advert, ad):
    if "disclaimers" in ad:
        for disclaimer in ad["disclaimers"]:
            warn = DISCLAIMERS.get(disclaimer, {}).get(ad["lang"])
            if warn:
                ET.SubElement(advert, "disclaimer").text = warn.decode('utf-8')


def add_icons(parent, ad, icons_dir, icons_meta):
    if ad.get("branding_transformed") and ad.get("branding") and ad["branding"]["active"]:
        data = ad["branding_transformed"]

        for icon_data in icons_meta:
            icon_url = get_icon_url(data, icon_data)
            file_name = download_image(icons_dir, icon_url)

            field_name = icon_data["name"]

            style_field = ET.SubElement(parent, "field")
            style_field.text = file_name.replace(".svg", "")
            style_field.attrib["name"] = "style" + field_name

            anchor_field = ET.SubElement(parent, "field")
            anchor_field.text = icon_data["anchor"]
            anchor_field.attrib["name"] = "anchor" + field_name

            size_field = ET.SubElement(parent, "field")
            size_field.text = icon_data["size"]
            size_field.attrib["name"] = "size" + field_name


def add_phones(advert_data, phones):
    override_metadata_element = ET.SubElement(advert_data, "OverrideMetadata")
    phones_element = ET.SubElement(override_metadata_element, "Phones")
    main_used = False
    for phone in phones:
        phone_element = ET.SubElement(phones_element, "Phone")
        phone_element.set("type", phone["type"])
        ET.SubElement(phone_element, "formatted").text = phone["formatted"]
        ET.SubElement(phone_element, "country").text = phone["country_code"]
        ET.SubElement(phone_element, "prefix").text = phone["region_code"]
        ET.SubElement(phone_element, "number").text = phone["number"]
        if not main_used:
            ET.SubElement(phone_element, "info").text = "main phone"
            main_used = True


def map_altay_company(row):
    company_name = None
    for name in row["names"]:
        if name["type"] == "main" and name["value"]["locale"] == "ru":
            company_name = name["value"]["value"]
        if name["type"] == "main" and name["value"]["locale"] == "en" and company_name is None:
            company_name = name["value"]["value"]
    phone_number = None
    for phone in row["phones"]:
        if phone["type"] == "phone" and phone.get("formatted"):
            phone_number = phone["formatted"]
            break
    yield {
        "company_permanent_id": row["permalink"],
        "name": company_name,
        "phone": phone_number,
        "phones": row["phones"]
    }


@yt.with_context
def reduce_with_company_names(key, rows, context):
    advert_rows = []
    company_row = None
    for row in rows:
        if row["@table_index"] == 0:
            advert_rows.append(row)
        else:
            company_row = row
    if company_row is None:
        company_row = {
            "name": None,
            "phone": None,
            "phones": []
        }
    for advert_row in advert_rows:
        advert_row.update({
            "company_name": company_row["name"],
            "company_phone": company_row["phone"],
            "company_phones": company_row["phones"]
        })
        yield advert_row


def add_mobile_navi_log_tag(parent):
    ET.SubElement(parent, "logId").text = "__gp_log"

    ET.SubElement(parent, "style").text = "__gp_style"

    tags_elem = ET.SubElement(parent, "Tags")

    ET.SubElement(tags_elem, "tag").text = "icon:pin"

    ET.SubElement(tags_elem, "tag").text = "icon:logo"

    ET.SubElement(tags_elem, "tag").text = "enable:pin"

    return tags_elem


def get_random_default_order(order):
    default_orders = order["default_orders"]
    if default_orders:
        return default_orders[random.randrange(len(default_orders))]
    else:
        return None


def add_navi_style(parent):
    style = ET.SubElement(parent, "Style")
    style.set("id", "__gp_style")
    icon = ET.SubElement(style, "Icon")
    icon.set("type", "logo")
    icon.text = "__gp_style_logo.svg"

    icon = ET.SubElement(style, "Icon")
    icon.set("type", "pin")
    icon.text = "__gp_style_pin.svg"


def add_adverts(parent, ad, with_empty):
    advert = ET.SubElement(parent, "Advert")
    advert.attrib[ET.QName(NS_MAP["xml"], "lang")] = ad["lang"]
    if is_banner_moderated(ad):
        add_mandatory_element(ad, advert, "title", None)
        if with_empty:
            add_mandatory_element(ad, advert, "text", empty_advert_text)
        else:
            add_mandatory_element(ad, advert, "text", "")
        add_element(ad, advert, "extra")
        add_disclaimers(advert, ad)
        if "age_mark" in ad and ad["age_mark"]:
            age = int(ad["age_mark"])/12
            ET.SubElement(advert, "disclaimer").text = "%d+" % age

        add_element(ad, advert, "url")
    elif with_empty:
        add_stub_text_element(advert, "text", empty_advert_text)
    else:
        add_stub_text_element(advert, "text", "")
    return advert


def add_advert_data(parent, order_id, ad, permanent_id, icons_dir, icons_meta):
    advert_data = ET.SubElement(parent, "AdvertData")
    for place in DEFAULT_PLACEMENT:
        ET.SubElement(advert_data, "pageId").text = place

    ET.SubElement(advert_data, "logId").text = str(order_id)

    tags_elem = ET.SubElement(advert_data, "Tags")

    add_icons(tags_elem, ad, icons_dir, icons_meta)

    advert = add_adverts(advert_data, ad, 1734740040 != permanent_id)

    companies = ET.SubElement(advert_data, "Companies")

    ET.SubElement(companies, "id").text = str(permanent_id)

    return advert


def add_advert_mobile_data(parent, ad, permanent_id, icons_dir, icons_meta):
    advert_data = ET.SubElement(parent, "AdvertData")

    ET.SubElement(advert_data, "pageId").text = "mobile_maps"

    tags_elem = add_mobile_navi_log_tag(advert_data)
    # tags_elem = ET.SubElement(advert_data, "Tags")

    advert = add_adverts(advert_data, ad, 1734740040 != permanent_id)

    companies = ET.SubElement(advert_data, "Companies")

    ET.SubElement(companies, "id").text = str(permanent_id)

    add_icons(tags_elem, ad, icons_dir, icons_meta)

    return advert


def add_advert_navi_data(parent, order, permanent_id, order_data):
    advert_data = ET.SubElement(parent, "AdvertData")
    for place in NAVI_PLACEMENT:
        ET.SubElement(advert_data, "pageId").text = place

    add_mobile_navi_log_tag(advert_data)

    advert = add_navi_adverts(advert_data, order["ads"][0], order_data, 1734740040 != permanent_id)
    companies = ET.SubElement(advert_data, "Companies")

    ET.SubElement(companies, "id").text = str(permanent_id)

    return advert


def add_navi_adverts(parent, ad, order_data, with_empty):

    advert = ET.SubElement(parent, "Advert")
    advert.attrib[ET.QName(NS_MAP["xml"], "lang")] = ad["lang"]
    title_elem = ET.SubElement(advert, "title")
    if order_data["company_name"]:
        title_elem.text = order_data["company_name"].decode('utf-8')

    if is_banner_moderated(ad):
        if ad.get("promo") is not None and ad["promo"].get("title") is not None:
            text = ad["promo"]["title"].decode('utf-8')
        elif ad.get("title") is not None:
            text = ad["title"].decode('utf-8')
        elif with_empty:
            text = empty_advert_text
        else:
            text = ""

        text_elem = ET.SubElement(advert, "text")
        text_elem.text = remove_control_characters(text)

        add_element(ad, advert, "extra")
        add_disclaimers(advert, ad)
        if "age_mark" in ad and ad["age_mark"]:
            age = int(ad["age_mark"])/12
            ET.SubElement(advert, "disclaimer").text = "%d+" % age

        add_element(ad, advert, "url")
    elif with_empty:
        add_stub_text_element(advert, "text", empty_advert_text)
    else:
        add_stub_text_element(advert, "text", "")

    ET.SubElement(advert, "telephone").text = order_data["company_phone"]

    return advert


def add_advert_test_data(parent, order, permanent_id, order_data):
    advert_data = ET.SubElement(parent, "AdvertData")

    ET.SubElement(advert_data, "pageId").text = "test_app"

    ET.SubElement(advert_data, "logId").text = "__gp_log"

    ET.SubElement(advert_data, "style").text = "__gp_style"

    tags_elem = ET.SubElement(advert_data, "Tags")

    ET.SubElement(tags_elem, "tag").text = "icon:pin"

    ET.SubElement(tags_elem, "tag").text = "icon:logo"

    ET.SubElement(tags_elem, "tag").text = "enable:pin"

    style_icon_field = ET.SubElement(tags_elem, "field")
    style_icon_field.text = "kfc_ru_ru_13_5e7def_pin"
    style_icon_field.attrib["name"] = "styleIcon"

    advert = add_test_adverts(advert_data, order["ads"][0], order_data, 1734740040 != permanent_id)
    companies = ET.SubElement(advert_data, "Companies")
    ET.SubElement(companies, "id").text = str(1194340989)

    return advert


def add_test_adverts(parent, ad, order_data, with_empty):
    advert = ET.SubElement(parent, "Advert")
    advert.attrib[ET.QName(NS_MAP["xml"], "lang")] = ad["lang"]
    title_elem = ET.SubElement(advert, "title")
    if order_data["company_name"]:
        title_elem.text = order_data["company_name"].decode('utf-8')

    if is_banner_moderated(ad):
        if ad.get("promo") is not None and ad["promo"].get("title") is not None:
            text = ad["promo"]["title"].decode('utf-8')
        elif ad.get("title") is not None:
            text = ad["title"].decode('utf-8')
        elif with_empty:
            text = empty_advert_text
        else:
            text = ""

        ET.SubElement(advert, "text").text = remove_control_characters(text)

        add_element(ad, advert, "extra")
        add_disclaimers(advert, ad)
        if "age_mark" in ad and ad["age_mark"]:
            age = int(ad["age_mark"])/12
            ET.SubElement(advert, "disclaimer").text = "%d+" % age

        add_element(ad, advert, "url")
        promo = ET.SubElement(advert, "Promo")
        ET.SubElement(promo, "title").text = "Покупайте цветы только у нас".decode('utf-8')
    elif with_empty:
        add_stub_text_element(advert, "text", empty_advert_text)
        promo = ET.SubElement(advert, "Promo")
        ET.SubElement(promo, "title").text = "Покупайте цветы только у нас".decode('utf-8')
    else:
        add_stub_text_element(advert, "text", "")

    ET.SubElement(advert, "telephone").text = order_data["company_phone"]

    return advert


def generate_tycoon_ads(orders_table, ads_file, icons_dir):

    ad_data_list = ET.Element("AdvertDataList", nsmap={None: NS_MAP[None], "xml": NS_MAP["xml"]})
    # styles_list = ET.parse("Styles.xml").getroot()

    # add_navi_style(styles_list)
    """yt.copy(orders_table, ORDERS_TEMP_TABLE, force=True)
    yt.run_sort(ORDERS_TEMP_TABLE, sort_by="company_permanent_id")
    yt.run_map(map_altay_company, ALTAY_TABLE, COMPANY_NAMES_TABLE, format=yt.JsonFormat())
    yt.run_sort(COMPANY_NAMES_TABLE, sort_by="company_permanent_id")
    yt.run_reduce(
        reduce_with_company_names,
        [ORDERS_TEMP_TABLE, COMPANY_NAMES_TABLE],
        ADVERTS_WITH_COMPANY_NAMES_TABLE,
        format=yt.JsonFormat(control_attributes_mode="row_fields"),
        reduce_by="company_permanent_id"
    )"""

    input_table = ADVERTS_WITH_COMPANY_NAMES_TABLE
    # ad_data_list.append(styles_list)
    ET.SubElement(ad_data_list, "MenuItems")

    override_phones = {}
    for phone in yt.read_table(PHONES_TABLE, format="yson"):
        override_phones[phone["original_number"]] = json.loads(phone["redirect_phone_details"])["phone"][0]
    # test_exported = False
    for order in yt.read_table(input_table, format="yson"):
        default_order = get_random_default_order(order)
        if default_order:

            company_phones = []
            if default_order["override_phones"]:
                for phone in order["company_phones"]:
                    company_src_phone_number = phone["country_code"] + phone["region_code"] + phone["number"]
                    override_phone = override_phones.get(company_src_phone_number)
                    if not override_phone:
                        company_src_phone_number = "+" + company_src_phone_number
                        override_phone = override_phones.get(company_src_phone_number)
                    if override_phone:
                        phone["country_code"] = override_phone["country_code"]
                        phone["region_code"] = override_phone["region_code"]
                        phone["number"] = override_phone["number"]
                        phone["formatted"] = override_phone["formatted"]
                    company_phones.append(phone)

            maps_advert = add_advert_data(
                parent=ad_data_list,
                order_id=default_order["order_id"],
                ad=default_order["ads"][0],
                permanent_id=order["company_permanent_id"],
                icons_dir=icons_dir,
                icons_meta=mobile_maps_icons
            )

            mobile_maps_advert = add_advert_mobile_data(
                parent=ad_data_list,
                ad=default_order["ads"][0],
                permanent_id=order["company_permanent_id"],
                icons_dir=icons_dir,
                icons_meta=mobile_maps_icons
            )

            if len(company_phones) > 0:
                order["company_phone"] = company_phones[0]["formatted"]

            navi_advert = add_advert_navi_data(
                parent=ad_data_list,
                order=default_order,
                permanent_id=order["company_permanent_id"],
                order_data=order
            )

            """if not test_exported:
                add_advert_test_data(
                    parent=ad_data_list,
                    order=default_order,
                    permanent_id=order["company_permanent_id"],
                    order_data=order
                )
                test_exported = True"""

            if len(company_phones) > 0:
                for advert in [maps_advert, navi_advert, mobile_maps_advert]:
                    add_phones(advert, company_phones)

    ET.ElementTree(ad_data_list).write(ads_file, encoding='utf-8', xml_declaration=True, pretty_print=True)


def main(args):

    orders_table = "//home/sprav/tycoon/{}/{}".format(ENV_TYPE, "tmp/combined_ad_order")
    os.mkdir(args.outputDir, 0755)

    icons_dir = os.path.join(args.outputDir, "img")
    os.mkdir(icons_dir, 0755)
    tar_file = os.path.join(args.outputDir, "styles.tar.gz")

    generate_tycoon_ads(
        orders_table=orders_table,
        ads_file=os.path.join(args.outputDir, "ads.xml"),
        icons_dir=icons_dir
    )

    for icon_url in default_icons:
        download_image(icons_dir, icon_url)

    make_tarfile(tar_file, icons_dir)


if __name__ == "__main__":
    yt.config['pickling']['module_filter'] = lambda module: 'hashlib' not in getattr(module, '__name__', '') \
                                                            and 'lxml' not in getattr(module, '__name__', '')

    parser = argparse.ArgumentParser(description='Generates adverts XML files')
    parser.add_argument('outputDir', help='Output directory')

    yt.update_config({'pickling': {'python_binary': '/skynet/python/bin/python'}})
    logging.basicConfig(level=logging.INFO)
    main(parser.parse_args())
