# -*- coding: utf-8 -*-

import os.path

from sandbox import common
from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.common.types.task import Status
from sandbox.projects.maps.common.juggler_alerts import (
    TaskJugglerReportWithParameters
)
from sandbox.projects.yql.RunYQL2 import RunYQL2

YQL_TIMEOUT = 60 * 30


class GeoAdvPriorityCompaniesExport(TaskJugglerReportWithParameters):
    class Parameters(TaskJugglerReportWithParameters.Parameters):
        destination_root = sdk2.parameters.String(
            'Destination YT root',
            default='//home/geoadv/export/production',
        )
        western_union_chain_permalink = sdk2.parameters.String(
            'Permalinks for Western Union chain to unprotect it',
            default='[805550707]',
        )
        allowed_not_protected_campaigns = sdk2.parameters.String(
            'Allowed not protected campaigns',
            default='[1292184, 1139090, 3034134]',
        )
        protected_critical_change_rate = sdk2.parameters.Float(
            'Critical change rate for protected export',
            default=0.1,
        )
        unprotected_critical_change_rate = sdk2.parameters.Float(
            'Critical change rate for unprotected export',
            default=0.1,
        )
        disable_real_campaigns_check = sdk2.parameters.Bool(
            'Disable real campaigns checks in case geoadv_campaign table is in an inconsistent state',
            default=False,
        )

        yql_vault = sdk2.parameters.Vault('YQL vault', default='YQL_TOKEN')

    def _read_query(self, file_name):
        path = os.path.join(os.path.dirname(__file__), file_name)
        if common.import_hook.inside_the_binary():
            from library.python import resource
            resource_data = resource.find(path)
            if resource_data is not None:
                return resource_data
        with open(path, 'r') as f:
            resource_data = f.read()
        return resource_data

    def on_execute(self):
        with self.memoize_stage.execute_yql:
            sub_task = RunYQL2(
                self,
                description='Export priority companies',
                create_sub_task=False,
                query=self._read_query('export.yql'),
                use_v1_syntax=True,
                trace_query=True,
                owner=self.Parameters.owner,
                publish_query=True,
                yql_token_vault_name=self.Parameters.yql_vault,
                custom_placeholders={
                    '%DST_ROOT%': self.Parameters.destination_root,
                    '%WU_CHAIN_PERMALINKS%': self.Parameters.western_union_chain_permalink,
                    '%NOT_PROTECTED_CAMPAIGNS%': self.Parameters.allowed_not_protected_campaigns,
                    '%PROTECTED_CRITICAL_RATE%': str(self.Parameters.protected_critical_change_rate),
                    '%UNPROTECTED_CRITICAL_RATE%': str(self.Parameters.unprotected_critical_change_rate),
                    '%DISABLE_REAL_CAMPAIGNS_CHECK%': 'TRUE' if self.Parameters.disable_real_campaigns_check else 'FALSE',
                }
            )
            sub_task.enqueue()
            raise sdk2.WaitTask(
                sub_task,
                (Status.Group.FINISH, Status.Group.BREAK),
                timeout=YQL_TIMEOUT
            )

        child_tasks = self.find()
        for task in child_tasks:
            if task.status not in Status.Group.SUCCEED:
                raise TaskError('Child task failed')
