import datetime
import logging
import time

from sandbox import sdk2
from sandbox.common import utils
from sandbox.sandboxsdk.environments import PipEnvironment

console = logging.StreamHandler()
console.setLevel(logging.INFO)

formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
console.setFormatter(formatter)

log = logging.getLogger('release-permalinks-table')
log.addHandler(console)

hahn = None
arnold = None
clusters = None


def init_releasers(token, dest_clusters):
    global yt
    global yson
    global YPath

    import yt.wrapper as yt
    import yt.yson as yson
    from yt.wrapper.ypath import YPath

    global hahn, arnold, clusters

    hahn = create_client('hahn', token)
    arnold = create_client('arnold', token)
    clusters = list([create_client(c, token) for c in dest_clusters])


def create_client(proxy, token):
    class WrappedYtClient(yt.YtClient):
        def __str__(self):
            return str(self.config['proxy']['url'])

        __repr__ = __str__

    return WrappedYtClient(proxy=proxy, token=token)


class BaseReleaseProcess:
    def __init__(self, env, folder_name):
        if env is None:
            raise ValueError("No env")
        if folder_name is None:
            raise ValueError("No folder")
        now = datetime.datetime.utcnow()
        self.env = env
        self.folder_name = folder_name
        self.out_table_name = now.strftime('%Y-%m-%dT%H:%M:%S%Z')
        self.time_started = now

    def build_schema(self):
        raise NotImplementedError

    def sort_keys(self):
        raise NotImplementedError

    def static_tables_directory(self):
        return YPath('//home/geoadv/geoprod_backend/{}/{}'.format(self.env, self.folder_name))

    def dynamic_tables_directory(self):
        return YPath('//home/geoadv_stat/{}/{}'.format(self.env, self.folder_name))

    def static_table(self):
        return self.static_tables_directory().join(self.out_table_name)

    def mirror_table_to_arnold(self, table):
        return arnold.run_remote_copy(source_table=table, destination_table=table, sync=True, cluster_name='hahn')

    def dynamic_table(self):
        return self.dynamic_tables_directory().join(self.out_table_name)

    def tmp_seneca_table(self):
        return self.dynamic_tables_directory().join('tmp')

    def dynamic_tables_link(self):
        return self.dynamic_tables_directory().join('current')

    def sort_table(self, input_table, output_table):
        log.info('sorting table {} -> {} on {}'.format(input_table, output_table, str(hahn)))
        hahn.create(type='table', path=output_table, recursive=True, ignore_existing=False,
                    attributes={"schema": self.build_schema()})
        hahn.run_sort(source_table=input_table, destination_table=output_table, sync=True, sort_by=self.sort_keys())

    def send_to_cluster(self, cluster, from_table, tmp_table):
        log.info('sending {} to {} on {}'.format(from_table, tmp_table, cluster))
        d = yt.ypath.ypath_dirname(tmp_table)
        log.info('creating {} on {}'.format(d, cluster))
        cluster.create(type='map_node', path=d, ignore_existing=True)
        log.info('removing old {} on {}'.format(tmp_table, cluster))
        cluster.remove(tmp_table, force=True)
        log.info('creating {} on {}'.format(tmp_table, cluster))
        cluster.create(type='table', path=tmp_table, ignore_existing=False)
        log.info('running to {}'.format(cluster))
        cluster.run_remote_copy(source_table=from_table, destination_table=tmp_table, sync=True, cluster_name='hahn')
        log.info('copied')

    def mount(self, cluster, table):
        log.info('mounting {} on {}'.format(table, cluster))
        yt.mount_table(table, client=cluster, sync=True)

    def get_previous_table(self, cluster):
        try:
            link_path = self.dynamic_tables_link()
            if not yt.exists(link_path, client=cluster):
                return None
            return YPath(cluster.get_attribute(str(link_path) + '&', 'target_path'))
        except Exception as e:
            log.warning('Cannot get prev table', exc_info=True)
            return None

    def unmount(self, cluster, path):
        try:
            cluster.unmount_table(path)
        except Exception:
            log.warning('Cannot unmount prev table {}'.format(path), exc_info=True)
            return None

    def mark_for_removing(self, cluster, path):
        try:
            exp_time = self.time_started + datetime.timedelta(days=7)
            yt.set_attribute(str(path), 'expiration_time', exp_time.isoformat(), client=cluster)
        except Exception:
            log.warning('Cannot mark for removing prev table {}'.format(path), exc_info=True)

    def convert_to_dynamic(self, cluster, from_table, to_table):
        log.info('Converting from static {} to dynamic {}'.format(from_table, to_table))
        attrs = {
            'optimize_for': 'scan',
            'schema': self.build_schema(),
            'in_memory_mode': 'uncompressed',
            'primary_medium': 'ssd_blobs',
            'tablet_cell_bundle': 'geoadv_stat',
            'strict': True,
            'unique_keys': True,
        }
        cluster.create('table',
                       path=to_table,
                       recursive=False,
                       ignore_existing=False,
                       attributes=attrs
                       )
        cluster.run_merge(source_table=from_table, destination_table=to_table, sync=True, mode='sorted',
                          table_writer={
                              'block_size': 256 * 1024,
                              'desired_chunk_size': 100 * 1024 * 1024
                          })
        cluster.alter_table(path=to_table, dynamic=True, schema=self.build_schema())

    def switch_link_to(self, target_path, cluster):
        link_path = self.dynamic_tables_link()
        log.info('switching link {} -> {} on {}'.format(link_path, target_path, cluster))
        yt.link(target_path=target_path, link_path=link_path, client=cluster, force=True)

    def release_new_table(self, path):

        log.info('starting release {} as budgets for {}'.format(path, self.env))
        path = YPath(path)
        now = datetime.datetime.utcnow()

        static_table = self.static_table()
        self.sort_table(path, static_table)
        self.mirror_table_to_arnold(static_table)

        tmp_table = self.tmp_seneca_table()
        for cluster in clusters:
            self.send_to_cluster(cluster, static_table, tmp_table)

        dynamic_table = self.dynamic_table()

        for cluster in clusters:
            self.convert_to_dynamic(cluster, from_table=tmp_table, to_table=dynamic_table)

        for cluster in clusters:
            self.mount(cluster, dynamic_table)

        log.info('WORKAROUND (still needed?): Waiting 3 minutes after mounting...')
        time.sleep(3 * 60)
        log.info('Continuing...')

        previous_table = self.get_previous_table(clusters[0])

        for cluster in clusters:
            self.switch_link_to(target_path=dynamic_table, cluster=cluster)

        log.info('*****************************')
        log.info('****** CONGRATULATIONS ******')
        log.info('*****************************')
        log.info('****** Table RELEASED! ******')
        log.info('*****************************')

        if previous_table is not None and previous_table != dynamic_table:
            for cluster in clusters:
                self.unmount(cluster, previous_table)
                self.mark_for_removing(cluster, previous_table)

        log.info('Done.')


class PermalinksTableReleaseProcess(BaseReleaseProcess):
    def __init__(self, env):
        BaseReleaseProcess.__init__(self, env=env, folder_name='budgets')

    def build_schema(self):
        schema = yson.YsonList([
            {'name': 'permalink', 'type': 'int64', 'required': True, 'sort_order': 'ascending'},
            {'name': 'is_arbitrate_allowed', 'type': 'boolean', 'required': True},
            {'name': 'is_priority_allowed', 'type': 'boolean', 'required': True},
            {'name': 'priority', 'type': 'any', 'required': False},
            {'name': 'arbitrage_min', 'type': 'any', 'required': False},
            {'name': 'arbitrage_opt', 'type': 'any', 'required': False},
            {'name': 'arbitrage_max', 'type': 'any', 'required': False},
            {'name': 'predict_version', 'type': 'string', 'required': False},
            {'name': 'exp', 'type': 'any', 'required': False},
            {'name': 'priority_min', 'type': 'any', 'required': False},
            {'name': 'priority_opt', 'type': 'any', 'required': False},
        ])
        schema.attributes = {
            'strict': True,
            'unique_keys': True,
        }
        return schema

    def sort_keys(self):
        return ['permalink']


class FallbackTableReleaseProcess(BaseReleaseProcess):
    def __init__(self, env):
        BaseReleaseProcess.__init__(self, env=env, folder_name='fallback_budgets')

    def build_schema(self):
        schema = yson.YsonList([
            {'name': 'rubric', 'type': 'int64', 'required': True, 'sort_order': 'ascending'},
            {'name': 'region', 'type': 'int64', 'required': True, 'sort_order': 'ascending'},
            {'name': 'priority', 'type': 'any', 'required': False},
            {'name': 'arbitrage_min', 'type': 'any', 'required': False},
            {'name': 'arbitrage_opt', 'type': 'any', 'required': False},
            {'name': 'arbitrage_max', 'type': 'any', 'required': False},
            {'name': 'predict_version', 'type': 'string', 'required': False},
            {'name': 'priority_min', 'type': 'any', 'required': False},
            {'name': 'priority_opt', 'type': 'any', 'required': False},
        ])
        schema.attributes = {
            'strict': True,
            'unique_keys': True,
        }
        return schema

    def sort_keys(self):
        return ['rubric', 'region']


class SubsidyBudgetsReleaseProcess(BaseReleaseProcess):
    def __init__(self, env):
        BaseReleaseProcess.__init__(self, env=env, folder_name='subsidy_budgets')

    def build_schema(self):
        schema = yson.YsonList([
            {'name': 'permalink', 'type': 'int64', 'required': True, 'sort_order': 'ascending'},
            {'name': 'subsidy', 'type': 'any', 'required': False},
            {'name': 'predict_version', 'type': 'string', 'required': True},
            {'name': 'exp', 'type': 'any', 'required': False}
        ])
        schema.attributes = {
            'strict': True,
            'unique_keys': True,
        }
        return schema

    def sort_keys(self):
        return ['permalink']


class SubsidyBudgetsFallbackReleaseProcess(BaseReleaseProcess):
    def __init__(self, env):
        BaseReleaseProcess.__init__(self, env=env, folder_name='subsidy_fallback')

    def build_schema(self):
        schema = yson.YsonList([
            {'name': 'main_rubric_id', 'type': 'int64', 'required': True, 'sort_order': 'ascending'},
            {'name': 'subsidy', 'type': 'any', 'required': False},
            {'name': 'predict_version', 'type': 'string', 'required': True}
        ])
        schema.attributes = {
            'strict': True,
            'unique_keys': True,
        }
        return schema

    def sort_keys(self):
        return ['main_rubric_id']


class PriceTableType(utils.Enum):
    utils.Enum.lower_case()
    PERMALINKS = None
    FALLBACK = None
    SUBSIDY_BUDGETS = None
    SUBSIDY_FALLBACKS = None


class GeoAdvEnv(utils.Enum):
    utils.Enum.lower_case()
    PRODUCTION = None
    TESTING = None


def release_price(env, table_type, path, yt_token, dest_clusters):
    init_releasers(yt_token, dest_clusters)
    if table_type == PriceTableType.PERMALINKS:
        table_manager_type = PermalinksTableReleaseProcess
    elif table_type == PriceTableType.FALLBACK:
        table_manager_type = FallbackTableReleaseProcess
    elif table_type == PriceTableType.SUBSIDY_BUDGETS:
        table_manager_type = SubsidyBudgetsReleaseProcess
    elif table_type == PriceTableType.SUBSIDY_FALLBACKS:
        table_manager_type = SubsidyBudgetsFallbackReleaseProcess
    else:
        raise ValueError('unknown table type: {}'.format(table_type))
    table_manager = table_manager_type(env)
    table_manager.release_new_table(path)


class GeoadvReleasePrices(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        environment = sdk2.parameters.String(
            "Environment",
            choices=list((v, v) for v in GeoAdvEnv),
            default_value=GeoAdvEnv.TESTING,
            required=True
        )
        table_path = sdk2.parameters.String("TablePath", required=True)
        table_type = sdk2.parameters.String(
            "TableType",
            choices=list((v, v) for v in PriceTableType),
            required=True
        )
        dest_clusters = sdk2.parameters.List(
            'DestClusters',
            value_type=sdk2.parameters.String,
            default=['seneca-vla', 'seneca-sas', 'seneca-man']
        )

    class Requirements(sdk2.Requirements):
        environments = [
            PipEnvironment("yandex-yt"),
            PipEnvironment("yandex-yt-yson-bindings-skynet"),
        ]

    def on_execute(self):
        env = self.Parameters.environment
        table_type = self.Parameters.table_type
        table_path = self.Parameters.table_path
        dest_clusters = self.Parameters.dest_clusters
        yt_token = sdk2.yav.Secret('sec-01d68afgz53swtrpk5b4dx0s7s').data()['GEOADV_YT_TOKEN']
        release_price(env, table_type, table_path, yt_token, dest_clusters)
