# coding: utf-8

import logging
import re
import time

from sandbox import sdk2
from sandbox.common import utils
from sandbox.common.types.task import ReleaseStatus
from sandbox.projects.common.juggler import jclient
from sandbox.projects.common.yabs.graphite import Graphite, metric_point


class JugglerStatus(utils.Enum):
    OK = None
    WARN = None
    CRIT = None


GRAPHITE_SERVERS = [
    "gr.mega-graphite-sas.search.yandex.net:2024",
    "gr.mega-graphite-man.search.yandex.net:2024",
]


class GeoadvEnvironment(utils.Enum):
    utils.Enum.lower_case()
    PRODUCTION = None
    TESTING = None


RELEASE_STATUS_BY_ENV = {
    GeoadvEnvironment.TESTING: ReleaseStatus.TESTING,
    GeoadvEnvironment.PRODUCTION: ReleaseStatus.STABLE,
}


JUGGLER_HOST_BY_ENV = {
    GeoadvEnvironment.TESTING: "tycoon-back-test",
    GeoadvEnvironment.PRODUCTION: "tycoon-back-prod",
}

ALTAY_ENV_BY_ENV = {
    GeoadvEnvironment.TESTING: "testing",
    GeoadvEnvironment.PRODUCTION: "prod",
}


class GeoadvTask(sdk2.Task):
    """
        Basic tycoon task
    """

    class Parameters(sdk2.Task.Parameters):
        environment = sdk2.parameters.String(
            "Environment",
            choices=list((v, v) for v in GeoadvEnvironment),
            default_value=GeoadvEnvironment.TESTING,
            required=True
        )

    def release_status(self):
        return RELEASE_STATUS_BY_ENV.get(self.Parameters.environment)

    def notify_juggler(self, service, status, value=None):
        host = JUGGLER_HOST_BY_ENV.get(self.Parameters.environment)
        jclient.send_events_to_juggler(host, service, status, str(value) if value else str(self.id))
        logging.info("Notification to juggler %s %s %s", host, service, status)

    def send_metric(self, name, value):
        metric_name = "geo.tycoon.{}.sandbox.server.{}".format(self.Parameters.environment, name)
        metric = metric_point(metric_name, value, time.time())
        Graphite(hosts=GRAPHITE_SERVERS).send([metric])
        logging.info("Metric to graphite %s", metric)

    def yql_query(self, template, **substitutions):
        subst = {
            "ENV": self.Parameters.environment,
            "ALTAY_ENV": ALTAY_ENV_BY_ENV.get(self.Parameters.environment),
            "EARTH_REGION_ID": str(10000),
        }
        subst.update(substitutions)
        return re.sub(
            r"\{\{(\w+)\}\}",
            lambda match: subst[match.group(1)] if match.group(1) in subst else "",
            template
        )
