from datetime import datetime

from sandbox.projects.common.binary_task import LastBinaryTaskRelease, binary_release_parameters
from sandbox.projects.geoadv_ranking.lib.tokens import get_token_from_yav
from sandbox.sdk2 import Task
from sandbox.sdk2.parameters import YavSecret, RadioGroup, String, JSON


class PublishTableArtifactsForCompetitorAdvertLogsProcessing(LastBinaryTaskRelease, Task):
    class Parameters(Task.Parameters):
        ext_params = binary_release_parameters(stable=True)
        yt_token = YavSecret('Yav secret with yt token.', required=True)
        yt_cluster = RadioGroup('Yt cluster', choices=[('Hahn', 'hahn'), ('Arnold', 'arnold')], default='hahn')
        storage = String('Path to yt document with last published dates for corresponding directories.', required=True)
        directories_to_watch = JSON('Config of directories tp publish new table from.', required=True)
        reactor_token = YavSecret('Yav secret with reactor token.', required=True)

    def on_execute(self):
        from reactor_client.reactor_api import ReactorAPIClientV1
        from reactor_client.reactor_objects import ArtifactIdentifier, NamespaceIdentifier, Metadata
        from yt.wrapper import YtClient, ypath_join

        yt_client = YtClient(
            proxy=self.Parameters.yt_cluster,
            token=get_token_from_yav(secret_data=self.Parameters.yt_token),
        )
        reactor_client = ReactorAPIClientV1(
            base_url='https://reactor.yandex-team.ru',
            token=get_token_from_yav(secret_data=self.Parameters.reactor_token),
        )

        with yt_client.Transaction():
            last_published = yt_client.get(self.Parameters.storage)

            for path_config in self.Parameters.directories_to_watch:
                path = path_config['path']
                last_date = yt_client.list(path)[-1]
                if path not in last_published or last_date > last_published[path]:
                    last_published[path] = last_date

                artifact_id = ArtifactIdentifier(
                    namespace_identifier=NamespaceIdentifier(namespace_path=path_config['artifact_id']),
                )
                ypath_meta = Metadata(
                    type_='/yandex.reactor.artifact.YtPathArtifactValueProto',
                    dict_obj={'cluster': self.Parameters.yt_cluster, 'path': ypath_join(path, last_date)},
                )
                reactor_client.artifact_instance.instantiate(
                    artifact_identifier=artifact_id,
                    metadata=ypath_meta,
                    user_time=datetime.strptime(last_date, '%Y-%m-%d'),
                    create_if_not_exist=True,
                )

            yt_client.set(self.Parameters.storage, last_published)
