import hashlib
import logging

from sandbox.common.proxy import OAuth
from sandbox.common.rest import Client
from sandbox.projects.common.binary_task import LastBinaryTaskRelease, binary_release_parameters
from sandbox.projects.geoadv_ranking.lib.tokens import (
    get_token_from_yav, SANDBOX_ROBOT_EXTDATA_SECRET_ID, NANNY_ROBOT_EXTDATA_SECRET_ID, YAV_ROBOT_EXTDATA_SECRET_ID
)
from sandbox.sdk2 import Task


logging.basicConfig()


INSTANCECTL_UPPER_MD5 = '70548c14b9cf84a2994520aa29a0f207'
PROD_CONFIGS_RESOURCE_ID = 3110921238

NANNY_URL = 'https://nanny.yandex-team.ru'


class SomethingChangedException(Exception):
    pass


def filter_by_local_path(resources_list, to_delete):
    return list(filter(lambda r: r['local_path'] not in to_delete, resources_list))


def get_md5(text):
    return hashlib.md5(text.encode('utf-8')).hexdigest()


def find_last_released_resource(sandbox_client, resource_type):
    response = sandbox_client.resource.read(type=resource_type, attrs={'released': 'stable'}, order='-id', limit=1)

    if not response['items']:
        raise RuntimeError('Failed to find any released {resource_type} resource.'.format(resource_type=resource_type))

    resource = response['items'][0]
    return resource['id'], resource['task']['id']


ADDRS_UPPER_NEW_RESOURCE_LOCAL_PATHS = ('update_its', 'configs')
ADDRS_UPPER_NEW_RESOURCE_TYPES = ('GEOPRODUCT_ADDRS_ITS_UPDATER', 'GEOPRODUCT_ADDRS_CONFIGS')


class SynchronizeAddrsGeoproductBetas(LastBinaryTaskRelease, Task):
    class Parameters(Task.Parameters):
        ext_params = binary_release_parameters(stable=True)

    BETA_TO_PROD = {
        'addrs_base_geoproduct_beta_1': 'addrs_base',
        'addrs_base_geoproduct_beta_2': 'addrs_base',
        'addrs_wizard_geoproduct_beta_1': 'addrs_wizard_yp',
        'addrs_wizard_geoproduct_beta_2': 'addrs_wizard_yp',
        'addrs_nmeta_geoproduct_beta_1': 'addrs_upper',
        'addrs_nmeta_geoproduct_beta_2': 'addrs_upper',
        'addrs_mmeta_geoproduct_beta_1': 'addrs_upper',
        'addrs_mmeta_geoproduct_beta_2': 'addrs_upper',
    }

    @staticmethod
    def _set_vault_secret(env, yav_client, secret_version, secret_id, secret_name=None):
        delegation_token, _ = yav_client.create_token(secret_id)
        vault_secret = {'secretVer': secret_version, 'secretId': secret_id, 'delegationToken': delegation_token}
        if secret_name is not None:
            vault_secret['secretName'] = secret_name
        env['valueFrom'] = {
            'secretEnv': {'secretName': ''},
            'literalEnv': {'value': ''},
            'vaultSecretEnv': {'field': '', 'vaultSecret': vault_secret},
            'type': 'VAULT_SECRET_ENV',
        }

    @classmethod
    def _patch_metasearch_attrs(cls, beta_runtime_attrs, sandbox_client, yav_client):
        import library.python.resource

        resources = beta_runtime_attrs['content']['resources']

        for resource in resources['sandbox_files']:
            if resource['local_path'] == 'configs' and int(resource['resource_id']) > PROD_CONFIGS_RESOURCE_ID:
                raise SomethingChangedException('New version of configs')

        resources['sandbox_files'] = filter_by_local_path(
            resources['sandbox_files'],
            ('push-client', 'configs', 'configs_old', 'configs_for_geohosts'),
        )

        for local_path, resource_type in zip(ADDRS_UPPER_NEW_RESOURCE_LOCAL_PATHS, ADDRS_UPPER_NEW_RESOURCE_TYPES):
            resource_id, task_id = find_last_released_resource(sandbox_client, resource_type)
            resources['sandbox_files'].append({
                'task_type': 'YA_MAKE_2',
                'task_id': str(task_id),
                'resource_id': str(resource_id),
                'extract_path': '',
                'is_dynamic': False,
                'local_path': local_path,
                'resource_type': resource_type,
            })

        resources['static_files'] = filter_by_local_path(
            resources['static_files'],
            ('push-client-nonepipe.conf', 'push-client-nonepipe-middle.conf'),
        )

        for template_file in resources['template_set_files']:
            if template_file['local_path'] != 'instancectl.conf':
                continue
            if get_md5(template_file['layout']) != INSTANCECTL_UPPER_MD5:
                raise SomethingChangedException('New version of instancectl.conf')
            template_file['layout'] = library.python.resource.find(
                'sandbox/projects/geoadv_ranking/SynchronizeAddrsGeoproductBetas/instancectl.conf',
            ).decode('utf-8')

        instance_spec = beta_runtime_attrs['content']['instance_spec']
        instance_spec['volume'] = [v for v in instance_spec['volume'] if 'push-client' not in v['name']]
        instance_spec['containers'] = [c for c in instance_spec['containers'] if 'push-client' not in c['name']]
        for container in instance_spec['containers']:
            container['env'] = [env for env in container['env']
                                if not env['name'].startswith('TVM_SECRET') or env['name'] == 'TVM_SECRET_2008261']
            for env in container['env']:
                if env['name'] == 'TVM_SECRET_2008261':
                    cls._set_vault_secret(
                        env,
                        yav_client,
                        'ver-01g023za4kk763phbdmd6zgngw',
                        'sec-01g023za46nxctescaz4amydeb',
                    )

    @classmethod
    def _patch_basesearch_attrs(cls, beta_runtime_attrs, yav_client):
        for container in beta_runtime_attrs['content']['instance_spec']['containers']:
            for env in container['env']:
                if env['name'] == 'SANDBOX_TOKEN':
                    cls._set_vault_secret(
                        env,
                        yav_client,
                        'ver-01g01zhy42nxjtc4z1cqsxp56n',
                        'sec-01g01zhy35x6cxh7rp6vd1egjz',
                        'geosearch-secrets.sandbox_token',
                    )

    @classmethod
    def _sync_services(cls, prod_service_id, beta_service_id, nanny_client, sandbox_client, yav_client):
        prod_runtime_attrs = nanny_client.get_runtime_attrs(prod_service_id)
        beta_runtime_attrs = nanny_client.get_runtime_attrs(beta_service_id)
        for section in ('instance_spec', 'resources'):
            beta_runtime_attrs['content'][section] = prod_runtime_attrs['content'][section]

        if prod_service_id == 'addrs_upper':
            cls._patch_metasearch_attrs(beta_runtime_attrs, sandbox_client, yav_client)

        elif prod_service_id == 'addrs_base':
            cls._patch_basesearch_attrs(beta_runtime_attrs, yav_client)

        beta_runtime_attrs['snapshot_id'] = beta_runtime_attrs['_id']
        beta_runtime_attrs['comment'] = 'Sync service with {}'.format(prod_service_id)
        for field in ('change_info', '_id', 'parent_id'):
            del beta_runtime_attrs[field]

        new_snapshot_id = nanny_client.put_runtime_attrs(beta_service_id, beta_runtime_attrs)['_id']

        info_beta_attrs = nanny_client.get_info_attrs(beta_service_id)

        recipes = info_beta_attrs['content']['recipes']
        deploy_content = {
            'state': 'ACTIVE',
            'snapshot_id': new_snapshot_id,
            'comment': '-',
            'recipe': recipes['content'][0]['id'],
        }
        if recipes['prepare_recipes']:
            deploy_content['prepare_recipe'] = recipes['prepare_recipes'][0]['id']

        nanny_client.create_event(beta_service_id, {'type': 'SET_SNAPSHOT_STATE', 'content': deploy_content})

    def on_execute(self):
        from infra.nanny.nanny_services_rest.nanny_services_rest.client import ServiceRepoClient
        from library.python.vault_client.instances import Production as VaultClient

        nanny_client = ServiceRepoClient(NANNY_URL, get_token_from_yav(NANNY_ROBOT_EXTDATA_SECRET_ID))
        sandbox_client = Client(auth=OAuth(get_token_from_yav(SANDBOX_ROBOT_EXTDATA_SECRET_ID)))
        yav_client = VaultClient(
            authorization='OAuth {}'.format(get_token_from_yav(YAV_ROBOT_EXTDATA_SECRET_ID)),
            decode_files=True,
        )

        for beta_service_id, prod_service_id in self.BETA_TO_PROD.iteritems():
            self._sync_services(prod_service_id, beta_service_id, nanny_client, sandbox_client, yav_client)
