import json
import os
import shutil
import six
import subprocess

from abc import abstractmethod
from collections import namedtuple

from sandbox.common.types.client import Tag
from sandbox.projects.common.binary_task import LastBinaryTaskRelease, binary_release_parameters
from sandbox.projects.common.build.parameters import ArcadiaUrl
from sandbox.projects.common.build.YaMake2 import YaMake2
from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask2
from sandbox.projects.common.utils import sync_resource
from sandbox.projects.geoadv_ranking.lib.subtasks import wait_subtasks, validate_subtask_statuses
from sandbox.projects.resource_types import AbstractResource
from sandbox.sdk2 import Task, Resource, ResourceData
from sandbox.sdk2.parameters import JSON


GeoadvRankingBuildTarget = namedtuple('GeoadvRankingBuildTarget', ['path', 'resource_type', 'is_file'])


def get_resource_type_by_name(resource_name):
    for type_ in AbstractResource:
        if type_.name == resource_name:
            return type_
    raise ValueError('Resource type {!r} does not exist'.format(resource_name))


class BuildTargetsParameter(JSON):
    name = 'geoadv_ranking_build_target'
    description = 'Description of geoadv target to build'

    KEY_TO_FACTORY = {'path': unicode, 'is_file': bool, 'resource_type': get_resource_type_by_name}

    @classmethod
    def cast(cls, value, return_orig=True):
        if isinstance(value, six.string_types):
            value = json.loads(value)
        else:
            json.dumps(value)
        if not isinstance(value, list):
            raise ValueError('Value {!r} should be a list'.format(value))
        result = []
        for i, item in enumerate(value):
            for key in cls.KEY_TO_FACTORY:
                if key not in item:
                    raise ValueError('Field {} is missing in item {}'.format(key, i))
            for key in item:
                if key not in cls.KEY_TO_FACTORY:
                    raise ValueError('Field {} is unknown in item {}'.format(key, i))
            result.append(
                GeoadvRankingBuildTarget(**{k: cls.KEY_TO_FACTORY[k](item[k]) for k in cls.KEY_TO_FACTORY}),
            )
        return value if return_orig else result


class AbstractGeoadvRankingBuildTask(LastBinaryTaskRelease, ReleaseToNannyTask2, Task):
    class Parameters(Task.Parameters):
        arcadia_url = ArcadiaUrl()

        ext_params = binary_release_parameters(stable=True)

    @abstractmethod
    def get_build_targets(self):
        return []

    def on_execute(self):
        super(AbstractGeoadvRankingBuildTask, self).on_execute()

        build_targets = {target.path: target for target in self.get_build_targets()}

        with self.memoize_stage.launch_ya_make_tasks:
            self.Context.ya_make_tasks = {}

            for target in build_targets.values():
                directory, _ = target.path.rsplit('/', 1)
                directory_path = directory if target.is_file else target.path
                task = YaMake2(
                    self,
                    description='Build {}.'.format(directory_path),
                    checkout_arcadia_from_url=self.Parameters.arcadia_url,
                    targets=directory_path,
                    arts=target.path,
                    use_aapi_fuse=True,
                    aapi_fallback=True,
                    result_rt=target.resource_type.name,
                    result_single_file=True,
                )
                task.Requirements.client_tags = Tag.Group.LINUX
                task.enqueue()
                self.Context.ya_make_tasks[target.path] = task.id

            wait_subtasks(self.Context.ya_make_tasks.values())

        with self.memoize_stage.check_ya_make_statuses:
            validate_subtask_statuses(self.Context.ya_make_tasks.values())

        with self.memoize_stage.sync_builded_targets:
            for path, task_id in self.Context.ya_make_tasks.iteritems():
                target = build_targets[path]
                resource = next(iter(Resource.find(task_id=task_id, type=target.resource_type).limit(1)), None)

                synced_resource = sync_resource(resource.id)
                local_path = os.path.basename(synced_resource)
                if target.is_file:
                    if os.path.exists(local_path):
                        subprocess.check_call(['mkdir', 'pack'])
                        local_path = 'pack/{}'.format(local_path)
                    shutil.copyfile(synced_resource, local_path)
                    subprocess.check_call(['chmod', '+x', str(local_path)])
                else:
                    shutil.copytree(synced_resource, local_path)
                registered_resource = target.resource_type(self, resource.description, local_path)
                ResourceData(registered_resource).ready()
