from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.common.types.resource import State as ResourceState
from sandbox.common.types.task import Semaphores
from sandbox.projects.common.apihelpers import get_last_resource_with_attribute
from sandbox.projects.common.binary_task import binary_release_parameters, LastBinaryTaskRelease
from sandbox.projects.common.utils import sync_resource

from .tokens import YT_ROBOT_EXTDATA_SECRET_ID, SECRET_KEY


class CommonSimilarAdvertParameters(sdk2.Task.Parameters):
    ext_params = binary_release_parameters(stable=True)

    acquire_semaphore = sdk2.parameters.Bool(
        'Whether to acquire semaphore',
        required=True,
        default=False,
    )

    with acquire_semaphore.value[True]:
        semaphore_name = sdk2.parameters.String(
            'Semaphore name',
            required=False,
        )

    yt_token_yav = sdk2.parameters.YavSecret('YT_TOKEN yav secret', required=True, default=YT_ROBOT_EXTDATA_SECRET_ID)


class AbstractSimilarAdvertTask(LastBinaryTaskRelease, sdk2.Task):
    DEFAULT_SEMAPHORE_NAME = None

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(CommonSimilarAdvertParameters):
        pass

    def _get_yt_token(self):
        return self.Parameters.yt_token_yav.data()[SECRET_KEY]

    @staticmethod
    def _sync_stable_resource(resource_type):
        return sync_resource(
            get_last_resource_with_attribute(resource_type, 'released', 'stable', ResourceState.READY).id,
        )

    def on_enqueue(self):
        if self.Parameters.acquire_semaphore:
            semaphore_name = self.Parameters.semaphore_name or self.DEFAULT_SEMAPHORE_NAME

            if not semaphore_name:
                raise TaskError('Semaphore\'s name is not defined.')

            self.Requirements.semaphores = Semaphores(acquires=[Semaphores.Acquire(name=semaphore_name, weight=1)])

        return super(AbstractSimilarAdvertTask, self).on_enqueue()
